# This manages an organizational hierarchy data structure.  It is only
# responsible for the internal data structure itself - it does no 
# communications.

# Things known to depend on this class are:
#	1) The organizational hierarchy server ohs
#	2) Hierarchy_interface.pm
#	3) The user interface which displays the hierarchy.

use Clog;
use Host;
use Department;

package Hierarchy;

#============================================================================#
#
# Function: new
#
# This function creates a new hierarchy object based on input either in the 
# form of a file, or a reference to a scalar with the data in.  It should 
# parse the input iff it is is correctly formatted according to the 
# grammar in the OHS design chapter.  However, it makes no guarantee 
# to diagnose exactly what is wrong with mis-formatted input.  This input is
# the second argument to the function.
#
# The first argument is a log object to report problems to.
#
# If a third argument is supplied, it should be a serial number to be 
# incorporated into all dept nodes.
#
# usage
#
# $hierarchy = new Hierarchy($log, $filename  [,$serial]);
#
# $hierarchy = new Hierarchy($log, \$input [,$serial]);
#
#----------------------------------------------------------------------------#   

sub new
{  
  my(@raw_nodes);
  my($type,$log,$input,$serial) = @_;
  my $self = {'_log' => $log};
  bless $self;

  return $self unless defined $input;
   
  my $old = $/; # save the input record separator
  if(ref $input eq 'SCALAR')
   {
    # The input is already supplied.
    @raw_nodes = split(/;/,$$input);
   }
  else
   {
    # Getting input from a file.
    unless(open(HIERARCHY, $input))
     {
      $log->warn("Could not open hierarchy file $input");
      return undef;
     }
    $/ = ';'; # Break up by semicolons;
    @raw_nodes = <HIERARCHY>;  
    close(HIERARCHY);
   }
  foreach (@raw_nodes)
   {
    chomp; 		# get rid of trailing semicolon
    s/\s+//g; 		# get rid of white space
    next if /^$/; 	# get rid of null fields
    my $new_node; 
    my $input = $_;
    if(/^host\,/)
     {
      # This entry is supposed to be a host
      $new_node = new Host($_);    
      unless(ref($new_node) eq 'Host')
       {$log->warn($new_node); return undef;}
     }
    elsif(/^dept\,/)
     {
      # This entry is supposed to be a host
      $new_node = new Department($_);    
      unless(ref($new_node) eq 'Department')
       {$log->warn($new_node); return undef;}
      $new_node->{'view_serial'} = $serial if defined $serial;
     }
    else
     {
      $log->warn("Bad item type: $input");
      last;
     }
    # Note the next line has unless(!X) rather than if(X) because
    # under Perl 5.001m on a 68040 NeXTstation, the second form produces
    # a compile error while the first form doesn't.  This is deeply
    # disturbing and mysterious.
    unless(!exists $self->{$new_node->{'name'}})
     {$log->warn("Node defined twice: $input"); return undef;}
    $self->{$new_node->{'name'}} = $new_node;
   }
  $/ = $old; # reset the separator

  my $node;
  foreach $node (values %$self)
   {
    if(ref $node eq 'Department')
     {
      while($node->{'parent'} ne 'null')
       {
        $node = $self->{$node->{'parent'}}
       }
      $self->{'root_name'} = $node->{'name'};
      last;
     }
   }
  return $self;
}

#============================================================================#
#
# Function:  save
#
# This saves the hierarchy to file.
#
# Inputs:
#  Actual Arguments:
#    File name
#
# Return Value(s):
#  True on success
#  False on failure
#
# Example of usage (from OHS):
#  $hierarchy->save($filename);
#
#----------------------------------------------------------------------------#   

sub save
{
  my($self, $file, $format) = @_;          # RC added format parm 7/1/97
  my($node);
  my($log) = $self->{'_log'};
  
  unless(open(HIERARCHY, ">$file"))
   {
    $log->warn("Could not open hierarchy file $file\n");
    return undef;
   }
  my $old_handle = select(HIERARCHY);

  if ($format eq 'prettyprint') {
    print "# DEPT FORMAT:\n"
        . "# 'dept', \$dept_name, \$parent_dept {\n"
        . "# \t\@hosts\n"
        . "# \t},\n"
        . "# \t{\@child_depts},\n"
        . "# \$man_host:\$man_port, \$agg_host:\$agg_port\n\n";
    }
  elsif ($format eq 'parse_config_file') {
    print "\$self->{'ohs_host'} = "
        . "'$Hierarchy_interface::grids_db->{'ohs_host'}';\n\n";
    print "\@dept_geometry = (\n";
    }

  foreach $node (values %$self) {
    next if (ref($node) eq 'Clog'  ||  ref($node) eq 'Host');
    if (ref($node) eq 'Department') {
      $node->output($format);
      }
    else {
      $log->warn("Bad item in hierarchy nodes: $node") unless $node eq 'ROOT';
      }
    }

  if ($format eq 'parse_config_file') {
    print "\t);\n\n########## last CONFIG; ??? ####################\n";
    #  Do not print any detailed host:port info to config file, so return:
    select($old_handle);
    close(HIERARCHY);
    return 1;
    }

  # print a spacer to selected filehandle, after depts, before hosts:
  print "# =================================================================\n";
  print "# HOST FORMAT:\n"
      . "# 'host', \$host_name, \$dept_name, \$mc_port\n\n";

  foreach $node (values %$self) {
    next if (ref($node) eq 'Clog'  ||  ref($node) eq 'Department');
    if (ref($node) eq 'Host') {
      $node->output();
      }
    }
  select($old_handle);
  close(HIERARCHY);
  return 1;
}

#============================================================================#
#
# Function:  sub_hierarchy_output
#
# This is outputs a portion of the hierarchy to a string.  It is generally
# used for shipping the hierarchy across a network.
#
# Inputs:
#  Actual Arguments:
#    Node which is to be output
#
# Return Value(s):
#  The portion of the hierarchy desired
#
# Example of usage (from OHS):
#  $output_string = $hierarchy->sub_hierarchy_output($node_name);
#
#----------------------------------------------------------------------------#   

sub sub_hierarchy_output
{
  my($self,$node) = @_;
  
  return $self->{$node}->string_print().";\n" if(ref $self->{$node} eq 'Host');
  return $self->{$node}->string_print().";\n".
  join('',map($self->sub_hierarchy_output($_),
  			@{$self->{$node}->{'host_children'}})).
  join('',map($self->sub_hierarchy_output($_)
  			,@{$self->{$node}->{'dept_children'}}));
}

#============================================================================#
#
# Function:  use_ACL
#
# This sets a global variable which determines if acess control is used.
#
# Inputs:
#  1/0 to set the variable
#
# Global variable changes:
#  $global::use_ACL
#
# Return Value(s):
#  none
#
# Example of usage:
#  Hierarchy::use_ACL(0);
#
#----------------------------------------------------------------------------#

sub use_ACL
{  
my ($value) = @_;
 
        $global::use_ACL = $value;
 
}
 

#============================================================================#
#============================================================================#
#
#				SANITY CHECKING FUNCTIONS
#
#	These are all involved with checking that some particular transaction
#	is ok to proceed with.  Their general remit is to check:
#		1. that this request is based on an up-to-date version number.
#		2. that this request is possible on the current hierarchy.
#		3. that this request is authorized.
#
#============================================================================#
#============================================================================#

#============================================================================#
#
# Function:  set_initial_user
#
# This function simply sets the initial user for an empty hierarchy.  Only 
# this user will be able to add a new root into the OHS.  This user is 
# provided to the OHS on the command line at start-up.
#
# Inputs:
#  Actual Arguments:
#    Name and password
#
#============================================================================#
 
sub set_initial_user
 {
  my $self = shift;
  my($user,$pass) = @_;

  $self->{'init_user'} = $user;
  $self->{'init_pass'} = $pass;

}# set_initial_user

#============================================================================#
# 
# Function:  authorization_ok
# 
# This function checks to see if the user is authorized for this transaction,
# and notes the outcome in the log.  This is only used internally by sanity
# functions.  It should not be accessed directly from outside of package.
# 
# Inputs:
#  Actual Arguments:
#    User, password, and the department requesting autorization for.
#
# Return Value(s):                           
#  True(1)/False(0)
# 
# Global variable modifications:
#  None
#
#============================================================================#
                                                      
sub authorization_ok
 {
  my($self,$user,$pass,$dept) = @_;
  my $log = $self->{'_log'};

  $log->warn("authorization check for $user");

  return 1 unless ($global::use_ACL == 1); # only check ACL if it is used

  unless (exists $self->{'ACL'}{$user})
   {
    $log->warn("User $user does not exist.");
     return 0;
   }

  unless ($self->{'ACL'}{$user}{'pass'} eq $pass)
   {
    $log->warn("Incorrect password for $user.");
    return 0;
   }

  # Department must be at or below the user's access level.
  unless($self->is_a_child($self->{'ACL'}{$user}{'dept'},$dept))
   {
    $log->warn("$user does not have authorization for $dept.");
    return 0;
   }
  
  return 1; 
} # end transaction_authorization_ok
 
#============================================================================#
#
# Function:  is_a_child
#
# This function checks to see if the department which is in question is
# available to the user requesting.  This is only used internally by sanity 
# functions.  It should not be accessed directly from outside of package. 
#
# Inputs:
#  Actual Arguments:
#   Department authorized for, department searching for
#
# Return Value(s):
#  True(1)/False(0)
#
# Global variable modifications:
#  None
#
#============================================================================#

sub is_a_child
{
  my($self,$curr_dept,$dept) = @_;
  my $log = $self->{'_log'};
    
  if($curr_dept eq $dept)
   {
    return 1;
   }

  # Recursively search children for the department.
  my @children = @{$self->{$curr_dept}{'dept_children'}};
  my $found = 0;
  my $child;
  while (@children)
   {
    $child = pop(@children);
    $found += $self->is_a_child($child,$dept);
   }
  return $found;
}# end is_a_child

sub view_authorization
{
  my($self,$user,$pass,$dept) = @_;
 
  # User must be authorized for the department.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    return 0;    
   }
  return 1;
}

#============================================================================#
#
# Function:  new_root_is_sane
# 
# This basic sanity checking on the new_root transaction.  It checks
# to see if the hierarcy contains information about the initial user.
# If it does, it checks to make sure that the current user is the initial
# user.  This is used to make sure the OHS knows it is talking to the right
# person.  Otherwise, it simply checks to see if the hierarchy is empty.
# This is used for the interface, which does not have any information about
# the initial user.  
#
# If the transaction is sane, then the user will be put into the ACL, and
# the initial user information removed (if it exists).
#
# Inputs:
#  Actual Arguments:
#   The function ignores its arguments.  However, for consistency
#   with other sanity checking functions in here, the htr message will
#   usually be passed in.
#
#  Global:
#    None
# 
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->new_root_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#   
sub new_root_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$dept) = @{$_[0]};

  return 1 unless ($global::use_ACL == 1); # only check ACL if it is used
 
  if (!(defined $self->{'init_user'}))
   {
    # Can only do new_root if hierarchy is empty.  (Interface)
    unless($self->is_empty()) 
     {
      $log->warn("hierarchy was not empty in htr_new_root_is_sane");
      return 0;
     }
   } 
  else 
   {
    # The user and password must be the initial ones.  (OHS)
    unless (($self->{'init_user'} eq $user) &&
            ($self->{'init_pass'} eq $pass))
     {  
      $log->warn("Invalid username or password to add new root.");
      return 0;
     }  
   } # end else

  # In either case, put the user into the ACL.
  $self->{'ACL'}{$user}{'pass'} = $pass; #Add initial user to ACL
  $self->{'ACL'}{$user}{'dept'} = $dept;
  delete $self->{'init_user'};
  delete $self->{'init_pass'};
  return 1
} # end sub new_root_is_sane

#============================================================================#
#
# Function:  change_user_is_sane
#
# This basic sanity checking on the add_user transaction.  It checks
# the authorization of the user adding.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1) 
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->add_user_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#
 
sub change_user_is_sane                                                           
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$user2,$pass2,$dept) = @{$_[0]};

  # User must be authorized for the department.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }

  if (exists $self->{'ACL'}{$user2})
   {
    $log->warn("User $user2 already exists.");
    return 0;
   }

  return 1;
}

#============================================================================#
#
# Function:  add_host_is_sane
# 
# This basic sanity checking on the add_host transaction.  It checks
# to see if the 'Department' already exists.  It then makes sure the new
# 'Host' is *not* known, and finally checks to make sure there is not a
# problem with the serial numbers.
# 
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
# 
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->add_host_is_sane';
#  unless(eval "$function_call".'($messageref)')
# 
#----------------------------------------------------------------------------#   
sub add_host_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$dept,$host,$port) = @{$_[0]};
  
  # The putative department must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Department request for $dept");
    return 0;
   }
   
  # The host must *not* already be known to us
  if(exists $self->{$host})
   {
    $log->warn("Attempt to duplicate host $host");
    return 0;
   }

  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # User must be authorized for the department. 
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }

  return 1;
} # end add_host_is_sane

#============================================================================#
#
# Function:  add_dept_is_sane
# 
# This basic sanity checking on the add_dept transaction.  It checks
# to see if the 'Parent' already exists.  It then makes sure the new 
# 'Department' is *not* known, and finally checks to make sure there is not a
# problem with the serial numbers.
# 
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
# 
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->add_dept_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#   
 
sub add_dept_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$parent,$dept,$manhost,
  						$aghost) = @{$_[0]};

  # The putative parent department must already be known to us.
  unless(ref $self->{$parent} eq 'Department')
   {
    $log->warn("Invalid Add Department request under $parent");
    return 0;
   }
   
  # The new department must *not* already be known to us
  if(exists $self->{$dept})
   {
    $log->warn("Attempt to duplicate department $dept");
    return 0;
   }

  # The putative parent department must already be known to us.
  unless(ref $self->{$manhost} eq 'Host')
   {
    $log->warn("Invalid Add Department request under $parent, ".
	"$manhost is not a host.");
    return 0;
   }


  # The supplied serial number must be as large as the serial number at $parent
  my $my_serial = $self->{$parent}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # User must be authorized for the department. 
  unless($self->authorization_ok($user,$pass,$parent))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
} # end add_dept_is_sane

#============================================================================#
#
# Function:  change_variable_is_sane
#
# This basic sanity checking on the change_variable transaction.  It checks
# to see if the 'Department' exists, and checks to make sure there is not a
# problem with the serial numbers.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if either test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->change_variable_is_sane'; 
#  unless(eval "$function_call".'($messageref)')
# 
#----------------------------------------------------------------------------#   

sub change_variable_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$dept) = @{$_[0]};
 
  # The putative department must already be known to us. 
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Change Variable request under department: $dept");
    return 0;
   }

  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # User must be authorized for the department.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end change_variable_is_sane

#============================================================================#
#
# Function:  move_dept_is_sane
#
# This basic sanity checking on the move_dept transaction.  It checks
# to see if the 'Parent' department to move to exists, as well as if
# the 'Department' to be moved exists.  It then checks to make sure 
# there is not a problem with the serial numbers for either the 'Parent'
# or the 'Department'.  Finally it does not allow moving a department to
# its self or to a descendant.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_dept_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#

sub move_dept_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$parent,$dept) = @{$_[0]};
 
  # The putative parent department must already be known to us.
  unless(ref $self->{$parent} eq 'Department')
   {
    $log->warn("Invalid Move Department request under $parent");
    return 0;
   }
 
  # The department to be moved must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Move Department request for $dept");
    return 0;
   }
 
  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }
 
  # The supplied serial number must be as large as the serial number at $parent
  $my_serial = $self->{$parent}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # The moved department and the new parent department must be different.
  unless ($dept ne $parent)
   {
    $log->warn("Tried to move $dept to itself");
    return 0;
   }

  # Determine if $parent is a descendant of $dept.  If so, no move is allowed
  my $found = 0;
  my $my_dept = $self->{$parent}->{'parent'}; 
  
  while ($my_dept ne 'null')
   {
    if ($my_dept eq $dept)
     {
       $log->warn("Tried to move $dept to $parent ($parent is a descendant of $dept)");
       return 0;
     }
    $my_dept = $self->{$my_dept}->{'parent'};
   }# end while

  # User must be authorized for both the department and the parent.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
   unless($self->authorization_ok($user,$pass,$parent))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end move_dept_is_sane


#============================================================================#
#
# Function:  remove_dept_is_sane
#
# This does basic sanity checking on the remove_dept transaction.  It checks
# to see if the 'Dept' to remove exists and has a proper serial number.
# Then it checks the entire hierarchy to see if any departments (except
# those in the subtree being removed) have SM's or engines running on the
# hosts in the subtree.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->remove_dept_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#

sub remove_dept_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$dept) = @{$_[0]};
  my $old_parent = $self->{$dept}->{'parent'};

  # The dep to be removed must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    my $warning = ("Invalid Remove Dept request for $dept");
    $log->warn("$warning");
    return (0);
   }

  if($dept eq $self->{'root_name'})
   {
    my $warning = ("Invalid Remove Dept request for $dept, cannot remove root.");
    $log->warn("$warning");
    return (0);
   }

  # The supplied serial number must be as large as the serial number at
  # $old_parent.
  $my_serial = $self->{$old_parent}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    my $warning = ("Supplied serial $serial smaller than $my_serial");
    $log->warn("$warning");
    return (0);
   }

  my (%host_assc,$host,@dept_array,$temp_dept);
                              
  # First we must get all of the children departments and hosts.
  push (@dept_array, $dept);
  @temp_dept_array = @dept_array;
 
  while ($temp_dept = pop(@dept_array))
   {
     push (@dept_array, @{$self->{$temp_dept}{'dept_children'}});
     foreach $host (@{$self->{$temp_dept}{'host_children'}})
      { $host_assc{$host} = 1; }
   } # end while
    
  # Check the hierarchy to see if any departments have SM's or engines
  # on any of the hosts which will be removed.
  my @depts_on_hosts; 
  push (@dept_array, $self->{'root_name'});
  while ($temp_dept = pop(@dept_array))
   {
     unless ($temp_dept eq $dept)
      {
       push (@dept_array, @{$self->{$temp_dept}{'dept_children'}});
       if (($host_assc{$self->{$temp_dept}{'manager_host'}}) ||
           ($host_assc{$self->{$temp_dept}{'aggregator_host'}}))
        { push (@depts_on_hosts, $temp_dept); }
      }# end unless
   } # end while

  if ($#depts_on_hosts+1)
   {
    # There is one or more SM or Engine on a host.  Therefore the dept 
    # cannot be removed. 
    $log->warn("Cannot remove dept $dept, SM or engine is running on a host\n".
               "for the following department(s):  ".join(' ',@depts_on_hosts));
    return (0);
   }#end if

  # User must be authorized for the old parent.
  unless($self->authorization_ok($user,$pass,$old_parent))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }

  return 1;
}# end remove_dept_is_sane
  
#============================================================================#
# 
# Function:  move_host_is_sane
# 
# This basic sanity checking on the move_host transaction.  It checks
# to see if the 'Host' to remove exists, as well as if the 'Department' 
# to be moved to exists.  It then checks to make sure there is not a
# problem with the serial numbers for either the parent of the host
# or the 'Department' to move to.  Does not allow moving in place.
# 
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
# 
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_host_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#   
sub move_host_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$dept,$host) = @{$_[0]};
  my $old_parent = $self->{$host}->{'parent'}; 
 
  # The department to be moved must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Move Host request for $dept");
    return 0;
   }

  # The host to be moved must already be known to us.
  unless(ref $self->{$host} eq 'Host')
   {
    $log->warn("Invalid Move Host request for $host");
    return 0;
   }
 
  # Does not allow moving host to its current position.
  if ($old_parent eq $dept)
   {
    $self->{'_log'}->warn("Tried to move $host over itself");
    return 0;
   }

  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }
 
  # The supplied serial number must be as large as the serial number at 
  # $old_parent
  $my_serial = $self->{$old_parent}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # User must be authorized for both the old and new departments.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
  unless($self->authorization_ok($user,$pass,$old_parent))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end move_host_is_sane

#============================================================================#
#
# Function:  remove_host_is_sane
#
# This basic sanity checking on the remove_host transaction.  It checks
# to see if the 'Host' to remove exists.  It then checks to see if there
# are any SM's or aggregators running on the host. 
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->remove_host_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#

sub remove_host_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$host) = @{$_[0]};
  my $old_parent = $self->{$host}->{'parent'};

  # The host to be removed must already be known to us.
  unless(ref $self->{$host} eq 'Host')
   {
    my $warning = ("Invalid Remove Host request for $host");
    $log->warn("$warning");
    return (0);
   }

  # The supplied serial number must be as large as the serial number at
  # $old_parent.
  $my_serial = $self->{$old_parent}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    my $warning = ("Supplied serial $serial smaller than $my_serial");
    $log->warn("$warning");
    return (0);
   }

  # Check all the departments in the Hierarchy to determine if they have
  # engines or SM's running on this host.

  my $dept = $self->{'root_name'};
  my (@depts_on_host,@dept_array);

  push (@dept_array, $dept);
  while ($dept = pop(@dept_array))
   {
     push (@dept_array, @{$self->{$dept}{'dept_children'}});
     if (($self->{$dept}{'manager_host'} eq $host) ||
         ($self->{$dept}{'aggregator_host'} eq $host))
      { push (@depts_on_host, $dept); }
   } # end while

  if ($#depts_on_host+1)
   {
    # There is one or more SM or Engine on this host.  Therefore it
    # cannot be removed. 
    $log->warn("Cannot remove host $host, SM or engine running on host for\n".
               "the following department(s):  ".join(' ',@depts_on_host));
    return (0);
   }#end if

  # User must be authorized for the old parent.
  unless($self->authorization_ok($user,$pass,$old_parent))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end remove_host_is_sane

#============================================================================#
#
# Function:  move_manager_is_sane
#
# This basic sanity checking on the move_manager transaction.  It checks
# to see if the department and host in question are known to the hierarchy.
# It also checks the serial numbers and makes sure we are not switching to
# the host we are already on.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_manager_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#

sub move_manager_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$dept,$host) = @{$_[0]};

  # The department must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Move Manager request, $dept does not exist.");
    return 0;
   }

  # The putative host must already be known to us.
  unless(ref $self->{$host} eq 'Host')
   {
    $log->warn("Invalid Move Manager request, $host does not exist");
    return 0;
   }

  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # The new host must be different from the old one.
  unless ($host ne $self->{$dept}{'manager_host'})
   {
    $log->warn("Tried to move SM for $dept to itself");
    return 0;
   }

  # User must be authorized for the department.
  unless($self->authorization_ok($user,$pass,$dept))
   {
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end move_manager_is_sane

#============================================================================#
#
# Function:  move_aggregator_is_sane
#
# This basic sanity checking on the move_aggregator transaction.  It checks
# to see if the department and host in question are known to the hierarchy.
# It also checks the serial numbers and makes sure we are not switching to
# the host we are already on.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#  Global:
#    None
#
# Return Value(s):
#  False (0) if any test fails, else
#  True  (1)
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_aggregator_is_sane';
#  unless(eval "$function_call".'($messageref)')
#
#----------------------------------------------------------------------------#

sub move_aggregator_is_sane
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($type,$trans_id,$user,$pass,$serial,$dept,$host) = @{$_[0]};

  # The department must already be known to us.
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Invalid Move Manager request, $dept does not exist.");
    return 0;
   }

  # The putative host must already be known to us.
  unless(ref $self->{$host} eq 'Host')
   {
    $log->warn("Invalid Move Aggregator request, $host does not exist");
    return 0;
   }

  # The supplied serial number must be as large as the serial number at $dept
  my $my_serial = $self->{$dept}{'view_serial'};
  if(defined $my_serial && $my_serial > $serial)
   {
    $log->warn("Supplied serial $serial smaller than $my_serial");
    return 0;
   }

  # The new host must be different from the old one.
  unless ($host ne $self->{$dept}{'aggregator_host'})
   {
    $log->warn("Tried to move aggregator for $dept to itself");
    return 0;
   }

  # User must be authorized for the department.
  unless($self->authorization_ok($user,$pass,$dept))
   { 
    $log->warn("Authorization failure for user $user");
    return 0;
   }
 
  return 1;
}# end move_aggregator_is_sane
 

#============================================================================#
#============================================================================#

#				TRANSACTION FUNCTIONS

#	These are the functions which actually update our data structures
#	after an interface has assured us that it has completed its 
#	operations with the remote parties.

#============================================================================#
#============================================================================#

#============================================================================#
#
# Function:  new_root
#
# Change our data structures to reflect a successfully completed new_root
# transaction.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->new_root';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#

sub new_root
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($message_ref,$htc_ref,$serial) = @_;
  my($type,$trans_id,$user,$pass,$dept,$man_host,$man_port, 
  		$agg_host,$agg_port) = @$message_ref;

  $self->{$dept} = new Department($dept,'null', $man_host, $man_port, 
  		$agg_host,$agg_port);
  $self->{'root_name'} = $dept;

  unless(defined $self->{$dept})
   {
    $log->warn("Bad arguments to new Department function");
    return undef;
   }
   
  $self->{$dept}{'view_serial'} = $serial if defined $serial;
  return 1;
}

 
#============================================================================#
#
# Function:  change_user
#
# This function addes a new user or removes one from the ACL.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->change_user';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub change_user
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($message_ref,$htc_ref,$in_serial) = @_;
  my($hdr,$trans_id,$user,$pass,$serial,$user2,$pass2,$dept) = @$message_ref;
  #my($hdr,$trans_id,$user,$pass,$serial,$user2,$pass2,$dept) = @{$_[0]};
 
  return 1 unless ($global::use_ACL == 1); # only check ACL if it is used

  $self->{'ACL'}{$user2}{'pass'} = $pass2; #Add user to ACL
  $self->{'ACL'}{$user2}{'dept'} = $dept;

  if ($pass2 eq ''){ #This is removing the user
    delete $self->{'ACL'}{$user2};
   } 

  $self->propagate_serial($dept,$in_serial) if defined $in_serial;

  return 1;
}

#============================================================================#
#
# Function:  add_host
#
# This function creates a new host obect and stores it in the hash.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->add_host';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub add_host
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($message_ref,$htc_ref,$in_serial) = @_;
  my($hdr,$trans_id,$user,$pass,$serial,$dept,$host,$port) = @$message_ref;
  
  # Create the new host object and store it in our hash
  $self->{$host} = new Host($host,$dept,$port);
  unless(ref $self->{$host} eq 'Host')
   {
    $log->warn("Bad arguments to new Host function - $self->{$host}");
    return undef;
   }
  
  # Tell the parent about this new child.
  push @{$self->{$dept}->{'host_children'}}, $host;

  $self->propagate_serial($dept,$in_serial) if defined $in_serial;
  
  return 1;
}

#============================================================================#
#
# Function:  add_dept
#
# This function creates a new department object and stores it in the
# hash.  It then tells the parent about its new child.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->add_dept';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub add_dept
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$parent,$dept,$manhost,
  						$aghost) = @{$_[0]};
  my($type,$trans_id,$manport,$agport) = @{$_[1]};
  my $in_serial = $_[2];
  
  # Create the new dept object and store it in our hash
  $self->{$dept} = new Department($dept,$parent,
  					$manhost,$manport,$aghost,$agport);
  unless(ref $self->{$dept} eq 'Department')
   {
    $log->warn("Bad arguments to new Department function - $self->{$dept}");
    return undef;
   }
  
  # Tell the parent about this new child.
  push @{$self->{$parent}->{'dept_children'}}, $dept;

  $self->propagate_serial($dept,$in_serial) if defined $in_serial;

  return 1;
}

#============================================================================#
#
# Function:  change_variable
# 
# This function does not need to change anything in the hierarchy.  Therefore,
# it updates the hierarchy serial number and returns.
# 
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None 
#
# Return Value(s):
#  True (1) for all paths
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->change_variable'; 
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
# 
#----------------------------------------------------------------------------#   

sub change_variable
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($message_ref,$htc_ref,$in_serial) = @_;
  my($type,$trans_id,$user,$pass,$serial,$dept) = @$message_ref;
 
  $self->propagate_serial($dept,$in_serial) if defined $in_serial;

  return 1;
}

#============================================================================#
# 
# Function:  move_dept
# 
# This searches the list of child departments under the parent
# whose child is to be moved and removes that child from the list.
# It then tels the new parent about the department, updates the
# moved department's parent information, then propagate the new
# serial number to all parties involved.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
# 
# Return Value(s):
#  True (1) for all paths
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_dept';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
# 
#----------------------------------------------------------------------------#   

sub move_dept
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$parent,$dept) = @{$_[0]};
  my($type,$trans_id) = @{$_[1]};
  my $in_serial = $_[2];
 
  # Get the parent of the department we are going to move.
  my $old_parent = $self->{$dept}->{'parent'};

  # Remove $dept as a child of $old_parent.
  @{$self->{$old_parent}->{'dept_children'}}
    = grep($_ ne $dept, @{$self->{$old_parent}->{'dept_children'}});
 
  # Tell the new parent about this new child.
  push @{$self->{$parent}->{'dept_children'}}, $dept;

  # Change parent information in moved department
  $self->{$dept}->{'parent'} = $parent;

  # Propagate new serial number to both the moved department and the old
  # parent department.
  $self->propagate_serial($dept,$in_serial) if defined $in_serial;
  $self->propagate_serial($old_parent,$in_serial) if defined $in_serial;
 
  return 1;
} # end move_dept 


#============================================================================#
#
# Function:  remove_dept
#
# This will remove a dept from the hierarchy.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) for all paths
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->remove_host';
#  unless(eval "$function_call".
#        '($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub remove_dept
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$dept) = @{$_[0]};
  my($type,$trans_id) = @{$_[1]};
  my $in_serial = $_[2];
  my (@dept_array,@temp_dept_array,$temp_dept);
  # Get the parent of the dept we are going to move.
  my $old_parent = $self->{$dept}->{'parent'};

  # Remove $dept as a dept on $old_parent.
  @{$self->{$old_parent}->{'dept_children'}}
    = grep($_ ne $dept, @{$self->{$old_parent}->{'dept_children'}});

  # Remove dept and its children from array.
  push (@dept_array, $dept);
  @temp_dept_array = @dept_array;
 
  while ($temp_dept = pop(@temp_dept_array))
   {
     push (@dept_array, @{$self->{$temp_dept}{'dept_children'}});
     push (@dept_array, @{$self->{$temp_dept}{'host_children'}});
     push (@temp_dept_array, @{$self->{$temp_dept}{'dept_children'}});
   } # end while

  while ($temp_dept = pop(@dept_array))
   { delete $self->{$temp_dept}; }

  # Give new serial number to the old parent department.
  if (defined $in_serial)
    {$self->{$old_parent}{'view_serial'} = $in_serial;} 

  return 1;
} # end remove_dept

#============================================================================#
# 
# Function:  move_host
#
# This will move a host from one department to another.  First it removes
# the host from its original parent.  Then it informs the new parent and
# updates the parent information on the host.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
# 
# Return Value(s):
#  True (1) for all paths
# 
# Global variable modifications:
#  None
# 
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_host';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#   
 
sub move_host
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$dept,$host) = @{$_[0]};
  my($type,$trans_id) = @{$_[1]};
  my $in_serial = $_[2];

  # Get the parent of the host we are going to move.
  my $old_parent = $self->{$host}->{'parent'};
 
  # Remove $host as a host on $old_parent.
  @{$self->{$old_parent}->{'host_children'}}
    = grep($_ ne $host, @{$self->{$old_parent}->{'host_children'}});
  
  # Tell the new department about this new host.
  push @{$self->{$dept}->{'host_children'}}, $host;
 
  # Change parent information in moved host
  $self->{$host}->{'parent'} = $dept;
 
  # Propagate new serial number to both the new parent department and the old
  # parent department.
  $self->propagate_serial($dept,$in_serial) if defined $in_serial;
  $self->propagate_serial($old_parent,$in_serial) if defined $in_serial;
 
  return 1;
} # end move_host

#============================================================================#
#
# Function:  remove_host
#
# This will remove a host from the hierarchy.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) for all paths
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->remove_host';
#  unless(eval "$function_call".
#        '($global::transactions_in_process->{$trans_id}'.
#        ',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub remove_host
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$host) = @{$_[0]};
  my($type,$trans_id) = @{$_[1]};
  my $in_serial = $_[2];

  # Get the parent of the host we are going to move.
  my $old_parent = $self->{$host}->{'parent'};
 
  # Remove $host as a host on $old_parent.
  @{$self->{$old_parent}->{'host_children'}}
    = grep($_ ne $host, @{$self->{$old_parent}->{'host_children'}});
 
  # Remove host from array.
  delete $self->{$host};

  # Propagate new serial number to the old parent department.
  $self->propagate_serial($old_parent,$in_serial) if defined $in_serial;
 
  return 1;
} # end remove_host

#============================================================================#
#
# Function:  move_manager
#
# This function updates the hierarchy with the new manager information.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_manager';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'
#        .',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#
 
sub move_manager                                                                  
{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$dept,$man_host) = @{$_[0]};
  my($type,$trans_id,$man_port) = @{$_[1]};
  my $in_serial = $_[2];
 
  # Change the manager information.
  $self->{$dept}{'manager_host'} = $man_host;
  $self->{$dept}{'manager_port'} = $man_port;
 
  $self->propagate_serial($dept,$in_serial) if defined $in_serial;
 
  return 1;
} # end move_manager

#============================================================================#
#
# Function:  move_aggregator
#
# This function updates the hierarchy with the new aggregator information.
#
# Inputs:
#  Actual Arguments:
#   An array containing original transaction information
#   An array containing htc transaction information
#   The new hierarchy serial number
#  Global:
#    None
#
# Return Value(s):
#  True (1) or undef
#
# Global variable modifications:
#  None
#
# Example of usage (from OHS):
#  my $function_call = '$global::hierarchy->move_aggregator';
#  unless(eval "$function_call".'($global::transactions_in_process->{$trans_id}'#        .',$messageref,$global::current_view)')
#
#----------------------------------------------------------------------------#

sub move_aggregator

{
  my $self = shift;
  my $log = $self->{'_log'};
  my($hdr,$trans_id,$user,$pass,$serial,$dept,$agg_host) = @{$_[0]};
  my($type,$trans_id,$agg_port) = @{$_[1]};
  my $in_serial = $_[2];

  # Change the manager information.
  $self->{$dept}{'aggregator_host'} = $agg_host;
  $self->{$dept}{'aggregator_port'} = $agg_port;

  $self->propagate_serial($dept,$in_serial) if defined $in_serial;

  return 1;
} # end move_aggregator
  
#============================================================================#
#
# Function:  compare
#
# Compares this hierarchy to another.
#
# Inputs:
#  Actual Arguments:
#	Name for our hierarchy
#	Name for other hierarchy
#	Other hierarchy object
#
# Return Value(s):
#  None
#
# Example of usage
#  return $hierarchy->compare('ohs','local',$local_h);
#
#----------------------------------------------------------------------------#   

sub compare
{
  my($self, $our_name, $their_name, $them) = @_;
  my($log) = $self->{'_log'};
  my($my_key,$their_key);
  my $success = 1;

  my @my_names = sort grep($_ ne '_log' && $_ ne 'root_name',keys %$self);
  foreach $my_key (@my_names)
   {
    # We better at least have something to compare.
    unless(exists $them->{$my_key})
     {
      $log->warn("$my_key in $our_name hierarchy but not in $their_name copy (1).");
      $success = 0;
      next;
     }    
 
   # Compare types
    unless(ref $them->{$my_key} eq ref $self->{$my_key})
     {
      $log->warn("$my_key in $our_name hierarchy of type ".ref $self->{$my_key}.
		"but in $their_name copy was ".ref $them->{$my_key}.".");
      delete $them->{$my_key};
      $success = 0;
      next;
     }    

    # At this point, they are the same type -- compare values.
    my($result,$warning) = $self->{$my_key}
		->compare($my_key,$our_name, $their_name, $them->{$my_key});
    unless($result)
     {
      $log->warn("$my_key was different: $warning");
      $success = 0;
      delete $them->{$my_key};
      next;
     }
    delete $them->{$my_key};
   }
  foreach $their_key (sort grep($_ ne '_log' && $_ ne 'root_name',keys %$them))
   {
    $log->warn("$their_key in $their_name hierarchy but not in $our_name copy (2).");
    $success = 0;
   }

  return $success;
}

#============================================================================#
#
# Function:  is_consistent
#
# Make sure this hierarchy is internally consistent.
#
# Inputs:
#	None.
#
# Return Value(s):
#  	0 on failure, 1 on success
#
# Example of usage
#  return $h->is_consistent();
#
#----------------------------------------------------------------------------#   

sub is_consistent
{
  my($self) = @_;
  my($log) = $self->{'_log'};
  my($key,$parent_name,$parent_object,$instance_count,$child,$child_object);
  my $success = 1;

  my @names = sort grep($_ ne '_log' && $_ ne 'root_name',keys %$self);
  foreach $key (@names)
   {
    if(ref $self->{$key} eq 'Host')
     {
      $parent_name = $self->{$key}{'parent'};
      $parent_object = $self->{$parent_name};
      unless($parent_object)
       {
        $success = 0;
        $log->warn("Parent $parent_name of host $key does not exist in hierarchy");
       }
      unless(ref $parent_object eq 'Department')
       {
        $success = 0;
        $log->warn("Parent $parent_name of host $key is a ".ref $parent_object
		." instead of a Department");
       }
      $instance_count = grep($_ eq $key,@{$parent_object->{'host_children'}});
      unless($instance_count == 1)
       {
        $success = 0;
        $log->warn("Parent $parent_name has child $key $instance_count times".
		" instead of once.");
       }
    }
    elsif(ref $self->{$key} eq 'Department')
     {
      $parent_name = $self->{$key}{'parent'};

      # Check our parent
      unless($parent_name eq 'null')
       {
        $parent_object = $self->{$parent_name};
        unless($parent_object)
         {
          $success = 0;
          $log->warn("Parent $parent_name of department $key does not exist in hierarchy");
         }
        unless(ref $parent_object eq 'Department')
         {
          $success = 0;
          $log->warn("Parent $parent_name of department $key is a "
		.ref($parent_object)." instead of a Department");
         }
        $instance_count 
		= grep($_ eq $key,@{$parent_object->{'dept_children'}});
        unless($instance_count == 1)
         {
          $success = 0;
          $log->warn("Parent $parent_name has child $key $instance_count"
		." times instead of once.");
         }
       }
      # Now check our children.  Only need to check that they exist, since
      # existing ones will have checked our correspondence for themselves.
      foreach $child (@{$self->{$key}{'host_children'}})
       {
        $child_object = $self->{$child};
        unless($child_object)
         {
          $success = 0;
          $log->warn("Child $child of Department $key does not exist in hierarchy");
         }
        unless(ref $child_object eq 'Host')
         {
          $success = 0;
          $log->warn("Child $child of department $key is a ".ref $child_object
		." instead of a Host");
         }
       }
      foreach $child (@{$self->{$key}{'dept_children'}})
       {
        $child_object = $self->{$child};
        unless($child_object)
         {
          $success = 0;
          $log->warn("Child $child of Department $key does not exist in hierarchy");
         }
        unless(ref $child_object eq 'Department')
         {
          $success = 0;
          $log->warn("Child $child of department $key is a ".ref $child_object
		." instead of a Department");
         }
       }
     }
    else
     {
      $success = 0;
      $log->warn("Bad element $key of type ".ref $self->{$key});
     }
   }

  return $success;
}

#============================================================================#
#
# Function:  propagate_serial
#
# Change the serial number of the supplied department and all its ancestors.
#
# Inputs:
#  Actual Arguments:
#    Department and serial number
#
# Return Value(s):
#  None
#
# Example of usage (from OHS):
#  $self->propagate_serial($dept,$in_serial) if defined $in_serial;
#
#----------------------------------------------------------------------------#   
sub propagate_serial
{
  my($self,$dept,$serial) = @_;

  return if $dept eq 'null';
  $self->{$dept}{'view_serial'} = $serial;
  $self->propagate_serial($self->{$dept}{'parent'},$serial);
}

#============================================================================#
#
# Function:  is_empty
#
# This is a boolean function to test wether or not the hierarchy is empty.
#
# Inputs:
#  Actual Arguments:
#    None
#
# Return Value(s):
#  True (1) or False (0)
#
# Example of usage (from OHS):
#  unless($self->is_empty()) 
#
#----------------------------------------------------------------------------#

sub is_empty
{
  my $self = shift;
  return 1 if scalar(keys %$self) == 1; # allow for the logfile entry
  return 0;
}
1;
	
