
# This file contains utility functions for remote start up of module 
# controllers via rsh.  It is mainly used by test scripts, start-up scripts,
# etc.  Note that these routines will only work if the remote host will
# accept rsh commands from this one (check hosts.equiv, .rhosts, etc).

no strict 'vars';
use Grdbm;
use Comm;

#============================================================================#

# Usage:
#
# %tcp_ports = start_module_controllers(@hostname);

# This will 
#	* start up new module controllers;

sub start_module_controllers
{
  package Mod_utility;
  my($log,@hostnames) = @_;
  my($hostname);
  my(%id,%handles,%module_port);

  foreach $hostname (@hostnames)
   {
    # MC NFS file naming convention uses short-form, unqualified hostnames:
    my $host = ( split ('\.', $hostname) )[0];   # safe even if already unqual.

#   if ($host eq ???) {  # could optimize for local mc, but rsh below works ok.
#     ??? = start_local_module_controller();
#     # Then must splice/delete it, to shrink length of @hostnames
#     }

    # Set up the new module controller on the appropriate file.
    $id{$hostname} = "/tmp/mod_output.temp.$host.".int(rand(1000000000));
#
#  Command string below might need a system specific Perl pathname. 
#  Manipulate the comments below if PERLLOC environment variable is needed.
#  22-Jul-97  J.Rowe
    my $command = "rsh $hostname \"cd \\\$GRIDSPATH/module_controller; "
                . "rm -f mod.control.$host.current.tasks; "
         . "\\\$PERLLOC/perl -w ./module_controller.pl\" > $id{$hostname} &";
#                            . "./module_controller.pl\" > $id{$hostname} &";
#
    $log->warn("About to start MC on $hostname\n");
    system($command); 
   }

  # Now try and get the various ports

# Delay things a bit ( 3 seconds ) to allow the module controller to catch up.
  sleep 3;

  my($MAXLOOPS) = 500;
  my($giveup)=0;
  while ($giveup < $MAXLOOPS) {
    # RC 4/8/97:  upped from 30 due to shorter sleep/select
    last if keys %module_port == @hostnames; # as many ports as hosts;
    my($new)=0;
    my $temp_handle;

    foreach $hostname (@hostnames) {
      unless($handles{$hostname}) {
        if( -f $id{$hostname}) {
          $handles{$hostname} = "FILE_$hostname";
          open($handles{$hostname},$id{$hostname}) 
				or die "Couldn't get $id{$hostname}";
         }
       }
      unless($module_port{$hostname}) {
        # RC 4/8/97:  reposition only just before reading:
        seek($handles{$hostname},0,1);       # needed to reset EOF error!
        $temp_handle = $handles{$hostname};  # Cannot read file via hash var!
        while(<$temp_handle>) {
          if(/MODULE CONTROLLER LISTENING ON PORT (\d+)/) {
            $module_port{$hostname} = $1;
            $log->warn("Got port $1 for MC on $hostname\n");
            $new++;
           }
         }
       }
     }
    # RC 4/8/97:  This should be faster than <sleep 1>:
    select (undef, undef, undef, 0.5);
    $giveup= $new ? 0 : $giveup +1;
    }

  foreach $hostname (grep(!$module_port{$_},@hostnames))
   {
    $log->warn("Could not start module contoller on $hostname");
    close($handles{$hostname});
   }
  $log->warn("Module controllers started successfully") if keys %module_port ==
             @hostnames;
  return %module_port;
}

#============================================================================#

# Usage:
#
# start_ohs($grids_db);

# This will 
#	* start up a new ohs

sub start_ohs
{
  package Mod_utility;
  my($grids_db) = @_;
#
# Running on non-UCDseclab environments ( perl binaries not in /pkg/bin ) may
# require a PERLLOC environment variable to point explicitly to the perl
# binary.  If so the following comment line must be manipulated. JRowe 2/9/98
#
  my $command = ("rsh $grids_db->{'ohs_host'} \"\\\$PERLLOC/perl -w \\\$GRIDSPATH/ohs/ohs ".
#  my $command = ("rsh $grids_db->{'ohs_host'} \"\\\$GRIDSPATH/ohs/ohs ".
	"global::output_mode=\\'comm\\' ".
	"global::ohs_port=$grids_db->{'ohs_port'} ".
	"global::init_user=\\'$grids_db->{'init_user'}\\' ".
	"global::init_pass=\\'$grids_db->{'init_pass'}\\' ".
	"</dev/null >/dev/null &\" &");
  system($command); 
  return 1;
}

#============================================================================#

# Usage:
#
# start_root_sm($grids_db);

# This will 
#	* start up a new root software manager

sub start_root_sm
{
  package Mod_utility;
  my($grids_db) = @_;
  my $command = "rsh $grids_db->{'root_sm_host'} \"\\\$GRIDSPATH/sm/sm ".
    "global::output_mode=\\'comm\\' ".
    "global::sm_port=$grids_db->{'root_sm_port'} ".
    "global::init_user=\\'$grids_db->{'init_user'}\\' ".
    "global::init_pass=\\'$grids_db->{'init_pass'}\\' ".
    "</dev/null >/dev/null &\" &";
  system($command); 
  return 1;
}

#============================================================================#

sub wait_for_ohs_and_root_sm
{
  package Mod_utility;
  my($grids_db,$log) = @_;
  unless($Comm::INIT)
   {
    my ($status) = Comm::init(0,0);
    unless ($status eq 'ok')
     {
      die "Couldn't get Comm.pm init in Mod_utility.pm\n";
     }
   }
  my $ohs = Comm::ping($grids_db->{'ohs_host'},$grids_db->{'ohs_port'},undef);
  die "Couldn't get ohs!" unless $ohs;
  $log->warn("Got OHS on $grids_db->{'ohs_host'}");
  my $sm = Comm::ping($grids_db->{'root_sm_host'},
					$grids_db->{'root_sm_port'},undef);
  die "Couldn't get root sm!" unless $sm;
  $log->warn("Got root SM on $grids_db->{'root_sm_host'}");
  return 1;
}

#============================================================================#

sub start_local_module_controller
{
  package Mod_utility;

  $localhost = `hostname`; chomp $localhost;

  # Get rid of any old module controllers
  unless(&main::slay($localhost,'module_'))
   {
    print STDERR "Couldn't slay\n";
    return undef;
   }

  # MC NFS file naming convention uses short-form, unqualified hostnames:
  $localhost = ( split ('\.', $localhost) )[0];   # safe even if already unqual.

  # Set up the new module controller on the appropriate file.
  system("rm -f ./mod.control.$localhost.current.tasks");
  my $id = "mod_output.temp.$localhost.".time;
  system("\\\$PERLLOC/perl -w ./module_controller.pl > /tmp/$id &"); 

  while(1) 
   {
    sleep 1; 
    if( -f "/tmp/$id") 
     { last;} 
   }

  open(MCOUTPUT,"/tmp/$id") or die "Couldn't get /tmp/$id";

  my $module_port;  
  GETTING_PORT:
  while(!$module_port)
   {
    while(<MCOUTPUT>)
     {
      print;
      if(/MODULE CONTROLLER LISTENING ON PORT (\d+)/)
       {
        $module_port = $1;
        last GETTING_PORT;
       }
     }
    seek(MCOUTPUT,0,1);
   }
  close(MCOUTPUT);
  return $module_port;
}

#============================================================================#

# Usage
#
# slay($hostname,$pattern [,$no_blocking])
#
# kills all processes on $hostname which match pattern in the appropriate
# ps listing.  Returns 1 if all went well, or undef if there was a problem

sub slay
{
  package Mod_utility;
  my($hostname,$pattern,$no_blocking) = @_;
  my $system_type = &main::get_system($hostname); 
  return undef unless $system_type;
  my $kill = '';

  if($system_type eq 'sunos' || $system_type eq 'nextstep')
   {
    $ps = '/bin/ps -gxww';
   }
  elsif($system_type eq 'solaris')
   {
    $ps = '/usr/ucb/ps -gxww';
   }
  elsif($system_type eq 'ultrix')
   {
    $ps = '/bin/ps -gxww';
   }
  elsif($system_type eq 'hpux')
   {
    $ps = '/bin/ps -ef';
   }
  elsif($system_type eq 'irix')
   {
    $ps = '/bin/ps -ef';
   }
  elsif($system_type eq 'linux')
   {
    $ps = '/bin/ps -xw';
   }
  elsif($system_type eq 'freebsd')
   {
    $ps = '/bin/ps -xw';
   }
  else
   {
    print STDERR "System type unknown in Mod_utility.pm\n";
    return undef;
   }
  
  open(PS,"rsh $hostname $ps|") or return undef;
  my @lines = <PS>; close(PS);
  foreach (@lines)
  {
      if(/$pattern/) {
	  if ($system_type eq 'hpux') {
	      /^\s*\w*\s*(\d+)\s+/;
	      $kill .= " $1";
	  }
	  elsif ($system_type eq 'irix') {
	      /^\s*\w*\s*(\d+)\s+/;
	      $kill .= " $1";
	  }
	  else {
	      /^\s*(\d+)\s+/;
	      $kill .= " $1";
	  }
      }
  }

  if($kill && $no_blocking)
   {
    system("rsh $hostname kill -9 $kill &"); 
   }
  elsif($kill)
   {
    system("rsh $hostname kill $kill"); # ought to check return value
   }
  return 1;
}

#============================================================================#

sub quick_slay
{
  my($style, $grids_db,@hosts) = @_;
  my($host,@replies);
  my($success) = 1;

  my @body = (
    'SET',                   #command
    'module_controller',     #module
    'N/A',                   #version
    'N/A',                   #department
    'mc_command', 'KILL',
    'mod_department', '*',
    'mod_module',     '*',
    'mod_version',    '*',
    );
  # must explicitly order regicide or suicide ;-)
  push (@body, 'kill_sm', 'kill_sm') if $style =~ /kill_sm/;
  push (@body, 'mc_shutdown', 'suicide') if $style =~ /suicide/;

  unless($Comm::INIT)
   {
    my ($status) = Comm::init(0,0);
    unless ($status eq 'ok')
     {
      die "Couldn't get Comm.pm init in Mod_utility.pm\n";
     }
   }
  foreach $host (@hosts) 
   {
    my $port = $grids_db->{'module_ports'}{$host};
    unless($port)
     {die("Cannot shut down $host - don't know about it.");}
    &Comm::tcp_send($host, $port, 's', @body);
   }

  return $success if $style =~ /suicide/; # don't want to hang waiting.

  foreach (0 .. $#hosts) 
   {
    my($rhost,$rport,$header,@result) = &Comm::tcp_recv ('blocking');
    unless($result[0] =~ /^OK|^WARN/)
     {
      warn "Bad reply:$result[0]";
      $success = 0;
     } 
    push @replies,$rhost;
   }
  my $replies = join(' ',sort @replies);
  my $sends = join(' ',sort @hosts);
  unless($replies eq $sends)
   {
    $success = 0;
    warn "Replies [$replies] different from sends [$sends]";
   }
  return $success;
}

#============================================================================#

# Usage
#
# $system_type = get_system($hostname)
#
# gets a string representing the system type of the remote host.  Currently
# known values are "sunos", "solaris", "nextstep", "irix".

sub get_system
{
  package Mod_utility;
  my($hostname) = @_;

  return $system_type{$hostname} if $system_type{$hostname};

  # We don't already know - consult uname.
  my $answer = `rsh $hostname uname -a`;

  if(!$answer)
   {
    # Two likely contingencies: a machine that doesn't have uname (unmodified 
    # NeXTs), or a machine on which we don't have permission.  It's probably 
    # unwise to proceed.
    $system_type{$hostname} = undef;
    return undef;
   }  

  $answer =~ tr/A-Z/a-z/; # make lower case

  if($answer =~ /^sunos\s+\S+\s+5\./)
   {
    # SunOS 5.* = Solaris (Sys V based)
    $system_type{$hostname} = 'solaris';
   }
  elsif($answer =~ /^sunos\s+\S+\s+4\./)
   {
    # SunOS 4.* = SunOS (BSD based)
    $system_type{$hostname} = 'sunos';
   }
  elsif($answer =~ /^nextstep/)
   {
    # Note - by default NeXTs do not have uname.  However, we've installed
    # a version on some systems.
    $system_type{$hostname} = 'nextstep';
   }
  elsif($answer =~ /^irix/)
   {
    # SGI systems.
    $system_type{$hostname} = 'irix';
   }
  elsif($answer =~ /^hp-ux/)
   {
    # HP systems.
    $system_type{$hostname} = 'hpux';
   }
  elsif($answer =~ /^ultrix/)
   {
    # DEC systems.
    $system_type{$hostname} = 'ultrix';
   }
  elsif($answer =~ /^linux/)
   {
    # Linux systems.
    $system_type{$hostname} = 'linux';
   }
  elsif($answer =~ /^freebsd/)
   {
    # FreeBSD systems.
    $system_type{$hostname} = 'freebsd';
   }

  return $system_type{$hostname};
}

#============================================================================#

1;
