# This file is intended to be used to check the static syntax of messages in 
# GCPF.  It has only one externally visible function which has usage
#
# if(&message_syntax_ok($message_header,$message_body_ref) { ....}
#
# If this function returns true, then, for supported message types, the
# message is syntactically correct as far as can be told by simple pattern
# matching without any semantic knowledge.

#============================================================================#

sub message_syntax_ok
{
  package Static_message_syntax;
  my($header,$messageref) = @_;
  
  return 0 unless $header_known{$header}; 
  
  # Call the appropriate function via an associative array of function
  # refs.
  return &{$message_checkers{$header}}($header,$messageref);
}

#============================================================================#

# Internals of operation
#
# From &message_syntax_ok, we call one of a series of functions in an
# associative array called %message_checkers, which is keyed by the header
# of the particular message to be checked.  

# The most common function to be used here is &check_patterns.  It works on 
# functions which always have a fixed number of fields with fixed syntax.
# It operates by comparing each field with a corresponding pattern.  The
# patterns are specified in an associative array %pattern_hash (also keyed
# by header).

# In some special cases, the syntax of the message depends in part on
# some of the field values in addition to the header itself.  For example,
# the htr and htc messages are this way.  These use a check_ht_special
# function in place of directly calling check_patterns.  This basically
# works by setting up a more complex key based on both the header and the
# variant field, and then calling check_patterns on the result.

#============================================================================#

package Static_message_syntax;

#============================================================================#

sub check_patterns
{
  my($header,$messageref) = @_;
  return 0 if($#{$pattern_hash{$header}} != $#$messageref);
  foreach (0..$#$messageref)
   {
    #print STDERR "Check $messageref->[$_] on ${$pattern_hash{$header}}[$_]\n";
    return 0 unless($messageref->[$_] =~ /${$pattern_hash{$header}}[$_]/);
   }
  return 1;
}

#============================================================================#

sub check_ht_special
{
  my($header,$messageref) = @_;
  return 0 unless defined $messageref->[0];
  my $key = "$header#$messageref->[0]";
  if($header eq 'htr')
   {return 0 unless defined $pattern_hash{$key};}
  elsif($header eq 'htc')
   {
    return &check_patterns($key,$messageref) if defined $pattern_hash{$key};
    return &check_patterns('htc',$messageref);
   }
  else 
   {return 0;}
  return &check_patterns($key,$messageref);
}

#============================================================================#

# This is the section which defines all the patterns we need to check 
# different kinds of messages;

# Types of header we currently recognize.
%header_known = ('htr' => 1,	'htp' => 1,	'htc' => 1,	'hte' => 1, 
		 'hvr' => 1,	'hv' => 1,	'hve' => 1,	'hvu' => 1);

# References to functions to handle the different headers.
%message_checkers =
		    (
	 	     'htr' => \&check_ht_special,
		     'htp' => \&check_patterns,
		     'htc' => \&check_ht_special,
		     'hte' => \&check_patterns,
		     'hvr' => \&check_patterns,
		     'hv' =>  \&check_patterns,
		     'hve' => \&check_patterns,
		     'hvu' => \&check_patterns
		    );
	
# Common patterns we use
$port      = '\A\d{1,5}\Z';		# From 1 to 5 didgets
$error     = '\A[!-~\s]*\Z';		# printable characters 
$host	   = '\A[a-zA-Z][-\w\.]*\Z';	# Alphanumeric plus '.' and '-'
$dept      = '\A[a-zA-Z]\w*\Z';		# Alphanumeric
$tr_id     = '\A\w+\Z';			# Alphanumeric
$type	   = '\A[a-zA-Z]\w*\Z';		# Alphanumeric
$user	   = '\A[a-zA-Z]\w*\Z';		# Alphanumeric
$pwd       = '\A[!-~\s]*\Z';		# printable characters 
$serial    = '\A\d+\Z';			# Digits only
$hierarchy = '\A[!-~\s]*\Z';		# printable characters 

%pattern_hash = 
(
 # Generic types of messages.
  
 'htp' => [$tr_id],
 'htc' => [$type,$tr_id],
 'hte' => [$tr_id,$error],
 'hvr' => [$user,$pwd,$dept],
 'hv'  => [$dept,$serial,$hierarchy],
 'hve' => [$user,$dept,$error],
 'hvu' => [$dept,$serial],
  
 # htr messages
  
 'htr#new_root' => [$type,$tr_id,$user,$pwd,$dept,$host,$port,$host,$port],
 'htr#add_dept' => [$type,$tr_id,$user,$pwd,$serial,$dept,$dept,$host,$host],
 'htr#add_host' => [$type,$tr_id,$user,$pwd,$serial,$dept,$host,$port],
 'htr#change_variable' => [$type,$tr_id,$user,$pwd,$serial,$dept],
 'htr#move_dept' => [$type,$tr_id,$user,$pwd,$serial,$dept,$dept],
 'htr#move_host' => [$type,$tr_id,$user,$pwd,$serial,$dept,$host],
 'htr#move_manager' => [$type,$tr_id,$user,$pwd,$serial,$dept,$host],
 'htr#move_aggregator' => [$type,$tr_id,$user,$pwd,$serial,$dept,$host],
 'htr#remove_dept' => [$type,$tr_id,$user,$pwd,$serial,$dept],
 'htr#remove_host' => [$type,$tr_id,$user,$pwd,$serial,$host],
 'htr#change_user' => [$type,$tr_id,$user,$pwd,$serial,$user,$pwd,$dept],
  
 # htc messages with special cases.
 
 'htc#add_dept' => [$type,$tr_id,$port,$port],
 'htc#move_manager' => [$type,$tr_id,$port],
 'htc#move_aggregator' => [$type,$tr_id,$port],
);

#============================================================================#

1;
