#!/pkg/bin/perl -w

# $Id: Timeout.pm,v 1.7 1996/10/10 03:39:50 hoagland Exp $

package Timeout;

use strict 'refs';

# TIME-OUT MANAGER

# The timeout manager is a utility for storing an unlimited number of events
# that need to occur at a particular clock time.  The main operations of the
# manager are adding and deleting of events, and the retrieval of events that
# have recently become due.

# The timeout manager is implemented as a Perl class named "Timeout".  An
# instance of class should be created for each group of events that are to be
# dealt with together.  For example, this may correspond to the activity
# reports in the buffer of the engine, or the graphs to timeout in a ruleset.

# The basic units that the timeout manager works with are "events".  Events
# can be any type of scalar, but should be unique within an instance of the
# timeout manager.  Events could, for example, be identifying strings or even
# Perl references.  The data stored inside the manager should be considered
# private and only accessed through the public methods described below.

# Notification of events that have come due, that is whose scheduled time has
# arrived is done through a query to the manager.  A synchronous design was
# chosen to avoid race conditions that can occur in an asynchronous
# methodology.

# These are the public methods of the timeout manager.


$BIG_NUM= 9999999999999999999;

sub BEGIN {
  $get_timeref=sub {CORE::time};  # the routine to call to get the time
}

# sets the time functon to the provided code reference
sub set_time_function {
  $get_timeref= $_[1];
}


# new- creates and returns a new instance of the Timeout class
sub new {
  return bless {'t2e'=> {},'e2t'=> {}, 'first_time' => $BIG_NUM, 'nevents'
=> 0};
}

# add- takes a time indicator and a event to be added to the schedule.  The
# time indicator is an integer that indicates when this event is scheduled to
# occur.  If its value is less than 100,000 is interpreted as an offset, in
# seconds, from the time of addition otherwise is a Unix time at which the
# event is to occur.  If the event given is already on the schedule, the
# former occurrence is forgotten.
sub add {
  my($self,$time,$event)= @_;
  my $oldt;
  
  $time+= &{$get_timeref} if $time < 100000;
  
  $self->delete($event) if ($oldt= $self->{'e2t'}{$event}); #there already
  $self->{'e2t'}{$event}= $time;
  push(@{$self->{'t2e'}{$time}},$event);
  $self->{'nevents'}++;
  $self->{'first_time'}= $time if $time < $self->{'first_time'};
}

# delete- The parameter is an event that has already been added to the data
# structures of the timeout manager.  This function deletes the specific item
# passed in from the schedule.  If the entry does not exist, then the
# deletion request is ignored.
sub delete {
  my($self,$event)= @_;
  my($time);

  if ($time= $self->{'e2t'}{$event}) {
    delete $self->{'e2t'}{$event};
    $self->{'nevents'}--;
    @{$self->{'t2e'}{$time}}= grep($_ ne $event,@{$self->{'t2e'}{$time}});
    unless (@{$self->{'t2e'}{$time}}) { # last event at this time
      delete $self->{'t2e'}{$time};
      if ($time == $self->{'first_time'}) { # the deleted event was first up
	$self->{'first_time'}= $BIG_NUM;
	if ($self->{'nevents'} > 0) {
	  foreach (keys %{$self->{'t2e'}}) {
	    $self->{'first_time'}= $_ if $_ < $self->{'first_time'};
	  }
	}
      }
    }
  }
}

# timed_out- The timed_out function checks the time from the system
# clock and the schedule and returns all of the events that have come
# due since this function was last called in a time-ordered ordered
# list.  If a second argument is given, its value used instead of the
# real system clock for the current time.
sub timed_out {
  my($self,$now)= @_;
  $now= &{$get_timeref} unless defined($now);
  
  return () unless $self->{'nevents'} > 0 && $now > $self->{'first_time'};

  my @times= (); # the time at which stored events should have occured

  # going through all keys may be faster if $now - first_time >
  # no. events, but we assume timed_out is usually called in a timely
  # manner, so $now - first_time is small
  # new: hueristic decides which way to do it
  if ($now-$self->{'first_time'} <= 2*$self->{'nevents'}) {
    # not a large time span compared to number of event entries
    foreach $time ($self->{'first_time'} .. $now) {
      push(@times,$time) if (defined($self->{'t2e'}{$time}));
    }
  } else { # a large time span compared to number of entries, so above
           # isn't efficient
    foreach $time (keys %{$self->{'t2e'}}) {
      push(@times,$time) if ($time <= $now);
    }
  }
  my @evs= ();  # the times at which events occured
  foreach $time (@times) {
    push(@evs,@{$self->{'t2e'}{$time}});
    delete $self->{'t2e'}{$time};
  }

  $self->{'nevents'}-= @evs;
  # update first_time
  if ($self->{'nevents'} > 0) {
    $self->{'first_time'}= $BIG_NUM;
    foreach (keys %{$self->{'t2e'}}) {
      $self->{'first_time'}= $_ if $_ < $self->{'first_time'};
    }
  } else {
    $self->{'first_time'}= $BIG_NUM;
  }

  
  foreach (@evs) {
    delete $self->{'e2t'}{$_};
  }
  return @evs;
}


# how_long-  Allows the callee to find out when the next event will come due
# if no new additions are made.  The return value is the Unix time of that
# occurrence.  If there are no events in the data structure, then zero is
# returned instead of a Unix time.
sub how_long {
  return $_[0]->{'first_time'} >= $BIG_NUM ? 0 : $_[0]->{'first_time'};
}

1;
