#!/pkg/bin/perl -w

# $Id: attr.pl,v 1.19 1998/01/19 23:57:01 hoagland Exp $

require "parse.pl";

# attributes are represented in the graph engine in three different
# ways, depending on their type:
# + scalars are represented as perl scalar
# + lists are represented as a reference to a perl list
# + sets are represented as a reference to a hash, where the keys of the hash
#    that have value other than '' or 0 are the members of the set

# this file contains routines that do some handy stuff given attributes

# This function returns whether or not the attribute argument is a scalar.
#
# is_scalar: attribute -> boolean
#
# e.g. if (&is_scalar($scalattr)) { ...
#
sub is_scalar {
  return !ref($_[0]);
}

# This function returns whether or not the attribute argument is a list.
#
# is_list: attribute -> boolean
#
# e.g. if (&is_list($attr)) { ...
#
sub is_list {
  return ref($_[0]) eq 'ARRAY';
}

# This function returns whether or not the attribute argument is a set.
#
# is_set: attribute -> boolean
#
# e.g. if (&is_set($attr)) { ...
#
sub is_set {
  return ref($_[0]) eq 'HASH';
}

###############

# This function returns the list that the attribute represents.
#
# list_members: list-attribute -> list
#
# e.g. @list= &list_members($listattr)
#
sub list_members {
  return @{$_[0]};
}

# This function returns a list of the members of the set that this
# attribute represents.
#
# set_members: set-attribute -> list
#
# e.g. @list= &set_members($setattr)
#
sub set_members {
  my(%set)= %{$_[0]};
  return grep($set{$_},keys %set);
}

# This function returns whether of not a given scalar is a member of
# the set that this attribute represents.
#
# is_a_member: set-attribute x scalar -> boolean
#
# e.g. if (&is_a_member($setattr, $possible_member)) { ...
#
sub is_a_member {
  return ${$_[0]}{$_[1]};
}

# This function returns a list of the scalar members of whatever type of 
# attribute it is.  For sets and lists, the meaning is clear; with a 
# scalar, it just returns the scalar itself and undefined returns undefined.
#
# members: attribute -> list
#
# e.g. @list=&members($attr)
#
sub members {
  my($attr)=shift;
  return undef unless defined($attr);
  return ($attr) if &is_scalar($attr);
  return &list_members($attr) if &is_list($attr);
  return &set_members($attr);
}

# head returns the first element of list.
#
# head: list-attribute -> scalar
#
# e.g. $x= &head($listattr);
#
sub head {
  return ${$_[0]}[0];
}

# last returns the last element of list.
#
# last: list-attribute -> scalar
#
# e.g. $x= &last($listattr);
#
sub last {
  my(@list)= @{$_[0]};
  return $list[$#list];
}

################

# This function tests to attributes for equality.  They are equal if
# they are of the same type (list,set,scalar) and their contents are
# identical.  The boolean result is returned (true indicating equality).
#
# attr_equal: attribute x attribute -> boolean
#
# e.g. if (&attr_equal('a',$attr)) { ...
#
sub attr_equal {
  my($a,$b)= @_;
  return 0 unless (ref($a) eq ref($b));
  if (&is_scalar($a)) {
    return $a eq $b;
  } else { # a list or set
    my(@ma)= &members($a);
    my(@mb)= &members($b);
    # make sure same size
    return 0 unless ($#ma == $#mb);
    # compare each element in list;
    # this works for sets too since perl's keys function will always return
    # the keys in the same order if the same keys are present
    for ($[..$#ma) {
      return 0 unless $ma[$_] eq $mb[$_];
    }
    return 1;
  }
}

################

# This function creates a set attribute with the arguments as initial
# members.  Undefined arguments are ignored.
#
# create_set: scalar x scalar x ... -> set-attribute 
#
# e.g. $setattr= &create_set('a','d')
#
sub create_set {
  my(%set)=();
  foreach (@_) {
    $set{$_}= 1 if defined($_);
  }
  return \%set;
}

# This function adds the arguments to an existing set attribute and
# returns the result for convenience.
#
# add_to_set: set-attribute x scalar x scalar x ... -> set-attribute 
#
# e.g. &add_to_set($setattr,'1',6')
#
sub add_to_set {
  my($setref)=shift;
  foreach (@_) {
    ${$setref}{$_}= 1;
  }
  return $setref;
}

# This function deletes the arguments in an existing set attribute and
# returns the result for convenience.
#
# remove_from_set: set-attribute x scalar x scalar x ... -> set-attribute 
#
# e.g. &remove_from_set($setattr,'1',6')
#
sub remove_from_set {
  my($setref)=shift;
  foreach (@_) {
    delete ${$setref}{$_};
  }
  return $setref;
}

# This function creates a list attribute with the arguments as initial members.
#
# create_list: scalar x scalar x ... -> list-attribute 
#
# e.g. $listattr= &create_list('a','d')
#
sub create_list {
  return [@_];
}

# This function adds the arguments to the start of an existing list
# attribute and returns the result for convenience.
#
# prepend: list-attribute x scalar x scalar x ... -> list-attribute 
#
# e.g. &prepend($listattr,'1',6')
#
sub prepend {
  my($listref)=shift;
  unshift(@{$listref},@_);
  return $listref;
}

# This function adds the arguments to the end of an existing list
# attribute and returns the result for convenience.
#
# append: list-attribute x scalar x scalar x ... -> list-attribute 
#
# e.g. &append($listattr,'1',6')
#
sub append {
  my($listref)=shift;
  push(@{$listref},@_);
  return $listref;
}


######

# This function returns a copy of the given attribute. I.e. if a set
# is passed in, then a new set is constructed.
#
# copy_attr: attribute -> attribute
#
# e.g. $listattr2= &copy_attr($listattr)
#
sub copy_attr {
  my($attr)=shift;
  return ($attr) if &is_scalar($attr);
  return [@{$attr}] if &is_list($attr);
  return &create_set(&set_members($attr));
}

########

# These functions deal with converting attribute between these
# representations: DOT, flattened, and the perl representation used
# elsewhere in this file.  These representations are detailed here and in the tech. report.
#
# internal perl representation: as described above
# 
# flattened attribute representation: This converts any of the three
# attribute types into a single string.
# + Sets are represented as a comma-separated list surrounded by
# curly-braces, where the elements in the list are the scalar members
# of the set.
# + Lists are represented as a comma-separated list surrounded by
# square-brackets, where the elements in the list are the scalar
# members of the list in the sane order.
# + All scalars (including those inside sets and lists) need to be
# encoded to avoid ambiguity.  For example, what if you have a comma
# in one of the scalars in a list?  Any character in scalars is free
# to be encoded as follows, but at a minimum, '{','}','[',']',',' and
# '%' need to be encoded.  Encoded characters are translated to the
# string %xx where xx is the two-digit ascii value of the character in
# hex format.  This encoded is referred to as application-independent
# (ai) encoding.
#
# DOT representation of attributes: (the destination language is is
# really not DOT but the Graph Language described in the tech. report,
# which is a restricted set of DOT).  This encoded is to make a
# somehow-encoded-as-a-string (flattened-attribute encoding?)
# attribute safe to put as an attribute value in DOT.  Double quotes
# need to be put around the string.  Certain characters need to be
# encoded to prevent ambiguity.  Any character in the sting is free to
# be encoded as follows, but at a minimum, '"', '\', '#', whitespace
# and non-printable characters need to be encoded.  Encoded characters
# are translated to the string #xx where xx is the two-digit ascii
# value of the character in hex format.  After this encoding is done,
# the resulting string is okay to pass to DOT.

# dot_decode this function undoes what is done in, i.e. dot_encode, by
# translating any #xx's encountered into the character whose
# hexadecimal value is xx. i.e. #21 -> !.  In addition, trailing and
# leading double quotes are removed.
#
# dot_decode: text -> text
#
# e.g. &text= &dot_decode($etext);
#
sub dot_decode {
  $_= shift;
  s/^\"//; s/\"$//;
  s/\#(..)/chr(hex($1))/eg;
  return $_;
}

# ai_decode this function undoes what is done in, i.e. ai_encode, by
# translating any %xx's encountered into the character whose
# hexadecimal value is xx. i.e. %21 -> !
#
# ai_decode: text -> text
#
# e.g. &text= &ai_decode($etext);
#
sub ai_decode {
  $_= shift;
  s/%(..)/chr(hex($1))/eg;
  return $_;
}


# ai_encode takes a string and translates it into something that is
# safe to be used as a scalar part of a flattened attribute text.
# This means that any character may be encoded as %xx where xx is the
# 2 digit hex representation of ascii value of the character being
# replaced, but at least the following characters must be encoded:
# {}[],%.
#
# ai_encode: text -> text
#
# e.g. $etext= &ai_encode($text);
#
sub ai_encode {
  $_=shift;
  s/([\{\}\[\],%])/sprintf("%%%02x",ord($1))/eg;
  return $_;
}

# dot_encode makes some flattened attribute text okay to put down as
# an attribute value in DOT, including surrounding quotes.  This means
# that any character may be encoded as #xx where xx is the 2 digit hex
# representation of ascii value of the character being replaced, but
# at least these must be encoded: "\# and non-printable characters and
# whitespace.  As a matter of fact this function does a close to
# minimal job of encoding.  In particular, any of the built-in dot
# attribute values remain the same.
#
# dot_encode: text -> text
#
# e.g. $etext= &dot_encode($text);
#
sub dot_encode {
  $_=shift;
  s/([\x00-\x1f\"\\\#\s\x80-\xff])/sprintf("#%02x",ord($1))/eg;
  return '"'.$_.'"';
}


# This function creates one of the three types of attributes based on
# the flattened text given.  This text may have come from, i.e.,
# parse_dot::split_text.
#
# unflatten_attr: text -> attribute
#
# e.g. $attr= &unflatten_attr('{a,b,c}')
#
sub unflatten_attr {
  $_=shift; 
  my(@scalars);
  if (s/^\{(.*)\}$/$1/) { # this is safe since '{' and '}' are encoded
                          # at the ai level
    @_= split(/,/,$_); # this is safe since commas are encoded at the ai level
    return &create_set(map(&ai_decode($_),@_));
  } elsif (s/^\[(.*)\]$/$1/) { # this is safe since '[' and ']' are
                               # encoded at the ai level
    @_= split(/,/,$_); # this is safe since commas are encoded at the ai level
    return &create_list(map(&ai_decode($_),@_));
  } else {
    s/%(..)/chr(hex($1))/eg; # inlining of ai_decode
    return $_;
  }
}

# This function flattens takes one of the three types of attributes
# and produces dot-encoded, flattened text that is usable in DOT.
#
# flatten_attr: attribute -> text
#
# e.g. $text= &flatten_attr($attr)
#
sub flatten_attr {
  my($attr)=shift;

  if (&is_scalar($attr)) {
    return &dot_encode(&ai_encode($attr));
  } elsif (&is_list($attr)) {
    return &dot_encode("[".join(',',map(&ai_encode($_),&list_members($attr)))."]");
  } else {
    return &dot_encode("{".join(',',map(&ai_encode($_),&set_members($attr)))."}");
  }
}

# This function takes one of the three types of attributes and produces a
# string version.  Sets are surrounded by '{' and '}' and lists by '[' and
# ']' and members seperated by commas.  Output may be ambiguous if the
# attribute contains any of these introduced characters
#
# simple_flatten_attr: attribute -> text
#
# e.g. $text= &simple_flatten_attr($attr)
#
sub simple_flatten_attr {
  my($attr)= shift;
  return &is_scalar($attr)?(defined($attr)?$attr:'*undef*'):(&is_list($attr)?'['.join(',',&list_members($attr)).']':'{'.join(',',&set_members($attr)).'}');
}

1;
