#!/pkg/bin/perl -w

# $Header: /home/cvs/grids/code/common/check_rules.pl,v 1.6 1996/09/26 20:55:06 hoagland Exp $

# This file contains functions that check the syntax of rulesets and
# queries.  This should be called before sending the text to the
# engine to allow easier reporting of errors to the user.

require 'parse.pl';
use strict "vars";

# grab_rule_block takes a list of lines, and returns a reference to a
# list of lines that are before the first line containing only a '}'
# and the remaining lines.  Any trailing ';' is removed from the lines
# in the block that are returned.
#
# grab_rule_block: {line} -> block_lines-ref x {line}
#
# e.g. ($attr_declsref,@lines)= &grab_rule_block(@lines);
#
sub grab_rule_block {
  my(@rules)= ();
  while (($_= shift) !~ /^}/) {
    s/;$//;
    push(@rules,$_);
  }
  return (\@rules,@_);
}

# check_ruleset take the source text of a ruleset and returns a list
# of any errors found in the syntax. If an optional second argument is
# given and is true, then the names of functions called are checked
# against those that have been set up in the Function package; if this
# hasn't been set up, then this option can't be used.
#
# check_ruleset: text [x boolean] -> {error_text}
#
# e.g. @errors=&check_ruleset($rule_text)
#      @errors=&check_ruleset($rule_text,1)
#
sub check_ruleset {
  my(@lines)= grep(!(/^\s*$/ || /^\s*\#/),split("\n",$_[0]));
  my($check_functions)= $_[1];

  my ($line,$map,$macro);
  my(@rules,@errors)= ();

  my(@tmp)= @lines; @lines=();
  foreach (@tmp) {
    s/^\s+//; s/\s+$//;
    push(@lines,$_);
  }

  @tmp= @lines; @lines=();
  while ($_= shift(@tmp)) {
    $line= $_;
    while ($line !~ /[\{\}\;]$/)  {
      $line.= ' ' . shift(@tmp);
    }
    push(@lines,$line);
  }
  
  # @lines all now start with a non-whitespace and end in ';', '{', or
  # '}'
  
  $_=shift(@lines);
  /^ruleset\s+(\w+)/ || push(@errors,"ruleset name not on first line \"$_\"");
  
  $_=shift(@lines);
   /^buffer\s+(\d+)/ || push(@errors,"buffer time not specified at \"$_\"");
  
  $_=shift(@lines);
  /^timeout\s+(\d+)/ || push(@errors,"graph timeout not specified at \"$_\"");
  
  # find all macros
  $_=shift(@lines);
  if (/^macros/) {
    my(%macros)= ();
    while (($_= shift(@lines)) ne '}') {
      s/;$//;
      my($name,$val)= split(/=\s*/,$_);
      $macros{$name}= $val;
    }
    
    # expand out all macros
    foreach $macro (keys %macros) {
      @_= @lines; @lines= ();
      foreach (@_) {
	s/\b$macro\b/$macros{$macro}/g;
	push(@lines,$_);
      }
    }
  } else {
    push(@errors,"macro section expected at \"$_\"");
  }

  $_=shift(@lines);
  if (/^attribute\s+declarations/) {
    (undef,@lines)=&grab_rule_block(@lines);
    # we should really make sure decls were okay
  } else {
    push(@errors,"attribute declarations section expected at \"$_\"");
  } 

  $_=shift(@lines);
  if (/^node\s+precondition (.*)/) {
    push(@rules,$1);
  } else {
    push(@errors,"node precondition expected at \"$_\"");
  }

  $_=shift(@lines);
  if (/^edge\s+precondition (.*)/) {
    push(@rules,$1);
  } else {
    push(@errors,"edge precondition expected at \"$_\"");
  }

  my($rulesref,$name);
  foreach $name ('global','node','edge') {
    $_=shift(@lines);
    if (/^report\s+$name\s+rules/) {
      ($rulesref,@lines)= &grab_rule_block(@lines);
      push(@rules,@{$rulesref});
    } else {
      push(@errors,"report $name rules section expected at \"$_\"");
    }
  }

  foreach $name ('global','node','edge') {
    $_=shift(@lines);
    if (/^$name\s+rules/) {
      ($rulesref,@lines)= &grab_rule_block(@lines);
      push(@rules,@{$rulesref});
    } else {
      push(@errors,"$name rules section expected at \"$_\"");
    }
  }
  
  $_=shift(@lines);
  if (/^assessments/) {
    while (($_= shift(@lines)) ne '}') {
      my($test,$actions)= split(/\s*==>\s*/,$_);
      push(@rules,$test);
      if (defined($actions) && $actions) {
	($actions,$map)= &encode_quotes($actions);
	push(@rules,map(&restore_quotes($_,$map),&top_level_pieces($actions)));
      } else {
	push(@errors,"action section expected in assessment rule \"$_\"");
      }
    }
  } else {
    push(@errors,"assessment section expected at \"$_\"");  
  }

  @lines && push(@errors,"leftover lines: ".join("\n",@lines));
  
  my($rule);
  foreach $rule (@rules) {
    push(@errors,map($_." in rule: $rule",&check_rule($rule,0,$check_functions)));
  }

  return @errors;
}

# check_query checks some query text and returns a list of any errors
# encountered. If an optional second argument is given and is true,
# then the names of functions called are checked against those that
# have been set up in the Function package; if this hasn't been set
# up, then this option can't be used.
#
# check_query: query_text [x boolean] -> {error_text}
#
# e.g. @errors= &check_query($query_text);
#      @errors= &check_query($query_text,1);
#
sub check_query {
  my(@tests)= grep($_,split(/\s*;\s*/,$_[0]));
  my($test);
  my(@errors)= ();
  foreach $test (@tests) {
    push(@errors,map($_." in test: $test",&check_rule($test,1,$_[1])));
  }
  return @errors;
}

# check_rule checks a single <action> or <assignment> portion of a
# ruleset rule (or query) and returns a list of errors encountered.
# The text should be macro-expanded and trailing ';''s are considered
# optional.  If an optional second argument is given and its value is
# true, then assignments are not considered valid operations.  If an
# optional third argument is given and is true, then the names of
# functions called are checked against those that have been set up in
# the Function package; if this hasn't been set up, then this option
# can't be used.
#
# check_rule: rule [x boolean [x boolean]] -> {error_text}
#
# e.g. @errors= &check_rule($rule);
#
sub check_rule {
  my($rule)= $_= shift;
  my($no_assign)= shift;
  my($check_functions)= shift;
  my(@errors)= ();

  # encode quotes, regexs, and remove trailing ';'
  my($qmap);
  my($bsechar,$qechar); # the encoding chars used by encode_quotes
                        # for the backslashed chars and the double
                        # quote
  ($_,$qmap)= &encode_quotes($_);
  (undef,$bsechar,$qechar)= @{$qmap};
  my($rmap);
  my($rechar); # the encoding chars used by encode_quotes slash chars
  ($_,$rmap)= &encode_quotes($_,'/');
  (undef,undef,$rechar)= @{$rmap};
  #  s/\s+//g;
  s/;$//;

  $_= " $_ "; #make sure padded with spaces
  
  # convert alphabetic operands to \x80
  s/([^\.])\b(eq|ne|gt|lt)\b/$1\x80/g;

  # check attributes and replace with \x81
  while (s/($qechar\d+$qechar)\.\w+(\.\w+|)\b(\s*$|\s*[^\s\.\(])/\x81$3/g) {}; # from a query
  while (s/\b([a-zA-Z]\w*(\.\w+(\.\w+(\.\w+(\.\w+(\.\w+|)|)|)|)|))\b(\s*$|\s*[^\s\.\(])/\x81$7/g) {};

  # check regex patterns and strings and replace with \x82
  while (s/($qechar|$rechar)\d+($qechar|$rechar)/\x82/) {};
  
  # check numeric constants and replace with \x82
  s/\b\d+(\.\d+|)\b/\x82/g;

  # check assignment (optional) & remove LHS
#  my($tmp)= $_;
#  $tmp =~ s/([\x80-\xFF])/'\x'.sprintf('%2X',ord($1))/eg;
#  print "$tmp from $rule\n";
  if (s/^\s*([^=]*?)\s*=([^~=].*)/$2/) {
    if (defined($no_assign) && $no_assign) {
      push(@errors,"assignment is prohibited in this expression");
    } else {
      push(@errors,"invalid assignment target in $rule") if $1 ne "\x81";
    }
  }
  
  # what's left fits the description of an <expr>
  my(@exprs)= ($_); # a list of exprs to be checked
  while (@exprs) {
    my($expr)= $_= shift(@exprs); # the expr to check this iteration
    s/^\s+//; s/\s+$//;

    if (/^$/) {
      push(@errors,"missing expression");
      next;
    }

    # divide up along binary operations not in parens
    # my($pat)=join('|',map(quotemeta($_),qw(** =~ == :: && || <= >= < > + - * /)),'\x80');
    my($pat)= '\*\*|\=\~|\=\=|\:\:|\&\&|\|\||\<\=|\>\=|\<|\>|\+|\-|\*|\/|\x80'; # same as above
    my(@operands)= &top_level_pieces($_,$pat);
    if (@operands >= 2) {
      unshift(@exprs,@operands);
      next;
    }

    # $expr is show to user in errors so clean it up
    $expr =~ s/\x81/\<attr\>/g; $expr =~ s/\x82/\<constant>/g;
 
    # check unary ops and remove
    s/^!\s*//g;

    if (/^[\x82\x81]$/) { # case statement on other forms of <expr>
      # check attr, regex (reduced), or constant; done if okay
    } elsif (s/^\(\s*//) { # shave of balanced parens & report if not balanced
      unless (s/\s*\)$//) {
	push(@errors,"unbalanced parenthesis in $expr");
	next;
      }
      unshift(@exprs,$_);
    } elsif (s/^\{\s*//) { # check set constructor and add inside exprs to @exprs
      unless (s/\s*\}$//) {
	push(@errors,"unbalanced set constructor in $expr");
	next;
      }
      unless (/^\s*$/) {
	unshift(@exprs,&top_level_pieces($_));
      }
    } elsif (s/^\[\s*//) { # check list constructor and add inside exprs to @exprs
      unless (s/\s*\]$//) {
	push(@errors,"unbalanced list constructor in $expr");
	next;
      }
      unless (/^\s*$/) {
	unshift(@exprs,&top_level_pieces($_));
      }
    } elsif (/^([\w\.]+|[\w\.]+\.[\w\s\.]+[\w.])\s*\((.*)\)$/) { # check function call
      my ($name,$args)= ($1,$2);
      push(@errors,"no such function \"$name\" defined in $expr")
	unless (!defined($check_functions) || !$check_functions || $name eq "defined" || $name eq "time" || Function->named($name));

      if ($args !~ /^\s*$/) {
	my(@args)= &top_level_pieces($args);
	unshift(@exprs,@args);
      }

      # TODO: check type formal arguments against actuals.
      # QUESTION: How do we determine the type of the actuals, esp. from queries?
    } else { # a syntax error
      push(@errors,"bad expression \"$expr\"");
    }
  }
  return @errors;
}

1;
