#!/pkg/bin/perl -w
#
# This is a library that handles communications of control variables
# for a data source.  It provides only the bare minimum of functionality
# in terms of storing variables, receiving commands, and updating the
# status.  All interesting operations must be handled by the user code.
#
# There are two kinds of control variables:  Indexed and straight.
# Straight variables have just a name, and a single value associated
# with that name.  Indexed control variables have a name and an index,
# both of which are strings.  Each name and index pair has a value.
#
# A set of control variables is stored as an associative array.  Each
# straight control variable is represented in the AA as simply its name
# mapping to a value.  The name of each indexed variable maps to a
# reference to an anonymous AA.  This second-level associative array
# contains a mapping of indices to their values.
#
# As a side effect of this method of implementation, a
# straight variable and an indexed variable may not have the same name
# for any given set of control variables.
#
# This class adds some functionality to the plain associative array,
# as follows:
#
# The value "__INFO__" of the AA is a reference to another associative
# array that contains information about this instantiation of the object.
# The user code does not have to worry about the contents here.  They
# are documented at the end of this comment block so that this code can
# be maintained.  In any case, don't name any control variables "__INFO__".
#
#
# STARTING UP
#
# This bit describes how to start up a GrIDS module.
#
# To use this library, a GrIDS module must follow certain standards.
# This library is purposely set up to be simple, so the module needs
# to do most of the work.  A wrapper library may be written later to
# do the work that most modules have to do.  The standards are as
# follows:
#
# It must not use the USR1 signal in any way.
#
# It must include the line:  use control_vars;
#
# It must establish itself as a module as it starts up.  This is done
# as follows:
#
# Instantiate an instance of this class, like this:
#
# vars = new control_vars("com_files_prefix","init_file_name");
#
# The names of the command and status files (constructed from
# "com_files_prefix") are known by the module controller.
# The module controller will pass that prefix, and an optional
# "init_file_name", to the data source module as command-line args.
# Hence, this package will use those args as defaults, unless the
# user explicitly over-rides those values.
# This new call must be made exactly once by any GrIDS
# module.  Making more than one of these object has undefined
# consequences.
#
# After creating a control_vars structure, call add_straight() and
# add_indexed() to establish the set of legal control variables.
#
# Treating the control_vars structure as described early in this
# comment block, set the control variables to appropriate initial
# values.  If they are not specifically set, the default values
# will be empty strings.
#
# If the instantiation is called "vars", the straight variables may
# be referred to as vars->{'varname'} and the indexed variables may
# be referred to as vars->{'varname'}{'varindex'}.
#
# In addition to the control variables established specifically
# for the module, the module should set straight variables named
# 'module', 'version', and 'department'.  The variables are pre-
# approved, and do not need to be established with add_straight().
#
# Call update().  This lets the module controller know that the
# GrIDS module is ready.
#
# The module is now properly registered and connected, and may continue
# with its normal operations.
#
#
# OPERATION
#
# This bit describes what a module must do once it is started up.
#
# Once in operation, the module must do the following:
#
# Update and read the control variables in this structure as appropriate.
#
# Call update() as needed to export the values to the rest of GrIDS.
#
# Periodically call a non-blocking get_command(), or simply call it
# in blocking mode and wait until it completes.
#
# Carry out the command ordered from get_command().  If this is a set,
# that will probably involve setting control variables and calling
# update().  The module writer is entirely responsible for interpreting
# and responding to the commands.  This class does not understand the
# semantics of the commands at all.  If a value of an indexed variable
# is set to the empty string, that index should be removed entirely.
#
# When finished executing the command, call complete_command().  The
# module has two seconds from the time the controller issues the command
# to actually finish calling complete_command().
#
#
# SHUTTING DOWN
#
# Just call shutdown() before exiting the module.
#
#
# MEMBER FUNCTIONS
#
# new(cmdfile_prefix, init_file_name)
#     Establish a new object.  This sets up a "connection" with the
#     module controller.  The first argument is used to construct the names
#     of the command and status files for this module.  The second argument
#     is assumed to be a pathname of an initialization file.  Both args
#     are optional.  If not provided, they are assumed to be in the
#     command-line args with which the module controller exec'ed the
#     data source module.
#
# add_straight(var1, var2, ....)
#     All of the names will be established as legal straight control
#     variables.  This may be done more than once, in which case
#     the new names will be added to the previous ones.  Names which
#     already exist (either straight or indexed or just with a value
#     set somehow) will be silently ignored.
#
# add_indexed(var1, var2, .....)
#     As for add_straight(), but for indexed variables.
#
# update()
#     Update the status file.  This will actually create the status
#     file.  It fills it in with the contents of the current control
#     variables.  Only control variables mentioned in an add_straight()
#     or add_indexed() are updated.
#
# get_command(block)
#     This reads the contents of the command file and returns them
#     as a reference to an ordinary control variable associative array.
#
#     If "block" is a true value, this call will block and wait until
#     such time as the module controller sends it a command.  Otherwise,
#     it will return immediately if no command is pending.  If it returns
#     without a command, it will return the undefined value.
#
#     Since this library is purposely avoiding using sophisticated
#     data structures in order to make everything an associative array,
#     the contents of the command are not distinguishable from ordinary
#     control variables, except by their names, as follows:
#
#     command:  Either "get" or "set".
#
#     module:  The name of the module.
#
#     department:  The department to which the module belongs.
#
#     version:  The version of this module.
#
#     The module programmer must deal with the structure as indicated,
#     and must avoid setting these variables, even though they may
#     arrive as part of a set command.
#
# do_set(*cmd, *vars)
#     Performs simple assignment to *independent* control $vars,
#     as specified by a SET $cmd.  If the values of control vars
#     are *coupled*, or if changing their values should cause certain
#     side-effects, then this simple function is no longer adequate,
#     and data source writers should use this function's code as a template
#     to write their own, custom routine.
#
# complete_command()
#     This indicates to the module controller that the module is finished
#     with the last command.  The actual implementation of this is that
#     it unlinks the command file.  This may change, so this is the
#     best way to do this.
#
# shutdown()
#     This removes the status file and thereby lets the module controller
#     know that this module is no longer operational.
# 
#
# INTERNALS
#
# The following information is provided for code maintainers, and
# can be ignored by anyone writing a module.
#
# The elements of the __INFO__ associative array are as follows:
#
# straight:  a reference to an AA.  The keys of this AA are the names
#     of the approved straight control variables.  The values associated
#     with each key are ignored.
#
# indexed: as with "straight", but for the indexed control variables.
#
# cmdfilename: name of the command file.
#
# statfilename: name of the status/response file.
#
# initfilename: name of the initialization file.
#

package control_vars;

# This subroutine is called when a USR1 signal is called.  It
# indicates that there is a command pending.
sub sighandler {
    $pending=1;
    print "\n\n **** SIGHANDLER called for PID [$$]\n"
      if $main::debug{'proc'}; #RC
}

# This makes a new control variable structure, establishes the
# file names, and indicates that there are no commands pending.
sub new {
    my $type = shift;
    my $self = {};
    my (@prgname) = split('/', $0);
    my $prgname = $prgname[$#prgname];


    $clog_failures = 0;   # for counting when to quit searching for Clog.

    $pending = 0;
    $SIG{'USR1'}=\&sighandler;

    if ($_[1]) {               # Use provided arguments, if supplied
        $initfile = "$_[1]";
    }
    elsif ($#ARGV >= 1) {      # Use default command-line arguments
        $initfile = $ARGV[1];
    }
    else {
        $initfile = $prgname.'.init';
    }

    $self = &initial_set($initfile) unless
        ($initfile eq "/dev/null"  ||  $initfile eq "" || ! defined $initfile);

    if ($_[0]) {               # Use provided arguments, if supplied
        $self->{'__INFO__'}{'cmdfilename'} = "$_[0].cmd";
        $self->{'__INFO__'}{'statfilename'}= "$_[0].status";
    }
    elsif ($#ARGV >= 0) {      # Use default command-line arguments
        $self->{'__INFO__'}{'cmdfilename'}=$ARGV[0].'.cmd';
        $self->{'__INFO__'}{'statfilename'}=$ARGV[0].'.status';
    }
    else {
        $self->{'__INFO__'}{'cmdfilename'}=$prgname.'.cmd';
        $self->{'__INFO__'}{'statfilename'}=$prgname.'.status';
        $clog = &find_clog();
        if ($clog) {
          $clog->warn ("Module forced to use [$prgname] as prefix " .
                       "for cmd/status files.");
          }
        else {
          # Might not be declared YET.
          print STDERR "Module forced to use [$prgname] as prefix " .
                       "for cmd/status files.";
          }
    }

    # Can't call method "add_straight" on unblessed reference,
    # so bless it first.
    bless $self, $type;

    $self->add_straight ('command', 'module', 'version', 'department',
                         # ? SHOULD I INCLUDE ===>  'listen_tcp', 'listen_udp',
                         'parent_aggregator', 'parent_manager', 'timestamp');

    if ($ENV{'GRIDS_STARTER'} && $ENV{'GRIDS_STARTER'} eq "module_controller") {
        $self->{'department'} = $ARGV[2] if $ARGV[2] && $ARGV[2] ne "''";
        $self->{'parent_aggregator'} = $ARGV[3] if $ARGV[3] &&  $ARGV[3] ne ":";
        }  # RC 8/6/96:  null ARGS should not supersede init_file defaults

    # Constants for flock call.
#   $LOCK_SH = 1
    $LOCK_EX = 2;
#   $LOCK_NB = 4;
    $LOCK_UN = 8;

    return $self;
}

# RC 7/9/96:  problematic to initialize control vars *before* user
#             defines their legal names!  But it is convenient and still safe.
sub initial_set {
    # Read the init file to set initial values
    open(CMDFILE, $_[0]) || return {};
    my $packetstr = '';
    read(CMDFILE, $packetstr, 999999);
    close(CMDFILE);

    # Return the command.
    return decode($packetstr);
}



# This adds the indicated variables names to the "straight" list and
# RC 7/9/96:  But it no longer ...
# establishes them as empty.
sub add_straight {
    my $self = shift;

    foreach $varname (@_) {
	if (# !exists($self->{$varname}) && 
            !exists($self->{'__INFO__'}{'straight'}{$varname}) &&
            !exists($self->{'__INFO__'}{'indexed'}{$varname})) {
            $self->{'__INFO__'}{'straight'}{$varname}='';
	    $self->{$varname} = '' unless $self->{$varname};
	    # RC 7/9/96: Allow us to officially add a var *after*
	    #            its value was set via an init file.
        }
    }
}

# This adds the indicated variables names to the "indexed" list and
# RC 7/9/96:  But it no longer ...
# establishes them as empty.
sub add_indexed {
    my $self = shift;

    foreach $varname (@_) {
	if (# !exists($self->{$varname}) && 
            !exists($self->{'__INFO__'}{'straight'}{$varname}) &&
            !exists($self->{'__INFO__'}{'indexed'}{$varname})) {
            $self->{'__INFO__'}{'indexed'}{$varname}='';
	    $self->{$varname} = {} unless $self->{$varname};
	    # RC 7/9/96: Allow us to officially add a var *after*
	    #            its value was set via an init file.
        }
    }
}

# This updates the status file
sub update {
    my $self = shift;

    # Update the timestamp;
    $self->{'timestamp'}=time();

#####  RC 7/30/96 so init_set gives error info like post_set:
    # We assume by the time a user calls update(),
    # they have finished all declarations via add_straight() or add_indexed().
    # Hence, unless module-coder screwed up and misprinted a control var,
    # this block should catch only errors caused by Initialization SETS.

  my $undefined_vars = '';

  foreach $item (keys %{$self}) {
    next if $item eq '__INFO__';
    next if  defined $self->{'__INFO__'}{'straight'}{$item} ||
             defined $self->{'__INFO__'}{'indexed'}{$item};
    # else it's not a legal target var!
    $undefined_vars .= " [$item]";
    delete $self->{$item};
    }

  if ($undefined_vars) {
    $undefined_vars = "WARNING- Attempted SET on UNDEFINED control var: "
                    . "$undefined_vars!";
    $clog = &find_clog() unless $clog;
    ref $clog  &&  $clog->warn ("\n update: $undefined_vars \n");
    if ($self->{'command'} =~ /^SET|GET|OK$/i) {
      $self->{'command'} = "$undefined_vars";
      }
    else {
      $self->{'command'} = "$undefined_vars" . $self->{'command'};
      }
    }
  else {
    $self->{'command'} = 'OK' if $self->{'command'} =~ /^SET$|^GET$|^$/i;
    }
#####

    # Make a packet
    my $packet = $self->encode();

    # Open the status file
    open(STATFILE, '>'.$self->{'__INFO__'}{'statfilename'})
                || die "Can't open status file.\n";


    # Lock the file
    flock(STATFILE,$LOCK_EX);

    # Store the packet in the file
    print STATFILE $packet;

    # Unlock it
    flock(STATFILE,$LOCK_UN);
    # Close the file
    close(STATFILE);

}

sub get_command {
    my $self=shift;
    my $block = shift;

    # Wait until data comes in.  Re-check to avoid race condition problem.
    while ($block && !$pending) {
        sleep 1;
    }

    # If we weren't blocking, return immediately.
    if (!$pending) {
        return {};      # RC 8/20/96:  modules expect a ref to a hash.
    }

    # Reset the "data pending" flag
    $pending=0;

    # Read the command file and get the packet
    open(CMDFILE, $self->{'__INFO__'}{'cmdfilename'}) || return 0;
    my $packetstr = '';
    read(CMDFILE, $packetstr, 999999);
    close(CMDFILE);

    # Return the command.
    return decode($packetstr);
}


###############################################################################
#  Usage of simple control_var assignment function &do_set() ::
#
#  $vars = control_vars->new ();
#  $cmd = $vars->get_command ($block);
#
#  &do_set (*cmd, *vars) if $cmd->{'command'} =~ /SET/i;     # or
#  &do_set (\$cmd, \$vars) if $cmd->{'command'} =~ /SET/i;
#
#  # Write changes to our status file:
#  $vars->update();
#
#  # see data.source.pl for complete template code of a dumb data source.
#
#
# Replaced orig. glob-hack invocation mode, since that can't handle <my> vars.
# It's backward compatible, ie, (at least for globals) you can still invoke as:
#
#    &do_set (*src_obj, *c_vars);
#
# But to handle <my> vars too, invoke using ref to a hash $src_obj
#                                      and a control_vars $dest_obj:
#
#    &do_set (\$src_obj, \$c_vars);
#

sub do_set {
  shift if ($#_ eq 2);    # ignore parm if called as $self
  my $source = ${$_[0]};
  my $dest = ${$_[1]};

  my $undefined_vars = "";
  $clog = &find_clog() unless $clog;
 
  return if $$source{'command'} =~ /^GET/i;
  $$dest{'command'} = '' if $$dest{'command'} =~ /^OK$|^SET$/i;
  delete $$source{'command'} if $$source{'command'} =~ /^OK$|^SET$/i;
  my $errmsg = '';
  if ($$source{'module'} ne $$dest{'module'}) {
    $errmsg = "SET command cannot change module name!  (" . $$source{'module'}
            . ") NOT EQUAL (" . $$dest{'module'} . ")\n";
    ref $clog  &&  $clog->warn ("do_set WARN: $errmsg");
    return;
    }
          #  $$source->{'version'} eq $$dest->{'version'}  ??
 
  if ($$source{'department'} ne $$dest{'department'}) {
    $errmsg = "Changing DEPT for [" . $$dest{'module'} . "] from [" .
         $$dest{'department'} . "] to [" . $$source{'department'} . "] \n";
    $$dest{'command'} = "OK- $errmsg";
    $clog->warn ("do_set OK: $errmsg") if ref $clog;
    }
 
  # assign both straight and indexed variables
  foreach $item (keys %{$source}) {
    next if $item eq '__INFO__';  # prohibit stealth mods to our "symbol table"
    unless ( (exists $$dest{'__INFO__'}{'straight'}{$item})  ||
             (exists $$dest{'__INFO__'}{'indexed'}{$item}) ) {
      # was: (defined $$dest{$item})
      # It's not a legal target var!
      $undefined_vars .= " [$item]";
      next;
      }
     unless ( ref ($$source{$item}) ) {       # ie, it must be a straight var.
      $$dest{$item} = $$source{$item};
      next;
      }
    if ( ref ($$source{$item}) eq "HASH" ) { # should not be any *other* ref!
      foreach $index (keys %{$$source{$item}}) {
        $$dest{$item}{$index} = $$source{$item}{$index};
        }
      }
    }
  if ($undefined_vars) {
    $undefined_vars = "WARNING- Attempted SET on UNDEFINED control var: "
                    . "$undefined_vars!";
    $clog->warn ("\n do_set: $undefined_vars \n") if ref $clog;
    $$dest{'command'} = "$undefined_vars  " . $$dest{'command'};
    }
  else {
    $$dest{'command'} = "OK";
    }
  }

  # fun with perl syntax equivalents ::
  # (keys %{$source->{$item}})  <== same as ==>  (keys %{$$source{$item}})
  # ($$dest{'foo'})  ==  ($main::mc_vars->{'foo'})  ==  ($$main::mc_vars{'foo'})


###############################################################################

# RC 7/17/96 - since user may not always declare same CLOG.pm name, "my_log"
sub find_clog {
  return $main::my_log
     if defined $main::my_log && (ref $main::my_log eq "Clog");
  foreach $key (keys %main::) {
    next if $key =~ /\W/;  # avoid eval of files, pkgs, weird stuff
    my $res = eval "\$main::$key";
    if (ref $res  &&  ref $res eq "Clog") {
      return $res;
      }
    }
  return 'No Clog' if ++$clog_failures > 5;
  return '';
  }      #  at init time, user may not yet have invoked Clog.
         #  beyond that, we search 5 times, then give up on Clog.








# This just removes the command file to indicate that the
# desired command has been completed.
sub complete_command {
    my $self=shift;

    unlink($self->{'__INFO__'}{'cmdfilename'});
}

# This just removes the stat file to indicate that the module
# is done.
sub shutdown {
    my $self=shift;

    unlink($self->{'__INFO__'}{'statfilename'});
}

# This takes the current control variable information and
# generates a packet, which it returns as a string.  It is
# in this separate subroutine for readability, and is mainly
# just used by update().
sub encode {              # RC 5/23/96: modified for gcpf file format
    my $self = shift;
    my $packet = '';
    my $varname;
    my $index;
    my $codevar;

    # encode the command, module name, and version
    $packet.=$self->{'command'}."\xff\n";
    $packet.=$self->{'module'}."\xff\n";
    $packet.=$self->{'version'}."\xff\n";
    $packet.=$self->{'department'}."\xff\n";

    $packet .= 'timestamp' . "\xff\n" . $self->{'timestamp'} . "\xff\n";  #RC
    
#RC deleted calls to  &encode_pound();

    # encode all the straight variables
    foreach $varname (keys %{$self->{'__INFO__'}{'straight'}} ) {
	next if $varname eq 'timestamp';     # RC; for repeatable position
	$packet.=$varname."\xff\n".
                 $self->{$varname}."\xff\n";
    }

    # encode all of the indexed variables
    foreach $varname (keys %{$self->{'__INFO__'}{'indexed'}}) {
	foreach $index (keys %{$self->{$varname}}) {
            $packet.=$varname.'{'.$index.'}'."\xff\n".
		     $self->{$varname}{$index}."\xff\n";
	}
	unless (keys %{$self->{$varname}}) {
            # 8/20/96: display all declared index vars, even if empty hash
            $packet .= "$varname\{\}\xff\n\xff\n";
	}
    }

    $packet.="\xff\n";

    return $packet;
}

# This takes a packet and decodes it into an associative array.
# It returns a reference to the array.
sub decode {              # RC 5/23/96: modified for gcpf file format
    my $array = {};
    my $inputstr = shift;
    my $name;
    my $value;

# Parse the packet
    my @fields = split("\xff\n",$inputstr);

# Load the special fields
    $array->{'command'}= shift @fields;
    $array->{'module'}= shift @fields;
    $array->{'version'}= shift @fields;
    $array->{'department'}= shift @fields;

#RC deleted calls to  &decode_pound();

# Load the regular control variables
    while (@fields) {
        $name= shift @fields;
	$value= shift @fields;

	# Check if it is an indexed name
        # RC changed so indexing string can contain any char except `}'.
        # Also, I strip off optional enclosing single or double quotes
        # so we only store a bare string as "canonical form".
	if ($name =~ /^(\w+)\s*\{['"]*([^\}'"]+)['"]*\}$/) {
	    $array->{$1}{$2} = $value;
	}
	else {
	    $array->{$name} = $value;
	}
    }

    return $array;
}



1;
