#!/pkg/bin/perl

# $Id: select_graph_help.pl,v 1.5 1998/01/26 20:15:40 hoagland Exp $

# This file contains various functions to help in the selection of graphs.

use Comm;
require 'parse_dot.pl';

sub BEGIN {
  Comm::init();
}

sub END {
  Comm::shutdown();
}

# Given a function reference, a destination file, and a list of input
# files, this function takes a stream of graphs (in reports, aggregated
# reports, query results, and alerts) in Comm packet format from the
# input files and outputs (in Comm packet format, to the given
# destination) the ones that succeed when the function is called with
# the graph header and the graph
sub output_selected_graphs {
  my($selfnref,$out_file,@in_files)= @_;
  my($gzip_out)= $out_file =~ /\.gz$/;
  my($gzip_comm)= "gzip - > $out_file";

  foreach $file (@in_files) {
    my($fref)= &open_graph_stream($file);

    ($proto,$host,$port,$header,@body)=&get_packet($fref);
    while (defined($proto)) { # until end of file
      $selected= 0;
      if ($header eq 'r' || $header eq 'a') {
	$selected= &{$selfnref}($header,$body[0]);
      } elsif ($header eq 'alt') {
	$selected= &{$selfnref}($header,$body[2]);
      } elsif ($header eq 'qr') {
	foreach (&parse_dot::get_graphs($body[0])) {
	  push(@graphs,$_) if &{$selfnref}($header,$_);
	}
	$body[0]= "digraph \"composite\" {\n  ".join("\n  ",@graphs)."\n}\n";
	$selected= @graphs;
      }
      if ($selected) {
	if ($gzip_out) {
	  &Comm::prog_send($gzip_comm,$header,@body);
	} else {
	  &Comm::file_send($out_file,$header,@body);
	}
      }
      ($proto,$host,$port,$header,@body)=&Comm::mesg_recv('nonblocking');
      #print "got:".join(',',$proto,$host,$port,$header,@body),"\n";
    }
    &close_graph_stream($fref);
  }
  if ($gzip_out) {
    &Comm::prog_close($gzip_comm);
  } else {
    &Comm::file_close($out_file);
  }
}

# This function returns particular global attribute values.  The graph is
# the first argument, the attribute names is the remaining arguments.  The
# values are returned in a list in the same order as the names.  If a
# attribute is found in the graph, it is returned in 'attr.pl''s format
# else undef is returned for it.
sub get_global_attr_val {
  my($graph,@attrnames)= @_;
  my(@out)=();
  my($name,$nodesref,$edgesref,@attrs)= &parse_dot::graph_parts($graph);
  ATTRNAME: foreach $attrname (@attrnames) {
    foreach (@attrs) {
      my($name,$val)= &parse_dot::split_attr($_);
      if ($name eq $attrname) {
	push(@out,&unflatten_attr($val));
	next ATTRNAME;
      }
    }
    push(@out,undef);
  }
  return @out;
}

# This function returns a particular edge attribute value for all edges
# in a graph.  The graph is the first argument, the attribute name is the
# second argument.  The values are returned in a list.  If a attribute is
# found on an edge, it is returned in 'attr.pl''s format else undef is
# returned for it.
sub get_edges_attr_val {
  my($graph,$attrname)= @_;
  my(@out)=();
  my($name,$nodesref,$edgesref,@attrs)= &parse_dot::graph_parts($graph);
  foreach $edge (@{$edgesref}) {
    my($src,$dest,@attrs)= &parse_dot::edge_parts($edge);
    foreach (@attrs) {
      my($name,$val)= &parse_dot::split_attr($_);
      if ($name eq $attrname) {
	push(@out,&unflatten_attr($val));
	last;
      }
    }
    push(@out,undef);
  }
  return @out;
}

# Given a list of input files, this function takes a stream of graphs (in
# reports, aggregated reports, query results, and alerts) in Comm packet
# format from the input files and returns them
sub get_graphs {
  my(@in_files)= @_;
  
  my(@graphs)=();
  my($graph,$file);
  foreach $file (@in_files) {
    $fref= &open_graph_stream($file);
    while (defined($graph=&get_next_graph($fref))) {
      push(@graphs,$graph);
    }
    &close_graph_stream($fref);
  }
  return @graphs;
}

###################

# opens a graph stream for subsequent reading by &get_next_graph; gunzips
# the indicated file if needed and remebers whether it did.  Returns a
# a file id for subsequent references to the file.
sub open_graph_stream {
  my($file)=shift;
  my($gzipped)= ($file =~ s/\.gz$//);
  `gunzip -c $file.gz > $file` if $gzipped;
  $GZIPPED{$file}= $gzipped;
  $GRAPHS{$file}= [];
  Comm::read_file($file);
  return $file;
}

# Given a file id (returned by &open_graph_stream), this routine closes a
# graph stream.  gzips the file if it was gziped before.
sub close_graph_stream {
  my($file)=shift;
  Comm::file_close($file);
  Comm::close_iterate_file($file);
  delete $GRAPHS{$file};
  unlink($file) if $GZIPPED{$file};
  delete $GZIPPED{$file};
}

# Given a file id (returned by &open_graph_stream), this routine returns the
# next graph from a graph stream.  Returns undef if there are no more graphs.
sub get_next_graph {
  my($file)=shift;
  return shift(@{$GRAPHS{$file}}) if @{$GRAPHS{$file}};

  my($proto,$host,$port,$header,@body)=&get_packet($file);
  return undef unless defined($proto); # at end
  if ($header eq 'r' || $header eq 'a') {
    return $body[0];
  } elsif ($header eq 'alt') {
    return $body[2];
  } elsif ($header eq 'qr') {
    $GRAPHS{$file}= [&parse_dot::get_graphs($body[0])];
    return shift(@{$GRAPHS{$file}});
  }
}

#############################
# Internal use only, probably

sub get_packet {
  my($file)= shift;
  my($proto,$host,$port,$header,@body);
  do {
    ($proto,$host,$port,$header,@body)=&Comm::mesg_recv('nonblocking',undef,undef,undef,$file);
    #print "got:".join(',',$proto,$host,$port,$header,@body),"\n";
    return (undef) unless defined($header);
    $proto eq 'file' || warn "That's strange; recieved a packet from a non-file:".join(',',$proto,$host,$port,$header,@body);
  } until ($proto eq 'file');
  return ($proto,$host,$port,$header,@body);
}

1;
