#!/pkg/bin/perl -w

# $Id: Network.pm,v 1.43 1998/02/17 19:39:57 rowe Exp $
use Comm;
use Buffer;
use control_vars;
require "log.pl";
require "query.pl";

package Network;

# override die and warn to use the main package's
use subs qw(die warn);
sub die {
  &::die(@_);
}
sub warn {
  &::warn(@_);
}

sub BEGIN {
  $Network::last_header= 'none';
}

# Assumptions:
#   Assumes Comm::init has been run.
#
#   Assumes one argument.  If 'blocking', then
#   this function will not return until a message
#   is available.  If 'nonblocking', will return
#   immediately if nothing is available.
#
# Receives a tcp or udp message and calls the
# appropriate function to handle the message
# depending on the message header.
#
# Checks control variables upon each call and
# deals with any changes to them.
#
sub receive {
  # look for changes to control vars
  &handle_control_vars;

  # get something from the network
  my($block_time)= $::fasttime ? 0 : 0.2;
  my($proto,$source_host,$src_port,$header,@body) = 
    &Comm::mesg_recv($block_time,undef,undef,undef,undef);

  if (defined($header)) {
    $::DEBUG && warn "got mess (".join(',',map(defined($_)?$_:'(undef)',$proto,$source_host,$src_port,$header,@body)).") from network\n";# unless $header eq 'r';
    
    # Is it a query?  If so, handle it
    if($header eq 'q') {
      $::DEBUG && warn "**************************** QUERY RECEIVED ************************************\n*$body[0] from $source_host:$src_port\n";

      if($proto eq 'file') {
        print "*************QUERY WAS FROM FILE*************\n";
        print &::do_query($body[0]);
      }
      if($proto eq 'udp') {
	&Comm::udp_send($source_host,$src_port,'qr',&::do_query($body[0])); 
      } elsif($proto eq 'tcp') {
	&Comm::tcp_send($source_host,$src_port,'qr',&::do_query($body[0]));
      }
    }
    
    # is it a graph language report?  If so, handle it
    elsif($header eq 'r') {
      $::DEBUG > 3 && warn "**************************** REPORT RECEIVED *******************\n";

      if($::DEBUG > 2 && $proto eq 'file') {
        print "***************REPORT FROM FILE*****************\n";
        print $body[0]."\n";
        print "*****************END OF REPORT******************\n";
      }
      # assumes for now that the body of a report does not specify the rulesets
      # which the report should be fed to.  Hence, we add it to all of them.
      &got_report($body[0]);
    }
    
    # is it an aggregated report?  If so, handle it
    elsif($header eq 'a') {
      # The first element of @body should be the report, and the rest should
      # be a list of all rulesets (probably just one) to feed this report to.
      &got_report(@body);
    }
    
    # is it garbage
    else {
      print STDERR "Garbage message received by Network.pm: header= $header, body=(",join(',',@body),")\n";
    }

    $Network::last_header= $header;
  } else { # nothing new input
    if ($Network::last_header eq '') { # noting input last time either
      $::simu_time++ if $::fasttime;  # so pretend a second has gone by

      if ($::stop_if_clear) {
	if ($::nobuffer) {
	  &::shutdown;
	} else { # buffers in use
	  $any_buffered=0;
	  foreach $tm ($Buffer::gamma_buffer,values %Buffer::rule_buffers) {
	    print "checking $tm for buffered items\n";
	    if ($tm->how_long) { # returns a positive number if any are buffered
	      print "$tm had buffered items\n";
	      $any_buffered=1;
	      last;
	    }
	  }
	  &::shutdown unless $any_buffered;
	}
      }
    }
    $Network::last_header= '';
  }

  # check for things coming off the buffer
  &Buffer::out_buffer unless $::nobuffer;
}

sub got_report {
  my($report)= shift;
  my(@rss)= map(Ruleset->named($_),@_);

  if ($::fasttime) {
    if ($report =~ /(\btime|\"time\")\s*=\s*"?(\d+)"?/) {
      $::simu_time= $2;
    }
  }

  unless ($::nobuffer) {
    &Buffer::add_buffer($report,@rsnames);
  } else { # no buffering is to be done -- pass directly to rulesets
    @rss= Ruleset->all unless @rss;
    &::add_report(Graph->new('',$report),@rss);
  }
}

# Assumptions:
#   Assumes that this function will be called once, before any
#   calls to &handle_control_vars.
# #
# #   Assumes the command file prefix, init file, version number, department and aggregator are arguments.
# #
# # This function establishes the global variable $::cvars to be the
# # control variable structure used in repetitive calls to
# # &handle_control_vars
# #
# note: die and warn may not be working yet
sub initiate_control_vars {
  my($command_file_prefix,$init_file,$version,$department,$aggregator,
     $listen_tcp, $listen_udp)= @_;    # RC 8/2: added listen_tcp & _udp parms
  $::cvars = new control_vars($command_file_prefix,$init_file);

  $::cvars->add_indexed('rulesets','debug','children','alert_recipients');
  $::cvars->add_straight('listen_udp','listen_tcp','beta','gamma','parent_aggregator');
#  $::DEBUG = 2;   # Changed by J.Rowe to avoid filling up the /tmp directory
                   # with engine messages during long term running. 8/14/97
  $::cvars->add_straight('DEBUG');
  $::cvars->{'DEBUG'} = $::DEBUG; 
#
# During long term evaluation, we might want to keep around larger versions
# of the engine log files.  Adjust the max and min logfile sizes here.
#  JRowe 1/11/98
  $::cvars->add_straight('max_logfile_size');
  $::cvars->add_straight('min_logfile_size');
  $::cvars->{'max_logfile_size'} =  100000;
  $::cvars->{'min_logfile_size'} =  20000;

  # RC 8/16:  When have time, make this indexed on Clog filter names.

  $::cvars->{'module'} = 'engine';
  $::cvars->{'version'} = $version;
  $::cvars->{'department'} = $department
     unless $::cvars->{'department'};   # RC 8/2: maybe already set
  $::cvars->{'parent_aggregator'}= $aggregator
     unless $::cvars->{'parent_aggregator'};   # RC 8/2: maybe already set
  $::cvars->{'listen_tcp'} = $listen_tcp;
  $::cvars->{'listen_udp'} = $listen_udp;

  # RC 8/2:  DEBUG- Should we not update until looking at rulesets?
  #          If so, we could return WARNING if bad syntax, instead of OK:
  #          $::cvars->{'command'} = 'OK';  # RC 8/2: 

  # RC 8/9: Problematic to call update() twice (our caller also does update).
  # Which update will MC get?  If it misses first update and bLOCKs waiting
  # for 2nd update, that could return mistaken idea of hang/timeout error.
  # Presumably that window is tiny compared to time spent processing rulesets.

# $uname = `/usr/bin/uname -a`;        # RC 8/14:  Solaris LOCK BUG workaround
# unless ($uname =~ /SunOS \S+ 5\.\d Generic/) {
#   $::cvars->update; # push the variable values out
#   }   # Let's wait as long as it takes on Solaris!
## RC 12/11/96:  Commented out this *initial* update, because
## race condition can still cause module-controller to read garbled file.
## Solaris and SunOS are now treated equally -- for both, the MC will wait
## (until possible timeout) for engine to digest its initial rulesets.)



  # some variables may be set up already; we need to set up other parts of
  # the engine to reflect this

  # set up rulesets if any specified; assuming these are non-null
  &::new_rulesets(%{$::cvars->{'rulesets'}});

  # set up any specified debug flags
  foreach (keys %{$::cvars->{'debug'}}) {
    print "init control var; changing debug flag on $_\n";
    if (Ruleset->named($_)) {
      Ruleset->named($_)->set_debug($command->{'debug'}{$_});
    } else {
      print STDERR "attempt to set debug flag to $command->{'debug'}{$_} on unknown ruleset $_\n";
    }
  }

  # internally, we assume that all keys of the conceptual sets control
  # variables are members of the set, so filter out false values
  my($cv);
  foreach $cv (qw(children alert_recipients)) {   # RC 8/2 spellcheck
    foreach (keys %{$::cvars->{$cv}}) {
      delete $::cvars->{$cv}{$_} unless $::cvars->{$cv}{$_} && $::cvars->{$cv}{$_} !~ /^false$/i;
    }
  }
}

# Assumptions:
#   Assumes $::cvars is the control variables structure.
#   (This should be set up in &initiate_control_vars.)
#
#   Assumes no arguments
#
# This function gets any control messages and services them.
# It does warns for control messages it doesn't recognize. 
#
sub handle_control_vars {
  my $command;

  # can simulate getting control vars the real way by using the arguments as
  # control vars
  if (@_) {
    $command= {@_};
    $command->{'command'}= 'set' unless defined($command->{'command'});
  } else {
    $command = $::cvars->get_command(undef);
    # is $command is a valid command structure?
    return undef unless $command && defined($command->{'command'}) && $command->{'command'} ne '';
  }

  warn "engine got a ".$command->{'command'}." control variable command";
  
  # is this a set command?
  if($command->{'command'} =~ /^set$/i) {

    my %set=();
    foreach (keys %{$command}) {
      $set{$_}= 1;
    }
    
    delete $set{'command'};

    warn "c.v. set provided changes to the control variables: ".join(',',keys %set);

    # can't change these; hope they're the same
    delete $set{'module'};
    delete $set{'version'};
    delete $set{'department'};
    
    # handle setting of rulesets, only changed rulesets are mentioned
    if (defined($command->{'rulesets'})) {
      my %rulesets= %{$command->{'rulesets'}};
      foreach (keys %rulesets) { # check for deletions
	if ($rulesets{$_} eq '') {
	  # print "control var change: deleting $_\n";
	  warn "control var change: removing ruleset $_";
	  Ruleset->remove($_);
	  Buffer::remove_ruleset($_);
	  delete $::cvars->{'rulesets'}{$_};
	  delete $::cvars->{'debug'}{$_};
	  delete $rulesets{$_};
	} else {
	  # print "control variable change: adding $_\n";
	  warn "control variable change: adding ruleset $_";
	}
      }
      &::new_rulesets(%rulesets);
      delete $set{'rulesets'};
    }
    
    # handle setting of debug flag for rulesets, only changed ones need be
    # mentioned
    if (defined($command->{'debug'})) {
      foreach (keys %{$command->{'debug'}}) {
	warn "control var change; changing debug flag on $_";
	if (Ruleset->named($_)) {
	  Ruleset->named($_)->set_debug($command->{'debug'}{$_});
	  $::cvars->{'debug'}{$_}= $command->{'debug'}{$_};
	} else {
	  warn "attempt to set debug flag to $command->{'debug'}{$_} on unknown ruleset $_";
	}
      }
      delete $set{'debug'};
    }

    if (defined($command->{'DEBUG'})) {
      $::cvars->{'DEBUG'} = $command->{'DEBUG'};
      $::DEBUG = $command->{'DEBUG'};
	  warn "control var change; changing DEBUG to $::DEBUG\n";
      delete $set{'DEBUG'};
      }
    
    # these are changes to the (conceptual) sets c.v.'s; update $::cvars
    my($cv);
    foreach $cv (qw(children alert_recipients)) {
      if (defined($command->{$cv})) {
	# Assumes $command->{$cv} contains incremental changes to the set
	foreach (keys %{$command->{$cv}}) {
	  if ($command->{$cv}{$_}) { # an addition
	    print "control var change; adding $_ to $cv\n" if $::DEBUG;
	    warn "control var change; adding $_ to $cv";
	    $::cvars->{$cv}{$_} = 1;
	  } else { # a deletion
	    delete $::cvars->{$cv}{$_};
	    print "control var change; deleting $_ from $cv\n" if $::DEBUG;
	    warn "control var change; deleting $_ from $cv";
	  }
	}
	delete $set{$cv};
      }
    }
    
    # catch anything else, no special treatment needed
    foreach $cv (keys %set) {
      print "control var change; changing $cv\n" if $::DEBUG;
      warn "control var change; changing $cv to ".$command->{$cv};
      $::cvars->{$cv}= $command->{$cv}; # just copy over
    }
    $::cvars->update;     # RC 8/14: We want a current snapshot for debugging.

  } elsif($command->{'command'} =~ /^get$/i) {
    # maybe we should really only send requested params here

    $::cvars->update;
  } else { # not a set or get command
    warn "unknown control variable command received: $command->{'command'}";
  }
  $::cvars->complete_command;
}


# print out the control variables' values for debugging purposes
sub print_cvs {
  my($verbose)= shift;
  $verbose= 0 unless defined($verbose);
  my $cv;
  foreach $cv (qw(module version department listen_udp listen_tcp parent_aggregator beta gamma)) {
    print STDERR "$cv= $::cvars->{$cv}\n";
  }
  foreach $cv (qw(rulesets)) {
    if (defined($::cvars->{$cv})) {
      foreach (keys %{$::cvars->{$cv}}) {
	my ($text)= $::cvars->{$cv}{$_};
	if (!$verbose && length($text) > 43) {
	  $text= substr($text,0,40).'...';
	  $text =~ tr/\n/ /;
	}
	print STDERR "$cv\{$_}= ",$text,"\n";
      }
    }
  }
  foreach $cv (qw(debug children alert_recipients)) {
    if (defined($::cvars->{$cv})) {
      foreach (keys %{$::cvars->{$cv}}) {
	print STDERR "$cv\{$_}= $::cvars->{$cv}{$_}\n";
      }
    }
  }
}

# send_alert sends alerts generated by rulesets to the interested user
# interfaces.  The arguments are: the ruleset that generated the
# alert, the text of the alert, the text of a graph (which may be ''),
# and the level of the alert.  This is passed in the same order the
# user interface(s).
#
# Network::send_alert: ruleset_name x message x graph_text x integer ->
#
# e.g. &Network::send_alert('worm_detector',`fortune`,'',1);
#
sub send_alert {
  foreach (keys %{$::cvars->{'alert_recipients'}}) {
    if (/:\d+$/) { # send to the network
      my($host,$port)= split(':',$_,2);
      $::DEBUG > 1 && warn "sending alert(".join(',',@_).") to $host\@$port via TCP\n";
      &Comm::tcp_send($host,$port,'alt',@_);
    } else { # send to a file
      my($file)= /^\// ? $_ : $ENV{'GRIDSPATH'} . '/'. $_;
      if ($::compress_alerts) {
	$file.= '.gz';
	&Comm::prog_send("gzip - > $file",'alt',@_);
      } else {
	&Comm::file_send($file,'alt',@_);
      }
      $::DEBUG > 1 && warn "sent alert(".join(',',@_).") to file $file\n";
    }      
  }
}

# send_aggr sends aggregated reports generated by rulesets to the
# parent aggregator (if there is one).  The arguments are: the ruleset
# to which to send the report, and the DOT text of the graph being
# sent.
#
# Network::send_aggr: ruleset_name x report_text ->
#
# e.g. &Network::send_aggr('worm_detector',$graph_text);
#
sub send_aggr {
  my($parent,$host,$port,$file);
  if (defined($parent= $::cvars->{'parent_aggregator'}) && $parent !~ /^\s*$/) {
    if ($parent =~ /:\d+$/) { # send to the network
      ($host,$port)= split(':',$parent,2);
      $::DEBUG > 1 && warn "sending aggregated report(".join(',',@_).") to $host\@$port via TCP\n";
      &Comm::tcp_send($host,$port,'a',@_);
    } else { # send to a file
      $file= ($parent =~ /^\//) ? $parent : $ENV{'GRIDSPATH'} . '/'. $parent;
      &Comm::file_send($file,'a',@_);
      $::DEBUG > 1 && warn "sending aggregated report(".join(',',@_).") to file $file\n";
    }
  }
}

1;

