#!/pkg/bin/perl -w

# $Id: Ruleset.pm,v 1.25 1998/01/09 22:43:40 hoagland Exp $

# This file contains the Ruleset class.  The class deals with the
# overall execution of the rules within a ruleset and stores the
# graphs associated with a ruleset.  Most details of the ruleset
# syntax are handled by execute.pl. This class also takes care of
# propagation of edges upward and the propagation of "global"
# attributes upwards as node reports.  The class also does logging of
# graphs.  The Ruleset package maintains a registry (by name) of
# Ruleset objects which have been created.

require 'execute.pl';
require 'data_struct.pl';
#require 'dump.pl';

use Timeout;

package Ruleset;

sub BEGIN {
  $ADD_GDUMP_COUNT= 100; # the number of additions to the ruleset
                         # before dumping the graph to the log file
  $DEBUG_DEFAULT= 1;
  %registry=();
}

# Ruleset::new_rules creates a new Ruleset from the given source and
# stores it in a registry under the given name.  If there is already a
# Ruleset with the given name, it is replaced with the new one.  A
# true value is returned if a ruleset with the given name already
# existed.
#
# Ruleset::new_rules: class x name x text -> boolean
#
# if (Ruleset->new_rules('worm_detector',"ruleset worm_detector; buffer 3; ...")) { ...
#
sub new_rules {
  my($class,$name,$src)= @_;

  my $res= defined($registry{$name});
  $registry{$name}= Ruleset->new($src);
  return $res;
}

# Ruleset::remove removes the Ruleset with the given name from the registry.
#
# Ruleset::remove: class x name -> 
#
# Ruleset->remove('worm_detector');
#
sub remove {
  delete $registry{$_[1]};
}

# Ruleset::named returns a Ruleset from the registry with the given name
#
# Ruleset::named: class x name -> Ruleset
#
# Ruleset->named('worm_detector')->set_debug(1)
#
sub named {
  return $registry{$_[1]};
}

# Ruleset::all returns all registered Rulesets in a list
#
# Ruleset::all: class -> {Ruleset}
#
# &::add_report($report,Ruleset->all);
#
sub all {
  return values %registry;
}

# Ruleset::names returns the names of all registered Rulesets in a list
#
# Ruleset::all: class -> {name}
#
# foreach (Ruleset->names) { ...
#
sub names {
  return keys %registry;
}

##########

# Ruleset::new creates a new Ruleset object given the text version.
#
# Ruleset::new: class x text -> Ruleset
#
# $ruleset= Ruleset->new("ruleset worm_detector; buffer 3; ...");
#
sub new {
  my ($class,$src)= @_;
  my($name,$buffer_time,$graph_timeout,$attr_declsref,
   $node_precond,$edge_precond,$report_globalref,$report_noderef,
   $report_edgeref,$global_rulesref,$node_rulesref,
   $edge_rulesref,$assessmentsref)= &execute::preprocess_ruleset($src);

  my(%attr_decls)= &::split_list_to_hash('\s+',@{$attr_declsref});

  my($node_combine,@node_rules)= &execute::separate_rules(@{$node_rulesref});
  my($edge_combine,@edge_rules)= &execute::separate_rules(@{$edge_rulesref});

  return bless { # 'Src' => $src,
    'Debug' => $DEBUG_DEFAULT,
    'Name' => $name,
    'Buffer_time' => $buffer_time,
    'Timeout' => ($graph_timeout+1 >= 100000 ? 99998 : $graph_timeout),
    'Attr_decls' => \%attr_decls,
    'Node_precond' => $node_precond,
    'Edge_precond' => $edge_precond,
    'Report_globals' => $report_globalref,
    'Report_nodes' => $report_noderef,
    'Report_edges' => $report_edgeref,
    'Global_rules' => $global_rulesref,
    'Node_rules' => \@node_rules,
    'Node_combine' => $node_combine,
    'Edge_rules' => \@edge_rules,
    'Edge_combine' => $edge_combine,
    'Assess_rules' => $assessmentsref,
    'Reports_added' => 0,
    'Graphs' => {},
    'Inst_gid' => {},  # $self-{'Inst_gid'}{$node}{$inst} is the graph
                       # ID associated with a given node instance
    'Gid_cnt' => {},   # $self->{'Gid_ref_cnt'}{$gid} is the number of
                       # references to $gid in the values of Inst_gid;
                       # if this is zero, and $gid is not around
                       # anymore, then can remove the entry in Gid_ref
                       # to release memory
    'Gid_ref' => {},   # $self->{'Gid_ref'}{$gid} is the graph
                       # reference associated with a given graph ID
    'Timeout_mgr' => Timeout->new};
}

################

# Ruleset::set_debug updates the debug status for this ruleset.  This
# implies dumping the graphs to the log file.
#
# set_debug: Ruleset x boolean ->
#
# e.g. $ruleset->set_debug(1);
#
sub set_debug {
  my($self,$flag)= @_;
  $self->{'Debug'}= $flag;

  &::log_graphspace($self->{'Name'},$self->graphs_dot_text);
  $self->{'Reports_added'}= 0; # reset count
}

############

# Ruleset::check_timeouts checks for new graph timeouts and deletes
# any graphs that have timed out.
#
# check_timeout: Ruleset ->
#
# e.g. $ruleset->check_timeouts
#
sub check_timeouts {
  my $self=shift;
  my @graphs=();

  $self->remove_graphs(@graphs) if @graphs=$self->{'Timeout_mgr'}->timed_out;
  #print join(',',map($_->id,@graphs))," timed out\n" if @graphs;
}

###################
### access methods 
###################

# Ruleset::name returns the name of this ruleset
#
# name: Ruleset -> name
#
# e.g. $name= $ruleset->name;
#
sub name {
  return $_[0]->{'Name'};
}

# Ruleset::debug returns whether or not to debugging is turned on for
# this ruleset
#
# debug: Ruleset -> boolean
#
# e.g. if ($ruleset->debug) {
#
sub debug {
  return $_[0]->{'Debug'};
}

# Ruleset::buffer_time returns how long to buffer reports before this
# ruleset sees it
#
# buffer_time: Ruleset -> integer
#
# e.g. $time= $ruleset->buffer_time;
#
sub buffer_time {
  return $_[0]->{'Buffer_time'};
}

# Ruleset::timeout returns how long to to wait between additions to
# before deleting graphs in this ruleset
#
# timeout: Ruleset -> integer
#
# e.g. $time= $ruleset->timeout;
#
sub timeout {
  return $_[0]->{'Timeout'};
}

# Ruleset::attr_type returns the declared type of a given attribute or the
# undefined value if there has been no declaration
#
# attr_type: Ruleset x name -> string
#
# e.g. $value= $ruleset->attr_type('time');
#
sub attr_type {
  return ${$_[0]->{'Attr_decls'}}{$_[1]};
}

#############

# Ruleset::graphs returns the graphs in this Ruleset
#
# graphs: Ruleset -> [Graph]
#
# e.g. @graphs= $self->graphs();
#
sub graphs {
  return values %{$_[0]->{'Graphs'}};
}

# Ruleset::add_graphs adds the given Graphs to this Ruleset
#
# add_graphs: Ruleset x {Graph} -> 
#
# e.g. $self->add_graphs(@graphs);
#
sub add_graphs {
  my $self=shift;
  foreach (@_) {
    $self->{'Graphs'}{$_}= $_;
  }
}

# Ruleset::remove_graphs removes the given Graphs from this Ruleset
#
# remove_graphs: Ruleset x {Graph} -> 
#
# e.g. $self->remove_graphs(@graphs);
#
sub remove_graphs {
  my $self=shift;
  my($graph,$gid);
  foreach $graph (@_) {
    $gid= $graph->id;
    # print "deleting $gid\n";
    delete $self->{'Gid_ref'}{$gid} unless defined($self->{'Gid_cnt'}{$gid}) && $self->{'Gid_cnt'}{$gid} > 0;
    delete $self->{'Graphs'}{$graph};
  }
}


################

# Ruleset::graphs_dot_text returns the text for the Graphs in this
# Ruleset in the format that will be used in the debugging log, a
# series of graphs in the graph language (which is like DOT).
#
# graphs_dot_text: Ruleset -> string
#
# e.g. $text= $ruleset->graphs_dot_text();
#
sub graphs_dot_text {
  return join("\n\n",map($_->dot_text,$_[0]->graphs));
}

################

# Ruleset::test_preconds return a boolean value indicating whether of
# not a given report meets the preconditions for this ruleset.
#
# test_preconds: Ruleset x report -> boolean
#
# e.g. if ($rs->test_preconds($report)) {
#
sub test_preconds {
  my($self,$report)= @_;
  my($cur,$cur_attr);
  
  foreach ($report->nodes) {
    return 1 if &execute::evaluate($self->{'Node_precond'},'node precondition',
		       'new.node' => $_,
		       'new.global' => $report,
		       'node' => $_,
		       'global' => $report);
  }
  foreach ($report->edges) {
    return 1 if &execute::evaluate($self->{'Edge_precond'},'edge precondition',
		       'new.edge' => $_,
		       'new.source' => $_->src,
		       'new.dest' => $_->dest,
		       'new.global' => $report,
		       'edge' => $_,
		       'source' => $_->src,
		       'dest' => $_->dest,
		       'global' => $report);
  }
  return 0;
}

# Ruleset::combinable tests whether or not given graph is okay to
# combine with the given graph based on the ruleset rules.
#
# combinable: Ruleset x graph x graph -> boolean
#
# e.g. $rs->combinable($new_graph,$graph)
#
sub combinable {
  my($self,$newg,$graph)= @_;
  my($res,@gedges);
  my($combine)=0;

  my($node_idsref,$edge_idsref)= $graph->intersection($newg);
  my(@intnodes)= @{$node_idsref};
  my(%intedges)= ();
  foreach (@{$edge_idsref}) {
    $intedges{$_}= 1;
  }

  # try edges for combining
  if (defined($self->{'Edge_combine'})) {
    foreach $src (@intnodes) {
      foreach $dest (@intnodes) {
	foreach $nedge ($newg->edge_from_to($src,$dest)) {
	  # defer to the actual edge with the same id if there,
	  # otherwise try all until one is found
	  @gedges= $intedges{$nedge->id} ? ($graph->edge($nedge->id)) : $graph->edge_from_to($src,$dest);
	  # print "\@gedges= ",join(',',@gedges),"\n";
	  foreach $gedge (@gedges) {
	    &execute::evaluate($self->{'Edge_combine'},'edge combine rule',
			       'new.edge' => $nedge,
			       'new.source' => $nedge->src,
			       'new.dest' => $nedge->dest,
			       'new.global' => $newg,
			       'cur.edge' => $gedge,
			       'cur.source' => $gedge->src,
			       'cur.dest' => $gedge->dest,
			       'cur.global' => $graph,
			       'res.edge' => $gedge->res_attrs,
			       'edge' => $gedge->res_attrs);
	    $res= $gedge->res_attrs->attr_val('combine');
	    $gedge->clear_res;
	    if ($res) { # the graphs get combined and these edges get merged
	      $combine=1;
	      unless ($gedge->id eq $nedge->id) {
		#print "combining ",$nedge->id," and ",$gedge->id,"\n";
		# set the id to be the new id
		$graph->remove_edge($gedge->id);
		$gedge->set_id($nedge->id);
		$graph->add_edge($gedge);
		last; # skip checking other edges between
	      }
	    }
	  }
	}
      }
    }
  }
  
  return 1 if $combine;

  # try nodes for combining
  if (defined($self->{'Node_combine'})) {
    foreach (@intnodes) {
      my($gnode)= $graph->node($_);
      my($nnode)= $newg->node($_);
      &execute::evaluate($self->{'Node_combine'},'node combine rule',
		'new.global' => $newg,
		'new.node' => $nnode,
		'cur.global' => $graph,
		'cur.node' => $gnode,
		'res.node' => $gnode->res_attrs,
		'node' => $gnode->res_attrs);
      $res= $gnode->res_attrs->attr_val('combine');
      $gnode->clear_res;
      return 1 if $res;
    }
  }

  return 0;
}

# Ruleset::graph_from_report makes the given report into a
# full-fletched graph by applying the rules and adding it properly
# into the ruleset (i.e. initialization items).
#
# graph_from_report: Ruleset x Graph -> Graph
#
# e.g. $new_graph= $self->graph_from_report($report);
#
sub graph_from_report {
  my($self,$report)= @_;
  my($graph)= Graph->new($self->name,$report);
  my(@touched)=($graph);
  my($edge,$node,$rule);

  foreach $rule (@{$self->{'Report_globals'}}) {
    &execute::evaluate($rule,'report global graph rule',
	      'new.global' => $graph,
	      'res.global' => $graph->res_attrs,
	      'global' => $graph->res_attrs);
  }
  $graph->auto_computed_to_res();

  foreach $node ($graph->nodes) {
    foreach $rule (@{$self->{'Report_nodes'}}) {
      &execute::evaluate($rule,'report node rule',
		'new.global' => $graph,
		'new.node' => $node,
		'res.global' => $graph->res_attrs,
		'res.node' => $node->res_attrs,
		'global' => $graph->res_attrs,
		'node' => $node->res_attrs);
    }
    $node->auto_computed_to_res();
    push(@touched,$node);
  }

  foreach $edge($graph->edges) {
    foreach $rule (@{$self->{'Report_edges'}}) {
      &execute::evaluate($rule,'report edge rule',
		'new.edge' => $edge,
		'new.source' => $edge->src,
		'new.dest' => $edge->dest,
		'new.global' => $graph,
		'res.edge' => $edge->res_attrs,
		'res.source' => $edge->src->res_attrs,
		'res.dest' => $edge->dest->res_attrs,
		'res.global' => $graph->res_attrs,
		'edge' => $edge->res_attrs,
		'source' => $edge->src->res_attrs,
		'dest' => $edge->dest->res_attrs,
		'global' => $graph->res_attrs);
    }
    $edge->auto_computed_to_res();
    push(@touched,$edge);
  }

  foreach (@touched) {
    $_->move_res_to_cur;
  }

  # init graph history structures for this Graph 
  # at this point gid($graph)= {id($report),id($graph)}.
  # it is safe (hopefully) to make gid($graph)= {id($graph)} since
  # $report really isn't a graph and no references have been made to
  # it
  $graph->set_attr('gids',&::create_set($graph->id));
  $self->{'Gid_ref'}{$graph->id}= $graph;

  return $graph;
}

# Ruleset::update_attrs updates the attributes of the given "cur"
# Graph given a "new" Graph following the combining rules.
#
# update_attrs: Ruleset x Graph x Graph -> 
#
# e.g. $rs->update_attrs($curg,$newg);
#
sub update_attrs {
  my($self,$cur,$new)= @_;
  my(@touched)=($cur);
  my ($rule);

  my($node_idsref,$edge_idsref)= $cur->intersection($new);

  foreach $rule (@{$self->{'Global_rules'}}) {
    &execute::evaluate($rule,'global graph rule',
	      'new.global' => $new,
	      'cur.global' => $cur,
	      'res.global' => $cur->res_attrs,
	      'global' => $cur->res_attrs);
  }
  $cur->merge_auto_computed($new);

  foreach (@{$node_idsref}) {
    my($cnode)= $cur->node($_);
    my($nnode)= $new->node($_);
    foreach $rule (@{$self->{'Node_rules'}}) {
      &execute::evaluate($rule,'node rule',
		'new.global' => $new,
		'new.node' => $nnode,
		'cur.global' => $cur,
		'cur.node' => $cnode,
		'res.global' => $cur->res_attrs,
		'res.node' => $cnode->res_attrs,
		'global' => $cur->res_attrs,
		'node' => $cnode->res_attrs);
    }
    $cnode->merge_auto_computed($nnode);
    push(@touched,$cnode);
  }

  foreach (@{$edge_idsref}) {
    my $nedge= $new->edge($_);
    my $cedge= $cur->edge($_);
    # print "attrs from $nedge and $cedge are being combined\n";
    foreach $rule (@{$self->{'Edge_rules'}}) {
      &execute::evaluate($rule,'edge rule',
		'new.edge' => $nedge,
		'new.source' => $nedge->src,
		'new.dest' => $nedge->dest,
		'new.global' => $new,
		'cur.edge' => $cedge,
		'cur.source' => $cedge->src,
		'cur.dest' => $cedge->dest,
		'cur.global' => $cur,
		'res.edge' => $cedge->res_attrs,
		'res.source' => $cedge->src->res_attrs,
		'res.dest' => $cedge->dest->res_attrs,
		'res.global' => $cur->res_attrs,
		'edge' => $cedge->res_attrs,
		'source' => $cedge->src->res_attrs,
		'dest' => $cedge->dest->res_attrs,
		'global' => $cur->res_attrs);
    }
    $cedge->merge_auto_computed($nedge);
    push(@touched,$cedge);
  }

  foreach (@touched) {
    $_->move_res_to_cur;
  }
  &::remove_from_set($cur->{'gids'},$new->id) if defined($new->{'_fresh_gid'}) && $new->{'_fresh_gid'}; # no need to retain gid from new since it wasn't used
}

# Ruleset::assess runs the assessment rules on the given "res" Graph.
#
# assess: Ruleset x Graph -> 
#
# e.g. $rs->assess($res)
#
sub assess {
  my($self,$res)= @_;
  my($action,$ruleref);

  foreach $ruleref (@{$self->{'Assess_rules'}}) {
    my($test,@actions)= @{$ruleref};
    if (&execute::evaluate($test,'assessment rule test',
			   'res.global' => $res,
			   'global' => $res)) { # passed the test
      foreach $action (@actions) {
	&execute::evaluate($action,'assessment rule',
			   'res.global' => $res,
			   'global' => $res);
      }
    }
  }
}

# Ruleset::add_report adds the given report to the Graph(s) of this
#  ruleset following the ruleset rules.  The reports are assumed to
#  have met all the necessary conditions for entry into the Ruleset.
#
# add_report: Ruleset x Graph -> 
#
# e.g. $rs->add_report($report)
#
sub add_report {
  my($self,$report)= @_;
  my($graph,$other,@gredges,$inst_ids,$node,$aggr_node);

  my(@inst_ids)= ();
  foreach $node ($report->nodes) { # may be a aggregated node report
    # assumes that only one node contains the inst_ids attribute, if any do
    if (defined($inst_ids=$node->attr_val('inst_ids'))) {
      @inst_ids= &::members($inst_ids);
      $aggr_node= $node->attr_val('name');
    }
  }

  #==================
  $self->check_timeouts; # see if any of graphs should go away

  $self->{'Reports_added'}++;

  #==================

  # make the report a graph through rules
  $new= $self->graph_from_report($report);
  $new->{'_fresh_gid'}= 1;

  @gredges= $new->edges; # if an edge report, the edges involved
  
  #=========================================
  # here we determine what graphs will merge
  #=========================================

  my(@graphs); # the graphs to consider merging with
  my(%gids,@gids,$gid);
  if (defined($aggr_node)) { # an aggreagted node report came in
    foreach (@inst_ids) {
      if (defined($self->{'Inst_gid'}{$aggr_node}) && defined($self->{'Inst_gid'}{$aggr_node}{$_})) {
	$gid= $self->{'Inst_gid'}{$aggr_node}{$_};
	# gid's may be repeated so hash it in %gids and take keys
	$gids{$gid}= 1;
      }
    }
    # delete $gids{$new->id}; # don't want to merge with ourselves
    @gids= keys %gids;
    if (@gids) {
      @graphs= map($self->{'Gid_ref'}{$_},@gids);
       #print "(",join(',',map($_->id,@graphs)),") are the gids associated with instance (",join(',',@inst_ids),") of $aggr_node\n";
    } else { # the targeted graph must have been deleted or hasn't
             # been created before, so it can merge with
             # anything. QUESTION: is this the right behaviour?
      @graphs= $self->graphs;
      # print "no gids for (",join(',',@inst_ids),") of $aggr_node were found; anything is okay\n";
    }
  } else {
    @graphs= $self->graphs;
  }
  #print "graphs are (",join(',',map($_->id,@graphs)),")\n";

  # find the adjacent graphs
  my(@adj_graphs)= grep($_->is_adjacent($new),@graphs);
  #print "adj graphs are (",join(',',map($_->id,@adj_graphs)),")\n";

  # find the adjacent graphs that it is okay to incorporate into
  my($gr);
  my(@okay_graphs)= ();
  foreach $gr (@adj_graphs) {
    push(@okay_graphs,$gr) if $self->combinable($new,$gr);
  }
  #print "okay graphs are (",join(',',map($_->id,@okay_graphs)),")\n";

  $self->add_graphs($new);

  #================================
  # here we do the actually merging
  #================================
  
  # $graph stores the result of combining graphs
  if (@okay_graphs) {
    # actually add report into graphs
    $graph= shift(@okay_graphs);
    $graph->record_changes(); # note what most of the attributes are like now
    $graph->dont_record_changes('nnodes','nedges','ruleset','gids');

    # add report to first and merge rest pairwise with first
    foreach $other ($new,@okay_graphs) {
      # print "combining ",$other->id," into ",$graph->id,"\n";
      $self->{'Timeout_mgr'}->delete($other); # no longer needs timeout since going away
      # $graph is the new reference for $other's gid
      $self->{'Gid_ref'}{$other->id}= $graph;
      $self->update_attrs($graph,$other);
      $graph->incorporate_graph($other);
      $self->remove_graphs($other);
    }
  } else {
    $graph= $new;
    delete $graph->{'_fresh_gid'};
  }
  my($gid)= $graph->id;
  
 # start or reset timeout clock for graph
  $self->{'Timeout_mgr'}->add($self->{'Timeout'}+1,$graph);

  if (defined($aggr_node)) {
    foreach (@inst_ids) {
      $self->{'Gid_cnt'}{$self->{'Inst_gid'}{$aggr_node}{$_}}-- if defined($self->{'Inst_gid'}{$aggr_node}{$_});
      # mark the instance as having been added to this graph
      $self->{'Inst_gid'}{$aggr_node}{$_}= $gid;

      $self->{'Gid_cnt'}{$graph->id}++;
    }   
  }

  #===================

  $self->assess($graph);

  # propagate up edges
  foreach (@gredges) {
    $self->edge_prop($_);
  }

  # propagate up Graph as node
  $self->graph_prop($graph) if $graph eq $new || $graph->changes_names;

  if ($self->{'Debug'}) {
    &::log_graphspace($self->{'Name'},$self->graphs_dot_text) if ($self->{'Reports_added'} % $ADD_GDUMP_COUNT) == 0;
  }
}

# Ruleset::edge_prop takes the given edge and, if either the source or
# destination are out of the department, then sends the report up with
# the names of the local nodes translated to the name of the
# department.  Returns a true value iff the propagation actually took
# place.
#
# edge_prop: Ruleset x Edge -> boolean
#
# e.g. $rs->edge_prop($edge)
#
sub edge_prop {
  my($self,$edge)= @_;
  my($srcname,$destname,$src,$dest);

  $srcname= $edge->src->id;
  $destname= $edge->dest->id;
  my($src_is_foreign)= !defined($::cvars->{'children'}{$srcname});
  my($dest_is_foreign)= !defined($::cvars->{'children'}{$destname});
  
  if ($src_is_foreign || $dest_is_foreign) { # need to propagate up
    my($tmpg)= Graph->new($self->name); # construct a scratch graph
    $srcname= $::cvars->{'department'} unless ($src_is_foreign);
    $destname= $::cvars->{'department'} unless ($dest_is_foreign);
    $tmpg->add_node($src=Node->new($srcname,$edge->src->attr_hash)); 
    $tmpg->add_node($dest=Node->new($destname,$edge->dest->attr_hash));
    # $src and $dest hold the new nodes created
#    $src->remove_attrs('name');
#    $dest->remove_attrs('name');
    $tmpg->add_edge(Edge->new($src,$dest,$edge->id,$edge->attr_hash));
    $tmpg->remove_attrs('nnodes','nedges','ruleset','gids');
    &Network::send_aggr($tmpg->dot_text,$self->name);
    return 1;
  }
  return 0;
}

# Ruleset::graph_prop takes the given graph and sends the report up
# with the graph represented as the a node for this department and the
# graph history translated into instance ids.  
#
# graph_prop: Ruleset x Graph -> 
#
# e.g. $rs->graph_prop($graph)
#
sub graph_prop {
  my($self,$graph)= @_;
  my($node);

  my($tmpg)= Graph->new($self->name); # construct a scratch graph
  $tmpg->add_node($node=Node->new($::cvars->{'department'},$graph->attr_hash,'inst_ids' => $graph->attr_val('gids'))); 
  $node->remove_attrs('nnodes','nedges','ruleset','gids','name');
  $tmpg->remove_attrs('nnodes','nedges','ruleset','gids');
  &Network::send_aggr($tmpg->dot_text,$self->name);
}

1;
