#!/pkg/bin/perl

# $Id: parselog.pl,v 1.5 1997/02/01 20:02:25 hoagland Exp $

# parse_control_value takes a line of input and checks
# it against the definition of <control-value> in the
# Global Log File grammar.  If the line is compatible
# with <id>=<id>, then a list is returned containing
# the two <id>'s in correct order.  Else, undef is
# returned
#
sub parse_control_value {
  $_ = shift;
  if (/^([a-zA-Z_](?:\w|(?:\"(?:[^\\\"]|\\\")*\"))*)=([a-zA-Z_](?:\w|(?:\"(?:[^\\\"]|\\\")*\"))*)$/)
  {
    return($1,$2);
  }
  else
  {
    return undef;
  }
}

# parse_control_change takes a line of input and checks
# it against the definition of <control-change> in the
# Global Log File grammar.  If the line is compatible
# with "change: " <control-value>, then
# parse_control_value is called and it's value returned.
# Otherwise, undef is returned.  Hence, on success, a
# list containing two <id>'s is returned.
#
sub parse_control_change {
  $_ = shift;
  if (/^change: (.*)$/)
  {
    return parse_control_value($1);
  }
  else
  {
    return undef;
  }
}

# parse_ruleset_change takes a line of input and checks
# it against the definition of <ruleset-change> in the
# Global Log File grammar.  If the line is compatible
# with "ruleset change: " <control-value>, then
# parse_control_value is called and it's value returned.
# Otherwise, undef is returned.  Hence, on success, a
# list containing two <id>'s is returned.
#
sub parse_ruleset_change {
  $_ = shift;
  if (/^ruleset change: (.*)$/) {
    return &parse_control_value($1);
  } else {
    return undef;
  }
}

# parse_control_list takes an array as input.  The first
# element of the array is shifted into $count.  $count
# then refers to the position in the remaining array to
# start parsing from.  From this position, line after
# line (perhaps 0 lines) is parsed until the parsing
# no longer matches.  Each line is evaluated with
# parse_control_change.  An array containing all the
# parsed out <id>'s (an even number of them) is returned
# with the first element of the array stating the line
# number of the first line that did not parse.
#
sub parse_control_list {
  my $count = shift;
  my @returnlist;
  while( ($a,$b) = parse_control_change($_[$count]) )
  {
      push(@returnlist,$a);
      push(@returnlist,$b);
      $count++;
  }
  return ($count,@returnlist);
}

# parse_file_id takes a line of input and checks
# it against the definition of <file-id> in the
# Global Log File grammar.  If the line is compatible
# with <filename> ": " <revision-number> "\n", then
# the list (filename,revision-number) is returned,
# else undef.
#
sub parse_file_id {
  $_ = shift;
  if (/^(\S*): (\d*)$/)
  {
    return ($1,$2);
  }
  else
  {
    return undef;
  }
}

# parse_file_list takes an array as input.  The first
# element of the array is shifted into $count.  $count
# then refers to the position in the remaining array to
# start parsing from.  From this position, lines are
# compared one at a time to <file-id> using
# parse_file_id.  An array containing all the parsed
# out <filename>'s and <revision-number>'s is returned
# with the first element of the array stating the line
# number of the first line that did not parse.
#
sub parse_file_list {
  my $count = shift;
  my @returnlist;
  while( ($a,$b) = parse_file_id($_[$count]) )
  {
    push(@returnlist,$a);
    push(@returnlist,$b);
    $count++;
  }
  return ($count,@returnlist);
}

# parse_comment takes a line of input and checks to
# see if it is a comment.  If so, it returns 'true',
# else undef.
#
sub parse_comment {
  $_ = shift;
  if (/^#.*$/)
  {
    return 'true';
  }
  else
  {
    return undef;
  }
}

# parse_shutdown takes a line of input and checks to
# see if it will parse as a <shutdown>.  If so, it
# returns <time> and <invocation-number> in a list,
# else undef.
#
sub parse_shutdown {
  $_ = shift;
  if (/^<(\d*);(\d*)$/)
  {
    return ($1,$2);
  }
  else
  {
    return undef;
  }
}

# parse_acceptance takes a line of input and checks to
# see if it will parse as an <acceptance> as defined
# in the Ruleset Log File grammar.  If so, the
# <report-number> is returned.  Else, undef.
#
sub parse_acceptance {
  $_ = shift;
  if (/^(\d*)$/)
  {
    return $1;
  }
  else
  {
    return undef;
  }
}

# parse_rejectance takes a line of input and checks to
# see if it will parse as a <rejectance> as defined
# in the Ruleset Log File grammar.  If so, the
# <report-number> is returned.  Else, undef.
#
sub parse_rejectance {
  $_ = shift;
  if (/^!(\d*)$/)
  {
    return $1;
  }
  else
  {
    return undef;
  }
}

1;
