#!/pkg/bin/perl -w

# $Id: primitive.pl,v 1.12 1997/02/01 20:24:43 hoagland Exp $

# This file contains the primitive functions for ruleset evaluations,
# i.e. the built-in functions

# For reasons of availability, the functions in this file
# directly access the attribute representation rather that using
# routines in attr.pl

# Here are the prototypes of the functions defined here:
#
# scalar in_set(scalar,set)
# scalar max(set)
# scalar min(set)
# scalar empty(set)
# scalar on_list(scalar,list)
# scalar head(list)
# scalar last(list)
# list   sort(list)
# list   sort_numerically(list)
# scalar abs(scalar)

# void   alert(scalar,scalar)
# void   report_graph(scalar,scalar)

# The in_set primitive returns whether the first argument is a member
# of the set which is the second argument. O(1).
#
# in_set: scalar x set -> boolean
#
# e.g. in_set("a",{"a","b","c"})
#
sub in_set {
  my($element,$setref)= @_;
  return defined(${$setref}{$element}) && ${$setref}{$element};
}

# The max primitive returns the element of the set with the highest
# numeric value. O(len).
#
# max: set -> scalar
#
# e.g. max({1,4,3})
#
sub max {
  my(%set)= %{$_[0]};
  my(@members)= grep($set{$_},keys %set);
  my($max)= shift(@members);
  foreach (@members) {
    $max=$_ if ($_ > $max);
  }
  return $max;
}

# The min primitive returns the element of the set with the lowest
# numeric value. O(len).
#
# min: set -> scalar
#
# e.g. min({1,4,3})
#
sub min {
  my(%set)= %{$_[0]};
  my(@members)= grep($set{$_},keys %set);
  my($min)= shift(@members);
  foreach (@members) {
    $min=$_ if ($_ < $min);
  }
  return $min;
}

# The empty primitive returns whether the set contains any elements
# ~O(len).
#
# empty: set -> boolean
#
# e.g. empty({})
#
sub empty {
  my(%set)= %{$_[0]};
  foreach (keys %set) {
    return 0 if $set{$_};
  }
  return 1;
}


#########

# The on_list primitive returns whether the first argument is on
# the list which is the second argument. O(len).
#
# on_list: scalar x list -> boolean
#
# e.g. on_list("a",["a","b","c"])
#
sub on_list {
  my($scalar,$listref)= @_;
  foreach (@{$listref}) {
    return 1 if $scalar eq $_;
  }
  return 0;
}


# The head primitive returns the first element of list.  O(1).
#
# head: list -> scalar
#
# e.g. head(["a","b","c"])
#
sub head {
  return ${$_[0]}[0];
}

# The last primitive returns the last element of list.  O(1).
#
# last: list -> scalar
#
# e.g. last(["a","b","c"])
#
sub last {
  my(@list)= @{$_[0]};
  return $list[$#list];
}

# The sort primitive returns the given list sorted lexically.
# O(klogk) for a list of length k.
#
# sort: list -> list
#
# e.g. sort(["a","c","b"])
#
sub sort {
  return [sort @{$_[0]}];
}

# The sort_numerically primitive returns the given list sorted numerically.
# O(klogk) for a list of length k.
#
# sort_numerically: list -> list
#
# e.g. sort_numerically([3,1,2])
#
sub sort_numerically {
  return [sort {$a <=> $b} @{$_[0]}];
}

#######

# The abs primitive returns the absolute value of the number passed in.  O(1).
#
# abs: scalar -> scalar
#
# e.g. abs(0-3)
#
sub abs {
  my $num= shift;
  return $num >= 0 ? $num : 0-$num;
}

#########

# The alert primitive sends the alert level in the first argument and
# text given in the second argument to the user interface.
#
# alert: scalar -> void
#
# e.g. alert(1,"Worm sweep detected!")
#
# assumes that the ruleset name is available as $graph->{'ruleset'}
#
sub alert {
  &Network::send_alert($graph->{'ruleset'},$_[1],'',$_[0]);
}

# The report_graph primitive sends the alert level in the first
# argument, text given in the second argument and the current graph to
# the user interface.
#
# report_graph: scalar -> void
#
# e.g. report_graph(2,"Worm sweep detected!")
#
# assumes that the current graph is available as $graph
#
sub report_graph {
  &Network::send_alert($graph->{'ruleset'},$_[1],$graph->dot_text,$_[0]);
}
 

1;
