
# A place for useful primitives that we don't want to have to document in
# the technical report just yet.

# Prototypes
# scalar report_threshold(scalar,scalar,scalar,scalar)
# scalar assign_primitive(scalar,scalar,scalar)
# scalar current_color(scalar,scalar)
# scalar widen(scalar,scalar,scalar)
# scalar cond(scalar,scalar,scalar)
# scalar dbg_print(scalar)
# scalar dbg_query(scalar)
# scalar round(scalar)



#######

#==== report_threshold($variable,$initial,$increment,$name) 

# The report_threshold primitive assists in reporting graphs with a reasonable
# frequency as they grow.  The size measure (such as global.nnodes) is passed
# in as the $variable argument.  As it grows, the report_threshold will
# report true once when variable gets to $initial, and then again every time
# $variable is some multiple of $increment more than $initial.  The state
# to do this is managed separately for each instance of $name - different
# $names do not interfere.  Typically the name should be chosen to distinguish
# the ruleset and the particular assessment rule.
#
# For example, report_threshold(global.nnodes,7,3,"waffles") will return
# true when global.nnodes first passes 7,10,13,16,19 and so on.

sub report_threshold {
  my ($variable,$initial,$increment,$name) = @_;
  my $raw_success;
  return 0 unless $variable >= $initial;
  my $ratio = ($variable - $initial)/$increment;
  return 0 unless $raw_success = (int($ratio) == $ratio);
  return 0 if $Report_threshold::state{$name}{$ratio};
  $Report_threshold::state{$name}{$ratio} = 1;
}

#########


####################
sub assign_primitive {
  my ($obj, $attr, $val) = @_;
  ${$bindings}{$obj}->set_attr ($attr, $val);

  &prim_warn ("Assign_primitive:: \${\$bindings}{$obj}->{$attr} "
             . "==> ${$bindings}{$obj}->{$attr}!");
  $val;
  }


#########


# &current_color ('r1', 5);
################# 
sub current_color {
  my ($ruleset_name, $timeout) = @_;
  if (defined $Current_color::initial_time{$ruleset_name}) {
    my $elapsed_time = time - $Current_color::initial_time{$ruleset_name};
    my $phase = int ($elapsed_time / $timeout);
    if ($phase > $#{$Current_color::phases{$ruleset_name}}) {
      $phase = $#{$Current_color::phases{$ruleset_name}};
      # or wraparound?  ...
      # $phase = $phase % $#{$Current_color::phases{$ruleset_name}};
      }
    return $Current_color::phases{$ruleset_name}[$phase];
    }
  else {
    $Current_color::phases{$ruleset_name} = [
        "X500",
        "Xa00",
        "Xf00",
        "X050",
        "X0a0",
        "X0f0",
        "X005",
        "X00a",
        "X00f",
       ]
         unless defined $Current_color::phases{$ruleset_name};
   $Current_color::initial_time{$ruleset_name} = time;
   return $Current_color::phases{$ruleset_name}[0];
   }
} # end current_color


#########



# &widen ('r2', 'x50', 200);
# &widen ('r2', 'w50', 200);
# &widen ('r2', 'w120', 200);
######### (ruleset_name, curr.edge.style, timeout)
sub widen {
  my ($ruleset_name, $curr_edge_width, $timeout) = @_;
                                     # $timeout  not currently used.
  my ($width) = $curr_edge_width =~ /^w(\d+)$/i ;
  unless ($width && 0 < $width && $width <= 128) {
    &prim_warn
    ("Widen::$ruleset_name- Incoming edge style <$curr_edge_width> illegal!");
    return 'w20';     # some "standard" width, where range:: 1 .. 128
    }
  unless (defined $Widen::increment{$ruleset_name}) {
    $Widen::increment{$ruleset_name} = 20;
    }
  $width += $Widen::increment{$ruleset_name};
  if ($width > 128) {
    $width = 128;
    }
  &prim_warn
       ("Widen::$ruleset_name- boosted <$curr_edge_width> to <w$width>.");
  return 'w' .  $width;
  }


#############
sub prim_warn {
  print STDERR "$_[0]\n";
  my $date = `date`;
  chomp $date;
  if ($ENV{'GRIDS_DEBUG'} =~ /no_prim_warn/) {
    return;
    }
  else {
    my $file = "$ENV{'GRIDSPATH'}/log/prim";
    open (PRIM, ">>$file");
    print PRIM "$date :: $_[0]\n";
    close (PRIM);
    }
  }


#############
# Use this to insert debugging stmts in a ruleset.  Prints to file log/prim
sub dbg_print {
  my $date = `date`;
  chomp $date;
  if ($_[0] eq "-- separator --") {
    print STDERR "----------------------  $date  -----------------------\n";
    }
  else {
    print STDERR join ' ', @_, "\n";
    }
  if ($ENV{'GRIDS_DEBUG'} =~ /no_prim_warn/) {
    return;
    }
  else {
    my $file = "$ENV{'GRIDSPATH'}/log/prim";
    open (PRIM, ">>$file");
    if ($_[0] eq "-- separator --") {
      print PRIM "----------------------  $date  -----------------------\n";
      }
    else {
      print PRIM join ' ', @_, "\n";
      }
    close (PRIM);
    }
  # unless explict "return 0;" stmt, its retval is treated as true by  "&&"
  }


#############
# Used as debugging call in a ruleset to view graphs.  Prints to file log/prim
sub dbg_query {
  my $file = "$ENV{'GRIDSPATH'}/log/prim";
  open (PRIM, ">>$file");
  print PRIM &::do_query($_[0]);
  close (PRIM);
  }


#############
sub cond {
  while (@_ > 1) {
    my($test,$res)= (shift(@_),shift(@_));
    return $res if $test;
  }
  return $_[0];
}

###########
# round the argument to the nearest integer
sub round {
  return int($_[0]+0.5);
}

1;
