#!/usr/bin/perl -w

# $Id: AttrObj.pm,v 1.8 1997/02/01 20:09:14 hoagland Exp $

# This file contains the AttrObj class and methods.

# AttrObj's are objects that have attributes; pseudo-attributes begin
# with a "_" and are (partially) hidden by the class.  Methods are
# also provided to selectively note changes to attributes stored in
# the object (see below).
#
# Attribute values stored in the class can be any scalar, but
# "attrs_dot_text" method assumes that they are in the format of
# attr.pl.  The default comparison operation (used in the "changed"
# functionality) also makes this assumption.


require 'attr.pl';

#------------------------------------------------------------------------------

package AttrObj;   
use strict 'refs';

# AttrObj::new creates a new AttrObj with the given attributes
#
# new: class x {attr_name x val} -> AttrObj
#
# e.g. $obj= AttrObj->new('a' => 'b', 'c' => 'd');
#
sub new {
  shift;
  bless {@_};
}

########

# AttrObj::set_attr sets the attribute value of this object.
# Returns the value as a convenience.
#
# set_attr: AttrObj x attr_name x attr_val -> attr_val
#
# e.g. $node->set_attr('count' => 3);
#
sub set_attr {
  my($self,$name,$val)= @_;
  $self->{$name}= $val;
  return $val;
}

# AttrObj::set_attrs sets the attribute values of this object.  The
# input is a hash where the keys are the attribute names and the
# corresponding value is the attribute value.  Returns itself as a
# convenience.
#
# set_attrs: AttrObj x {attr_name x attr_val} -> AttrObj
#
# e.g. $node->set_attrs(%attrs);
#
sub set_attrs {
  my($self,%attrs)= @_;
  foreach (keys %attrs) {
    $self->{$_}= $attrs{$_};
  }
  return $self;
}

# AttrObj::remove_attrs removes the given attributes of this object.
# The input is a list of the attribute names to remove.  Returns
# itself as a convenience.
#
# remove_attrs: AttrObj x {attr_name} -> AttrObj
#
# e.g. $graph->remove_attrs(@names);
#
sub remove_attrs {
  my($self)= shift;
  foreach (@_) {
    delete $self->{$_};
  }
  return $self;
}

# AttrObj::clear_attrs removes all the attributes of this object.  The
# pseudo-attributes (those beginning with an "_") are not removed.
# Returns itself as a convenience.
#
# clear_attrs: AttrObj -> AttrObj
#
# e.g. $graph->clear_attrs();
#
sub clear_attrs {
  my($self)= shift;
  foreach (keys %{$self}) {
    next if /^_/;
    delete $self->{$_};
  }
  return $self;
}


#######

# AttrObj::attr_names returns the attribute names of this object.
# Attributes with names starting with "_" are hidden.
#
# attr_names: AttrObj -> {attr_name}
#
# e.g. @attrs= $graph->attr_names();
#
sub attr_names {
  return grep (!/^_/,keys %{$_[0]});
}

# AttrObj::attr_val returns the attribute value of this object.
#
# attr_val: AttrObj x attr_name -> attr_value
#
# e.g. $val= $graph->attr_names('prot');
#
sub attr_val {
  return ${$_[0]}{$_[1]};
}


# AttrObj::attr_hash returns a hash with the attribute of this object,
# the name as the key and the value as the value. Keys starting with '_' are not returned.
#
# attr_hash: AttrObj -> attrs-hash
#
# e.g. %attrs= $graph->attr_hash();
#
sub attr_hash {
  return &::clean_hash('^_',%{$_[0]});
}

#######

# These functions enable recording of changes to the attribute values stored
# in this object.  These changes *need not* be made through certain functions
# (i.e. set_attr).  An attribute entry with an undefined value is considered
# the same as having no entry at all.  What "changed" means can be specified
# by providing a function which tests for equality; the default is to use
# attr.pl's attr_equal().

# AttrObj::record_changes takes a list of attribute names to pay attention
# to as to whether or not they change (starting now).  If record_changes is
# called on the same attribute twice, then the change recording for the
# attribute is reset.  If no attributes are given as arguments, then all
# attributes presently defined that don't start with an "_" are takes as the
# attributes in question.
#
# record_changes: AttrObj {attr_name} -> 
#
# e.g. $graph->record_changes('alerts','time');
#      $node->record_changes();
#
sub record_changes {
  # take a snapshot of current state and clear relevant "dont-records"
  my($self,@attrs)= @_;
  @attrs= $self->attr_names unless (@attrs); # all attrs
  foreach (@attrs) {
    $self->{'_snapshot'}{$_}= $self->{$_} if defined($self->{$_});
    delete $self->{'_dont_record'}{$_};
  }
}

# AttrObj::dont_record_changes takes a list of attribute names to explicitly
# not pay attention to as to whether the state has changed.  For a given
# attribute, if this method is called for that attribute, any noted changes
# in state are forgotten and no future ones will be noted (until an
# appropriate call to record_changes).  If no attribute names are given as
# arguments, then all attributes presently defined that don't start with an
# "_" are takes as the attributes in question.
#
# dont_record_changes: AttrObj {attr_name} -> 
#
# e.g. $graph->dont_record_changes('alerts','time');
#      $node->dont_record_changes();
#
sub dont_record_changes {
  # set relevant "dont-records" and clear some of snapshot
  my($self,@attrs)= @_;
  @attrs= $self->attr_names unless (@attrs); # all attrs
  foreach (@attrs) {
    $self->{'_dont_record'}{$_}= 1;
    delete $self->{'_snapshot'}{$_};
  }
}

# AttrObj::changes_names returns those attribute names whose changes in
# values were being paid attention to and have changed and those which are
# not on either the pay-attention-to or dont_record lists.  "Changed" is
# considered with respect to attributes unless an argument is given.  This
# argument is a reference to a subroutine which compares its first two
# arguments, returning a boolean if they are not different.  The default
# function is attr_equal.
#
# changes_names: AttrObj [function-ref] -> {attr_names}
#
# e.g. @changed= $graph->changes_names();
#      @changed= $graph->changes_names(\&numeq)); sub numeq {!($_[0] <=> $_[1])}
#
sub changes_names {
  my($self,$eqfnref)= @_;
  $eqfnref=\&::attr_equal unless defined($eqfnref);
  my (@changed)=();
  # for attributes on snapshot, return it if has changed or disappeared
  foreach (keys %{$self->{'_snapshot'}}) {
    unless (&{$eqfnref}($self->{'_snapshot'}{$_},$self->{$_})) {
      push(@changed,$_);
    }
  }
  # if an attribute is defined but not on the dont-record list of the snapshot, then it must be relatively new; return it
  foreach ($self->attr_names) {
    unless (defined($self->{'_snapshot'}{$_}) || defined($self->{'_dont_record'}{$_}))  {
      push(@changed,$_);
    }
  }
  return (@changed);
}

######

# AttrObj::attrs_dot_text returns the text for the attributes stored
# in this AttrObj in the Graph Language (which is like DOT) joined by
# the given text.  The attribute values are assumed to be in the
# format in attr.pl.  This method essentially joins together strings
# of the form "name"=<encoded-value>.  Attributes starting with "_"
# are ignored.
#
# attrs_dot_text: AttrObj -> string
#
# e.g. $text= $graph->attrs_dot_text(";\n\t");
#      $text= $node->attrs_dot_text(',');
#
sub attrs_dot_text {
  my($self,$jointext)= @_;
  return join($jointext,map(&::dot_encode($_)."=".&::flatten_attr($self->{$_}),$self->attr_names));
}

1;

