# This is a log object which performs one or more of the following
# logging functions:
#    + central logging (by default off; can be turned on)
#    + local logging (turned on if the user supplies a filename) 
#    + send message to STDERR (by default off; can be turned on)
#
# Assumptions:
#    + The central logging server lives in $LOG_SERVER_HOST
#      using port $LOG_SERVER_PORT
#
# Notes:
#    + A new message type, debug, is introduced. Messages that
#      carry this message types are for debugging purposes only.
#      As a result, debug messages and non-debug messages do not
#      interfere with each other.
#
#
# Basic usage:
#
# $my_log = new Clog($module_name,$dept_name,$filename);
#    $module_name is the name of the module (e.g., aggregator, ohs)
#    $dept_name is the name of the department. We need dept_name because
#    more than one instances of a module can live in a host.
#    When we have a standard way to access some control var for the
#    module name and department name, we can remove the first two
#    arguments and perform minor changes to the code.
#    log msg will be appended to the local log $filename if
#    $filename is non-null
# $my_log->warn("Minor info");		# Write to log and continue
# $my_log->die("Major disaster");	# Write to log and die.
#
#
#
# Optional things that can be done include:
#
# $my_log->separator();
#    places a separator line in local log file.
#
# $my_log->{'prefix'} = $info_string;   
#    $info_string will precede any subsequent local log messages.
#    Allows one to prefix all messages with process-id, program name,
#    etc without having to remember it in every "warn".
#
# $my_log->{'filter'}{'debug'} = 1;
# $my_log->warn("Warning",'debug');
#    If an optional second argument is supplied to warn, then the warning will
#    only be written into the log if that second argument is contained in the
#    hash $my_log->{'filter'} with a true value.  This allows one to smatter
#    the code with log messages of various kinds and then switch them on and
#    off as necessary.  (Note that they incur the function call overhead even
#    when switched off however.)
#
# $my_log->{'central_log'} = 1;
#    log msg will sent to central log server if $my_log->{'central_log'}
#    and the second argument, if any, is contained in the hash
#    $my_log{`filter'} with a true value. $my_log->{'central_log'} is
#    true by default.
#
# $my_log->{'stderr'} = 1;
#    If $my_log->{'stderr'} is true, output message to STDERR.
#    $my_log->{'stderr'} is false by default.
#
# $my_log->close();
#    Close the local log file.

# A test file for this module exists - Clog.pm.test

use Comm;

package Clog;

$LOG_SERVER_HOST = "olympus.cs.ucdavis.edu";
$LOG_SERVER_PORT = 2222;  # should be the same as that defined in log_server.pl 

$handle_count = 1;

# Should we prepend extra time & ID info to msgs logged locally?
$default_local_prepend = 1;
$default_stderr_prepend = 0;
$join_string = ':';  # string to join various items in Clog msgs.

$logerrs = 0;      # &logerr uses to decide re disabling central logging
$max_logerrs = 5;

#============================================================================#

# This function sets up any necessary files and sockets for the reporting
# of log information.

sub new
{
  my($type,$module_name,$dept_name,$file) = @_;
  my($self);
  my($date) = `date`; chomp $date;
  my($handle) = "LOG".$handle_count++;
  my $hostname = `/bin/uname -n` if ( -e "/bin/uname" );  
  $hostname = `/usr/bin/uname -n` if ( -e "/usr/bin/uname" );  
  chomp $hostname;
  if ($file) {
    system ("mkdir $ENV{'GRIDSPATH'}/log") unless -d "$ENV{'GRIDSPATH'}/log";
    $file = (split('/', $file)) [-1];   # get last path component
    $file = "$ENV{'GRIDSPATH'}/log/$file";
    open($handle,">>$file") || die "Can't open log file $file\n";
    # set output buffering to be line oriented.
    my $old_handle = select $handle;
    $| = 1;
    select $old_handle;
    my $now = localtime();
    print $handle <<"  End";
  ############################################################################
  ================================= STARTUP ==================================
  ======================  $date  ======================
  ============================================================================
  Debugging program $0 with process id $$ on $hostname at time $now representing department $dept_name.
  End
  }


  # kludge for testing &prune_logfile():
  $MAX_LOGFILE_SIZE = $main::PRUNETEST_MAX_LOGFILE_SIZE
                      if defined $main::PRUNETEST_MAX_LOGFILE_SIZE;
  $MIN_LOGFILE_SIZE = $main::PRUNETEST_MIN_LOGFILE_SIZE
                      if defined $main::PRUNETEST_MIN_LOGFILE_SIZE;
  $filecheck_modulus = $main::prunetest_filecheck_modulus
                       if defined $main::prunetest_filecheck_modulus;

  
  bless $self = {'central_log' => 0, 'dept_name' => $dept_name,
    'handle' => $handle, 'local_log' => $file,
    'module_name' => $module_name, 'prefix' => "", 'stderr' => 0,
    'local_prepend' => $default_local_prepend,
    'stderr_prepend' => $default_stderr_prepend,
    'join_string' => $join_string};
}

#============================================================================#

# Handle errors from attempts to tcp_send to a non-existent central_log:
sub logerr {
  my $self = shift;
  if ($_[0] =~ /true/i) {
    $logerrs = 0;
    return;
    }
  if (++$logerrs > $max_logerrs) {
    $self->{'central_log'} = 0;   # disable it until told otherwise
    print STDERR "\n Clog disabling unresponsive central_log: <$_[0]>\n";
    $_[0] = "Clog disabled central_log after $max_logerrs consecutive errors "
          . "from tcp_send.  Last error:: <$_[0]>\n";
    $self->warn ($_[0]) if $self->{'local_log'};
    }
}

#============================================================================#

# This logs some message supplied to it.
#
# Usage is
# 
# $mylog->warn("Help, I'm drowning");

sub warn
{
  my($self) = shift;
  my($handle) = $self->{'handle'};
  my $ret = 'true';

  if(defined $_[1])
   {
    return 'filter false' unless $self->{'filter'}{$_[1]};
   }
  if ($self->{'central_log'}) {
    $ret = &Comm::tcp_send($LOG_SERVER_HOST,$LOG_SERVER_PORT,'debug',
      (time, $self->{'module_name'}, $self->{'dept_name'},
       "$self->{'prefix'}$_[0]", $_[1]));
    $self->logerr ($ret);
    }
  if ($self->{'local_log'}) {
    &prune_logfile ($self) unless ++$filecheck_counter % $filecheck_modulus;
    if ($self->{'local_prepend'}) {
      print $handle join ($join_string, 
                    "$self->{'prefix'}$_[0]"),
                    (defined $_[1] ? "$join_string$_[1]" : ""),  "\n";
      }
    else {
      print $handle "$self->{'prefix'}$_[0] $_[1]\n";
      }
  }
  if ($self->{'stderr'}) {
    if ($self->{'stderr_prepend'}) {
      print STDERR join ($join_string, time,
                               $self->{'module_name'}, $self->{'dept_name'},
                    "$self->{'prefix'}$_[0]"),
                    (defined $_[1] ? "$join_string$_[1]" : ""),  "\n";
      }
    else {
      print STDERR "$self->{'prefix'}$_[0] $_[1]\n";
      }
  }
  return $ret;
}

#============================================================================#

# This logs some message supplied to it and then dies.
#
# Usage is
# 
# $mylog->die("Help, I'm drowning");

sub die
{
  my($self) = shift;
  my($handle) = $self->{'handle'};

  if(defined $_[1])
   {
    return unless $self->{'filter'}{$_[1]};
   }
  if ($self->{'central_log'}) {
    Comm::tcp_send($LOG_SERVER_HOST,$LOG_SERVER_PORT,'debug',
      (time, $self->{'module_name'}, $self->{'dept_name'},
       "$self->{'prefix'}$_[0]", $_[1]));
  }
  if ($self->{'local_log'}) {
    if ($self->{'local_prepend'}) {
      print $handle join ($join_string, time,
                               $self->{'module_name'}, $self->{'dept_name'},
                    "$self->{'prefix'}$_[0]"),
                    (defined $_[1] ? "$join_string$_[1]" : ""),  "\n";
      }
    else {
      print $handle "$self->{'prefix'}$_[0] $_[1]\n";
      }
  }
  if ($self->{'stderr'}) {
    my $errmsg;
    if ($self->{'stderr_prepend'}) {
      $errmsg = join ($join_string, time,
                               $self->{'module_name'}, $self->{'dept_name'},
                    "$self->{'prefix'}$_[0]") .
                    (defined $_[1] ? "$join_string$_[1]" : "") .  "\n";
      }
    else {
      $errmsg = "$self->{'prefix'}$_[0] $_[1]\n";
      }
    die ($errmsg);
    }
  else { exit $!; }
}

#============================================================================#

# This places a spacer in the local log file.

sub separator
{
  my($self) = shift;
  my($date) = `date`; chomp $date;
  my($handle) = $self->{'handle'};
  if (defined($self->{'local_log'})) { print $handle <<"End"; }
  ============================================================================
  ======================  $date  ======================

End

}

#============================================================================#

sub close
{
  my($self) = shift;
  if (defined($self->{'local_log'})) { close($self->{'handle'}); }
}

#============================================================================#

# Reopen could be called by anybody, but probably only by MC
# when all its kids are killed, and  /log is renamed to /log.1 .

sub reopen_logfile {
  my($self,$dept_name,$file) = @_;
  my($date) = `date`; chomp $date;
  my($handle) = "LOG".$handle_count++;
  my $hostname = `/bin/uname -n` if ( -e "/bin/uname" );  
  $hostname = `/usr/bin/uname -n` if ( -e "/usr/bin/uname" );  
  chomp $hostname;

  $self->close();
  $file = (split('/', $file)) [-1];   # get last path component
  $file = "$ENV{'GRIDSPATH'}/log/$file";
  system ("mkdir $ENV{'GRIDSPATH'}/log") unless -d "$ENV{'GRIDSPATH'}/log";
  open($handle,">>$file") || die "Can't open log file $file\n";
  # set output buffering to be line oriented.
  my $old_handle = select $handle;
  $| = 1;
  select $old_handle;
  my $now = localtime();
  print $handle <<"  End";
  ############################################################################
  =========================== REOPEN (no STARTUP) ============================
  ======================  $date  ======================
  ============================================================================
  Debugging program $0 with process id $$ on $hostname at time $now representing department $dept_name.
  End
  
  $self->{'handle'} = $handle;
  $self->{'local_log'} = $file;
  $self->{'dept_name'} = $dept_name;
  }



#============================================================================#

# RC 4/1/97 - periodically trim logfiles automatically

$MAX_LOGFILE_SIZE = 100 * 1000;
$MIN_LOGFILE_SIZE = 20 * 1000;
$filecheck_modulus = 10;

# Kludge for testing.  Overrides of default LOGFILE-SIZEs
#  -- and especially of $filecheck_modulus  --
# must be done at execute vs. compile time.
# Hence they have been moved to &new(), e.g:
    # $filecheck_modulus = $main::prunetest_filecheck_modulus
    #                      if defined $main::prunetest_filecheck_modulus;

 
sub prune_logfile {
  my ($self) = shift;
  my ($log_tail, $cvars) = (0,0);

  unless ($max_logfile_size) {     # this global initialized only once.
    $cvars = &find_cvars($self);
    if ($cvars  &&  $cvars->{'max_logfile_size'}) {
      $max_logfile_size = $cvars->{'max_logfile_size'};
      $min_logfile_size = $cvars->{'min_logfile_size'};
      }
    else {
      $max_logfile_size = $MAX_LOGFILE_SIZE;
      $min_logfile_size = $MIN_LOGFILE_SIZE;
      }
    $min_logfile_size = $max_logfile_size / 2
                        unless defined $min_logfile_size;
    }
  return unless  -s $self->{'local_log'} > $max_logfile_size;
  # else it's too big and must be pruned ...

  close $self->{'handle'};    # now open for read:
  if ( open ($self->{'handle'}, "<$self->{'local_log'}") ) {
    seek ($self->{'handle'}, -$min_logfile_size, 2);
                             # seek that far back from EOF
    read ($self->{'handle'}, $log_tail, $min_logfile_size);
    close $self->{'handle'};
    }
  open ($self->{'handle'}, ">$self->{'local_log'}");  # zero it out.
  print { $self->{'handle'} }  $log_tail;             # dump the data.
  ##### { above brackets } required; else perl prints 2 strings to STDOUT!
  close $self->{'handle'};

  open ($self->{'handle'}, ">>$self->{'local_log'}"); # open for future append.
  select ( (select($self->{'handle'}), $| = 1) [0] ); # unbuffer new handle
  }

#============================================================================#

# RC 4/1/97 - since user may not always declare same control_vars name
sub find_cvars {
  my ($self) = shift;
  return $main::vars
     if defined $main::vars && (ref $main::vars eq "control_vars");
  return $main::cvars
     if defined $main::cvars && (ref $main::cvars eq "control_vars");
  foreach $key (keys %main::) {
    next if $key =~ /\W/;  # avoid eval of files, pkgs, weird stuff
    my $res = eval "\$main::$key";
    if (ref $res  &&  ref $res eq "control_vars") {
      return $res;
      }
    }
  $self->warn ('prune_logfile CANNOT FIND CONTROL VARS!!!');   # only OHS?
  return '';
  }

#============================================================================#

1;



