#!/pkg/bin/perl -w

# Functions to use
#
#  init($TCP_PORT,$UDP_PORT,$MAX_CLIENT,$CLOSE_MESSAGE);
#  tcp_send($HOST,$HIS_TCP_PORT,$HEADER,@BODY);
#  udp_send($HOST,$HIS_UDP_PORT,$HEADER,@BODY);
#  file_send($FILENAME,$HEADER,@BODY);
#  prog_send($COMMAND,$HEADER,@BODY);
#  ($HOST,$HIS_UDP_PORT,$HEADER,@BODY) = udp_recv($BLOCKING);
#  ($HOST,$HIS_TCP_PORT,$HEADER,@BODY) = tcp_recv($BLOCKING);
#  ($proto,$host,$port,$header,@body) =
#     mesg_recv($BLOCKING,$NO_TCP,$NO_UDP,$HOST,$PORT);
#  tcp_close($HOST,$PORT);
#  file_close($FILENAME);
#  prog_close($COMMAND);
#  shutdown();
#  read_file($FILENAME);
#
# where init is to be called exactly once, when this package is first
# included.
#
# Example usage:
#
#   # Process for receiving udp messages, processing them, and
#   # sending a response back.  Exact same code could be used
#   # with udp_recv and udp_send changed to tcp_receive and
#   # tcp_send.  Also, 'nonblocking' could be changed to
#   # 'blocking', which would cause this process to never
#   # terminate, but instead to wait when no messages were
#   # available.
#   
#   #!/usr/bin/perl
#   use Comm;
#   init('sierra',3000,3000,5);
#   while(($host,$port,$header,@body) = udp_recv('nonblocking')) {
#     ($response_header,@response_body) = junk_producer($header,@body);
#     udp_send($host,$port,$response_header,@response_body);
#   }

# VARIABLES, HANDLES, AND CONSTANTS IN THIS PACKAGE
# 
#   TCP_S:  File handle on which TCP connections are
#           accepted.  Bound in function become_tcp_server.
#
#   UDP_S:  File handle on which UDP packets are received.
#           Bound in function become_udp_server.
#
#   bound{X}:  Hash registering all bound file handle
#              names.  bound{X} = 'tcp' iff X is a
#              tcp socket handle.  bound{X} = 'udp'
#              iff X is a udp socket handle.
#
#   descriptor{X}:  Hash storing the file descriptor for
#                   file handle named X.
#
#   handle{X}:  Hash storing the file handle for file
#               descriptor X.
#
#   %clients:  $clients{FH} = 'tcp' or 'udp' iff the file
#              handle FH is that of an open client.
#              This value is set in function
#              accept_tcp_client and in become_tcp_client. 
#
#   $sbuff{FH}:  String containing all that has been read
#                from file handle FH but not yet returned
#                as a packet.
#
#   %handle_to_host:  $handle_to_host{FH} = name of host to
#                     which FH connects us.
#
#   %handle_to_port:  $handle_to_port{FH} = port number on other
#                     side of connection associated with file
#                     handle FH
#
#   %hostport_to_handle:  $hostport_to_handle{HOST,PORT} =
#                         file handle of connection to the
#                         process running on HOST and PORT
#
#   $HOSTNAME:  Name of our host
#
#   $TCP_PORT:  Port number on which we accept tcp connections
#
#   $tcp_next:  'tcp'.$tcp_next is the name of the next file
#               handle to be generated for a tcp connection.
#
#   $UDP_PORT:  Port number on which we receive udp messages
#
package Comm;
use Fcntl;
use Socket;

# Four optional arguments:
#   Assumes the first argument is the port on which
#   we will accept tcp connections.
#
#   Assumes the second argument is the port on which
#   we will accept udp connections.
#
#   Assumes third argument, $max_client, states the
#   maximum number of tcp clients to accept.
#
#   Assumes fourth argument, $close_message, true if
#   we want a grids packet telling us when connections
#   have closed, undef else.
#
# This sets constants for this package.  This should
# be run when package is first used.
#
# If no arguments are given, we will not be able to receive UDP 
# messages at all, and we will only be able to receive
# TCP responses to messages we send out.  (No unsolicited
# connections will be allowed.) 
# 
# If only the first argument is given, we will reserve that port for
# TCP connections.  We will be able to send and receive TCP
# messages.  However, we will neither be able to send nor receive
# UDP messages.
#
# If first two arguments are given, we will bind both a TCP and
# a UDP port and will be able to send and receive both TCP and
# UDP messages.
#
# Either of the first two arguments may be set to 0, indicating
# that we want the function to pick an available port for us.
#
# The fourth argument defaults to 5 if not given.
#
# Returns ($TCP_PORT,$UDP_PORT) on success.
#
sub init {
  ($TCP_PORT,$UDP_PORT,$MAX_CLIENT,$CLOSE_MES) = @_;
  $INIT = 1;				# init has been run
  # $DEBUG = 'true';

  ########## idea taken from Jim Hoagland's 2/10/9? code
  # set up AF_INET and SOCK_STREAM and SOCK_DGRAM

  # DONT USe h2ph ANYMORE.  We use "use Socket" now.
  # just in case 'h2ph' wasn't run ...
  # unless(defined(&AF_INET)) {
  #  eval `/usr/bin/cat /usr/include/sys/socket.h | /pkg/bin/h2ph`;
  # } 

  unless (defined AF_INET) {
      print STDERR "Warning, AF_INET = 2 assumed\n";
      eval "sub AF_INET {2;}";
  }
  unless (defined SOCK_STREAM) {
    my $uname = `/bin/uname -a` if ( -e "/bin/uname" );
    $uname = `/usr/bin/uname -a` if ( -e "/usr/bin/uname" );
    if($uname =~ /SunOS \S+ 5\./) {  
      eval "sub SOCK_STREAM {2;}";
      eval "sub SOCK_DGRAM {1;}";
      print STDERR "Warning, SOCK_STREAM= 2 and SOCK_DGRAM= 1 assumed!\n";
    } else {
      eval "sub SOCK_STREAM {1;}";
      eval "sub SOCK_DGRAM {2;}";
      print STDERR "Warning, SOCK_STREAM= 1 and SOCK_DGRAM= 2 assumed!\n";
    }
  }
  ########## end of stolen idea

  $iterate_open = undef;

  $MAX_CLIENT = 5 unless $MAX_CLIENT;	# Tweekable

  $CONNECT_TIME = 10;	# Number of seconds we will wait for a client
			# to send host,port data before closing their
			# connection.

  $tcp_next = 0;	# Tweekable, though no reason to
  $file_next = 0;	# Tweekable, though no reason to
  $iterate_next = 0;	# Tweekable, though no reason to
  $prog_next = 0;	# Tweekable, though no reason to

  $undef_next = 0;	# Tweekable, though no reason to

  $READSIZE = 1000;     # Tweekable.  The number of bytes
                        # to read from a socket at a time.

  $max_length = 65536;  # Tweekable.  Length of longest allowed
                        # udp message.

  $ADDRTYPE = 2;	# Kludge.  Used in udp_recv to convert
                        # host addr to host name

  $SOCKADDR = 'S n a4 x8';

  ($HOSTNAME,$ADDRTYPE,$THISADDR) = &host_name;
  if(!$bound{'TCP_S'} && defined($TCP_PORT)) {
    return 'tcp_bad' unless(&become_tcp_server($TCP_PORT,$MAX_CLIENT));
  }
  if(!$bound{'UDP_S'} && defined($UDP_PORT)) {
    return 'udp_bad' unless(&become_udp_server($UDP_PORT));
  }
  ('ok',$TCP_PORT,$UDP_PORT);
}

# host_name returns (name,address type,address) for the
# machine we're running on, where name is fully qualified.
# if it can't determine these things, it dies.
#
sub host_name {
  my($hostname,$addrtype,$hostaddr);

  $hostname = `/bin/uname -n` if ( -e "/bin/uname" );
  $hostname = `/usr/bin/uname -n` if ( -e "/usr/bin/uname" );
  chomp($hostname);

  if(length($hostname)) {
    ($hostname,$_,$addrtype,$_,$hostaddr) = gethostbyname($hostname);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.edu$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.gov$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.com$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.org$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.int$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.net$/);
  
    ($hostname,$_,$addrtype,$_,$hostaddr) = gethostbyaddr($hostaddr,$addrtype);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.edu$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.gov$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.com$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.org$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.int$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.net$/);
  }
  
  chomp($hostname = `hostname`);
  if(length($hostname)) {
    ($hostname,$_,$addrtype,$_,$hostaddr) = gethostbyname($hostname);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.edu$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.gov$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.com$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.org$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.int$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.net$/);
  
    ($hostname) = gethostbyaddr($hostaddr,$addrtype);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.edu$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.gov$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.com$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.org$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.int$/);
    return ($hostname,$addrtype,$hostaddr) if($hostname =~ /.net$/);
  }

  $hostname= "$hostname.cs.ucdavis.edu";
  warn "host_name could not determine the fully qualified hostname; guessing $hostname\n";
  return ($hostname,$addrtype,$hostaddr);
}


# Assumptions:
#   Assumes the first argument is the host name of the
#   recipient.
#
#   Assumes the second argument is the port number of
#   the recipient.
#
#   Assumes the third argument is the GrIDS common
#   packet header.
#
#   Assumes the remaining arguments are one or more
#   strings to be sent as a list
#
# Tries to send the packet made from the third and
# on arguments to the destination.  Creates a socket if
# none exists.  Returns 'true' if the packet was sent,
# else an error code.  If init() hasn't been run yet,
# undef is returned.  If we try to make a connection to
# the recipient and the socket call fails, "socket: $!"
# is returned; if the bind call fails, "bind: $!" is
# returned; if the connect call fails, "connect: $!" is
# returned.  If we cannot locate the host,
# "host: cannot locate" is returned.
#
sub tcp_send {
  my($host,$port,$message,$handle);

  unless(defined($INIT)) {
    warn "Comm::tcp_send called before call to Comm::init\n";
    return undef; 
  }

  ($host) = gethostbyname($_[0]); shift;
  return "host: cannot locate" unless defined($host);

  $port = shift;
  return "port: not defined" unless defined($port);

  &encode_packet(\$message,\@_);

  # get whatever handle we may have for sending this message
  $handle = $hostport_to_handle{$host,$port};

  # Has destination closed the connection?  If so, read all
  # the data left on the connection and close it.  (This is
  # done by the following unintuitive line for the following
  # reasons.  If the destination has closed the connection,
  # then is_ready_to_read will return true, but there will
  # be no data there.  fill_tcp_buff will recognize this
  # condition and close the connection for us.  Hence, in
  # the next line, "unless($bound{$handle})", we will discover
  # that $handle is not bound (complements of fill_tcp_buff)
  # and create a new connection.)  Also, as a side effect,
  # read and buffer for later anything the destination has
  # sent us.
  if(defined($handle) && defined($bound{$handle}) && &is_ready_to_read($handle)) {
    &fill_tcp_buff($handle);
  }

  # if we have no handle, make a socket
  unless(defined($handle) && defined($bound{$handle})) {
    $handle = 'tcp'.$tcp_next;
    
    # If a connection can't be opened, the message can't be sent.
    unless(&become_tcp_client($host,$port,$handle)) {
      warn "ERROR_CODE undefined in tcp_send\n" unless(defined($ERROR_CODE));
      return $ERROR_CODE;
    }

    # code sanity check
    unless(defined($hostport_to_handle{$host,$port})) {
      die "hostport to handle undefined after call to become_tcp_client\n";
    }
    unless($handle eq $hostport_to_handle{$host,$port}) {
      die "hostport to handle wrong after call to become_tcp_client\n";
    }

    # since connection was opened, update $tcp_next
    $tcp_next++;
  }

  my($bytes_sent,$total_bytes_sent,$message_length);
  $message_length = length($message);

  $total_bytes_sent = 0;
  until($total_bytes_sent == $message_length) {
    if(&is_ready_to_write($handle,'blocking')) {
      $sent = send($handle,$message,0);
      if(defined($sent)) {
        $total_bytes_sent += $sent;
        $message =~ /^[\x0-\xff]{$sent}([\x0-\xff]*)$/;
        $message = $1;
      } else {
        warn "tcp_send: handle not ready\n";
        return 'handle not ready';
      }
    }
  }
  return 'true';
}

# Assumptions:
#   Assumes first argument is the destination host name
#
#   Assumes second arguement is the destination port number
#
#   Assumes third arguement is the GrIDS common packet header
#
#   Assumes remaining arguments are the body list
#
# Attempts to send the GrIDS packet in a UDP packet.  If successful,
# returns 'true', else undef.  Remember: successful UDP sending of a
# packet does not guarantee successful receipt on the other side.
#
sub udp_send {
  my($host,$port,$message,$name,$thataddr,$that);

  unless(defined($INIT)) {
    warn "Comm:udp_send called before call to Comm::init\n";
    return undef;
  }

  unless($UDP_PORT) {
    warn "Comm::udp_send called by module who isn't a UDP server!\n";
    return undef;
  }

  $host = shift;
  unless(defined($host)) {
    warn "host undefined in udp_send\n";
    return undef;
  }
  $port = shift;
  unless(defined($port)) {
    warn "port undefined in udp_send\n";
    return undef;
  }
  &encode_packet(\$message,\@_);

  # get hostname and address from cache if possible, else from 
  # a system call
  ($name,$_,$_,$_,$thataddr) = gethostbyname($host);
  unless(defined($name)) {
    warn "name undefined for host $host in udp_send\n";
    return undef;
  }
  unless(defined($thataddr)) {
    warn "thataddr undefined for host $host in udp_send\n";
    return undef;
  }
    
  $that = pack($SOCKADDR,AF_INET,$port,$thataddr);
  if(send(UDP_S,$message,0,$that)) {
    return 'true';
  } else {
    warn "send failure: $!\n";
    return undef;
  }
}

# Assumptions:
#   Assumes first argument is the name of the file to write to.
# 
#   Assumes second argument is the header.
#
#   Assumes remaining arguments are the body.
#
sub file_send {
  my $filename = shift;
  my($message,$handle);

  &encode_packet(\$message,\@_);

  # get whatever handle we may have for writing this message
  $handle = $file_to_handle{$filename};

  # if we have no handle for this file, make one
  unless(defined($handle) && defined($bound{$handle})) {
    $handle = 'file'.$file_next;
    
    # If a file can't be opened, the message can't be written.
    unless(&become_file_client($filename,$handle)) {
      warn "ERROR_CODE undefined in file_send\n" unless(defined($ERROR_CODE));
      return $ERROR_CODE;
    }

    # code sanity check
    unless(defined($file_to_handle{$filename})) {
      die "file to handle undefined after call to become_file_client\n";
    }
    unless($handle eq $file_to_handle{$filename}) {
      die "file to handle wrong after call to become_file_client\n";
    }

    # since file was opened, update $file_next
    $file_next++;
  }

  print $handle $message;
  return 'true';
}

# Assumptions:
#   Assumes first argument is the name of the UNIX command to write to.
# 
#   Assumes second argument is the header.
#
#   Assumes remaining arguments are the body.
#
sub prog_send {
  my $cmd = shift;
  my($message,$handle);

  &encode_packet(\$message,\@_);

  # get whatever handle we may have for writing this message
  $handle = $prog_to_handle{$cmd};

  # if we have no handle for this prog, make one
  unless(defined($handle) && defined($bound{$handle})) {
    $handle = 'prog'.$prog_next;
    
    # If the command can't be opened (ran), the message can't be written.
    unless(&become_prog_client($cmd,$handle)) {
      warn "ERROR_CODE undefined in prog_send\n" unless(defined($ERROR_CODE));
      return $ERROR_CODE;
    }

    # code sanity check
    unless(defined($prog_to_handle{$cmd})) {
      die "prog to handle undefined after call to become_prog_client\n";
    }
    unless($handle eq $prog_to_handle{$cmd}) {
      die "prog to handle wrong after call to become_prog_client\n";
    }

    # since prog was opened, update $prog_next
    $prog_next++;
  }

  print $handle $message;
  return 'true';
}

# Assumptions:
#   Assumes first argument 'blocking', 'nonblocking',
#   or the number of second we're willing to block.
#
#   Assumes second argument true if we want udp messages
#   only, undef else
#
#   Assumes third argument true if we want tcp messages
#   only, undef else
#
#   Assumes fourth and fifth arguments to be the host and
#   port we insist on receiving from, if we so insist.
#   undef,undef else.
#
#   Alternative form:  The fourth argument can also be a ref
#   to a list of lists of the form ($host,$port).  These
#   are the hosts and ports we
#   will accept replies from.  In this scenario, the fifth
#   argument is ignored.
#
#   Second Alternate form:  The fourth and fifth arguments
#   may also be undef,FILENAME, respectively, if you wish
#   to read from an already opened file.

#
# Returns ($proto,$host,$port,$header,@body) if available
# in the allowed time.  Note: blocking calls will always
# return a message unless the call was interupted by
# a signal (such as one from the control variables code).
#
sub mesg_recv {
  my($blocking,$udp_only,$tcp_only,$host,$port) = @_;
  my(@handles,$ready,$block_time,$start_time,$stop_time);

  unless(defined($INIT)) {
    warn "Comm::mesg_recv called before Comm::init!\n";
    return undef;
  }

  $start_time = time;
  if(defined($blocking)) {
    if($blocking eq 'blocking') {
      $blocking = undef;
    } elsif($blocking eq 'nonblocking') {
      $blocking = 0;
    } elsif($blocking =~ /^[0-9]*$/) {
      $stop_time = $start_time + $blocking;
    } else {
      $blocking = undef;
    }
  }

  # Note: this do loop is not a busy wait because the call to
  # are_ready_to_read will block all but the first time.
  # Hence, we only loop when we receive something (though not
  # necessarily what we're waiting for).
  $WAS_INTERUPTED = 'false';
  $block_time = 0;
  do {

    @handles = ();

    if(defined($TCP_PORT)) {
      while(&accept_tcp_client('nonblocking')) {; }
    }

    push(@handles,keys(%clients));
    push(@handles,'UDP_S') if $UDP_PORT;
    push(@handles,'TCP_S') if $TCP_PORT;

    foreach $ready (&are_ready_to_read($block_time,@handles)) {
      if($UDP_PORT && ($descriptor{$ready} == $descriptor{'UDP_S'})) {
        # Since UDP_S is ready for reading, fill_udp_buff
        # should not fail.
        unless(&fill_udp_buff) {
          warn "In Comm::mesg_recv, fill_udp_buff failed\n";
        }
      } elsif ($descriptor{$ready} == $descriptor{'TCP_S'}) {
        &accept_tcp_client('nonblocking',1);
      } else {
        &fill_tcp_buff($ready);
      }
    }

    unless($tcp_only || $udp_only) {
      # if a specific file name was given, use it
      if(!defined($host) && defined($port)) {
        ($host,$port,$header,@body) = &recv_file_buff($port);
        if(defined($header)) {
          return('file',$host,$port,$header,@body);
	} else {
          return undef;
        }
      }

      # if no file name given, try for each filename we know about
      foreach( keys %iterate_to_handle ) {
        ($host,$port,$header,@body) = recv_file_buff($_);
        if(defined($header)) {
          return('file',$host,$port,$header,@body);
        }
      }
    }
    unless($udp_only) {
      ($host,$port,$header,@body) = 
		&multi_recv_wrapper(\&recv_tcp_buff,$host,$port);
      if(defined($header)) {
        return('tcp',$host,$port,$header,@body);
      }
    }
    unless($tcp_only) {
      ($host,$port,$header,@body) = 
		&multi_recv_wrapper(\&recv_udp_buff,$host,$port);
      if(defined($header)) {
        return('udp',$host,$port,$header,@body);
      }
    }

    # set up to block for the amount of time we have left
    if(defined($stop_time)) {
      $blocking = $stop_time - time;
      $blocking = 0 unless($blocking > 0); # fix if negative
    }
    $block_time = $blocking;
  } while((!defined($blocking) || $blocking > 0) && ($WAS_INTERUPTED ne 'true'));
  $WAS_INTERUPTED = undef;
  undef;
}

# Function written so that we can receive from one of a number of hosts
# MARK - you may not want to do things this way - I was just trying to
# make minimal changes to the existing code -- Stuart.

sub multi_recv_wrapper
{
  my($funcref,$host,$port) = @_;
  my($host1,$port1);

  unless(defined($host) && (ref $host eq 'ARRAY'))
   {
    # Backward compatability mode - just call the function as before
    print STDERR "multi_recv - probable error in $0" 
		if (defined($host) && !defined($port));
    return &$funcref($host,$port);
   }

  # This is the new part.  We are willing to receive from any of the
  # hosts in the list.  We use a variable wrapper_index to try to guess
  # the best potential sender first on the assumption that senders will
  # send in the same order that they appear in our arguments.

  my @array = @$host; 
  my $npairs = @array;
  my $result;
  my ($index,$pair,$raw_index);
  $Comm::wrapper_index = 0 unless defined $Comm::wrapper_index;
  # Cycle through the list of potential recipients
  foreach $raw_index (0..$npairs-1)
   {
    $index = ($Comm::wrapper_index+$raw_index)%$npairs;
    my($mhost,$mport) = @{$array[$index]};
    ($host1,$port1,$header,@body) = &$funcref($mhost,$mport);
    if(defined($header))
     {
      # Got something - return it.
      $Comm::wrapper_index += $raw_index+1; # try to guess most efficient place
      return ($mhost,$mport,$header,@body);
     }
   }
  return ($host,$port);
}

# Assumptions:
#   Assumes one argument, a tcp file handle which is ready to be read from
#
# Returns the total number of bytes read.  If 0, then this
# handle was closed.  If system call read was interupted,
# global variable $WAS_INTERUPTED will be set to 'true'.
#
sub fill_tcp_buff {
  my($handle) = @_;
  my $host = $handle_to_host{$handle};
  my $port = $handle_to_port{$handle};

  return 0 unless(defined($bound{$handle}));
  die "host undefined in fill_tcp_buff\n" unless defined($host);
  die "port undefined in fill_tcp_buff\n" unless defined($port);

  my $key = "$host\xff$port";

  my($buff,$total_bytes_read);
  $total_bytes_read = 0;
  do {
    $buff = "";
    unless(defined(recv($handle,$buff,$READSIZE,0))) {
      $WAS_INTERUPTED = 'true' unless($! =~ /^Resource temporarily unavailable/);
    } else {
      $total_bytes_read += length($buff);
      unless(defined($tcp_buff{$key})) {
        $tcp_buff{$key} = "";
      }
      $tcp_buff{$key} .= $buff;
    }
  } while((length($buff) > 0) && ($WAS_INTERUPTED ne 'true'));

  # if read produced no output, then the connection is closed
  if($total_bytes_read == 0) {
    unless(defined($key) && defined($tcp_buff{$key}) && 
          exists($tcp_buff{$key}) && length($tcp_buff{$key}) > 0) {
      if(defined($handle_to_host{$handle}) && defined($handle_to_port{$handle})) {
        &remove_from_tcp_order($handle);
      } else {
        print STDERR "Tell Mark Dilger: Can't remove from tcp_order $handle\n";
        print STDERR "(This isn't an error that should affect you.)\n";
      }
    }
    &close_handle($handle);
  }
  $total_bytes_read;
}

# Assumptions:
#   Assumes no arguments
#
#   Assumes UDP_S is ready to be read from
#
sub fill_udp_buff {
  my($handle,$from,$msg,$port,$src_addr,$host);

  $handle = shift;
  $msg = "";
  $from = recv(UDP_S,$msg,$max_length,0);

  if($from) {
    ($_,$port,$src_addr) = unpack($SOCKADDR,$from);
    $host = gethostbyaddr($src_addr,$ADDRTYPE);
    push(@{$udp_buff{$host."\xff".$port}},$msg);
    return 'true';
  }
  undef;
}

# Assumptions:
#   Assumes first two arguments are the host and port we
#   wish to receive from.  If undef, assumes we don't care
#   who we receive from.
#
# Returns ($host,$port,$header,@body) if available, undef else
#
sub recv_tcp_buff {
  my($host,$port,$key,@message);
  ($host,$port) = @_;

  if(defined($host) && defined($port)) {
    $key = $host."\xff".$port;
    if(defined($tcp_buff{$key})) {
      $tcp_buff{$key} = &decode_packet(\$tcp_buff{$key},\@message);
      return ($host,$port,@message) if defined($message[0]);
    }
  }
  else {
    my $count;
    for($count = 0; $count <= $#tcp_order; $count++) {
      $key = $tcp_order[0];
      $tcp_buff{$key} = &decode_packet(\$tcp_buff{$key},\@message);
      if(defined($message[0])) {
        $key =~ /([\x0-\xfe]*)\xff([\x0-\xfe]*)/;
        ($host,$port) = ($1,$2);
        my $handle = $hostport_to_handle{$host,$port};
        $key = shift(@tcp_order); push(@tcp_order,$key);
        unless(defined($key)){
          unless(defined($tcp_buff{$key})){
            unless ((length($tcp_buff{$key}) > 0) || (defined($bound{$handle}))) {
              # if we've read the last message from a closed connection...
              &remove_from_tcp_order($host,$port);
            }
          }
        }
        return ($host,$port,@message);
      }
      $key = shift(@tcp_order); push(@tcp_order,$key);
    }
  }
  undef;
}

# Assumptions:
#   Assumes first argument is the name of the file to read from.
#
#   Assumes we have opened the file with read_file.
#
# Reads the next message from the file and returns it, if there is one.
#
sub recv_file_buff {
  my($filename) = $_[0];
  my(@message) = undef;

  return undef unless $iterate_open{$filename};
  if(read_iterate_file($filename))
  {
    $iterate_buff{$filename} = &decode_packet(\$iterate_buff{$filename},\@message);
    return (undef,undef,@message) if(defined($message[0]));
  }
  undef;
}

# Assumptions:
#   Assumes first two arguments are the host and port we
#   wish to receive from.  If undef, assumes we don't care
#   who we receive from.
#
# Returns ($host,$port,$header,@body) if available, undef else
#
sub recv_udp_buff {
  my($host,$port,$key,@message,$msg,$error); 
  ($host,$port) = @_;

  if(defined($host) && defined($port)) {
    $key = $host."\xff".$port;
    do {
      $msg = pop(@{$udp_buff{$key}});
      if(defined($msg)) {
        $error = &decode_packet(\$msg,\@message);
        warn "corrupted udp message received.  (Dumping it.)\n" if(defined($error));
      }
    } while(defined($error) && defined($msg));
    return ($host,$port,@message) if defined($message[0]);
  } else {
    foreach $key (keys(%udp_buff)) {
      $key =~ /([\x0-\xfe]*)\xff([\x0-\xfe]*)/;
      $host = $1;
      $port = $2;
      do {
        $msg = pop(@{$udp_buff{$key}});
        if(defined($msg)) {
          $error = &decode_packet(\$msg,\@message);
          if((defined($error)) && (length($error) > 0)) {
            warn "corrupted udp message received.  (Dumping it.)\n";
          }
        }
      } while(defined($error) && defined($msg));
      return ($host,$port,@message) if defined($message[0]);
    }
  }
  undef;
}

# Assumptions:
#   Assumes only argument is 'blocking', 'nonblocking',
#   or the number of seconds we're willing to block.
#
# Returns (SOURCE_HOST,SOURCE_PORT,$header,@body) on 
# success.  If 'nonblocking' and no message is available,
# returns undef.
#
sub udp_recv {
  my $blocking = shift;
  my($proto,$host,$port,$header,@body) = &mesg_recv($blocking,1,undef,undef,undef);
  ($host,$port,$header,@body);
}

# Assumptions:
#   Assumes first argument is 'blocking', 'nonblocking',
#   or the number of second we're willing to block.
#
# Returns (SOURCE_HOST,SOURCE_PORT,$header,@body).
#
sub tcp_recv {
  my $blocking = shift;
  $blocking = 'nonblocking' unless($blocking eq 'blocking');
  my($proto,$host,$port,$header,@body) = &mesg_recv($blocking,undef,1,undef,undef);
  ($host,$port,$header,@body);
}


# Assumptions:
#   Assumes no arguments.
#
# All file handles used by Comm will be closed by this function.
# No more messages can be sent without another call to init().
# However, any messages in our buffer will still be available
# for reading.  One should be careful though not to read in
# 'blocking' mode, for one would never get out if no messages
# are left to read.  (Also, 'nonblocking' mode is sufficient to
# read any messages in the buffer.)
#
sub shutdown {
  my($handle,$filename,$cmd);
  foreach $handle (keys(%clients)) {
    &close_handle_gracefully($handle);
  }
  foreach $filename (keys(%file_to_handle)) {
    &file_close($filename);
  }
  foreach $cmd (keys(%prog_to_handle)) {
    &prog_close($cmd);
  }

  if($TCP_PORT) {
    close('TCP_S');
    delete $bound{'TCP_S'};
    delete $handle{$descriptor{'TCP_S'}};
    delete $descriptor{'TCP_S'};
  } 
  if($UDP_PORT) {
    close('UDP_S');
    delete $bound{'UDP_S'};
    delete $handle{$descriptor{'UDP_S'}};
    delete $descriptor{'UDP_S'};
  }  

  foreach(keys %iterate_to_handle) {
    close_iterate_file($_);
  }
}

# Assumptions:
#   Assumes the first argument is the name of a file from
#   which to receive messages.
#
# Will open the file for iterative reading, though no reading will be done here.
#
sub read_file {
  my($filename) = @_;
  $iterate_to_handle{$filename} = 'iterate'.$iterate_next++;
  return undef unless(open($iterate_to_handle{$filename},$filename));
  $iterate_open{$filename} = 1;
}

# Assumptions:
#   Assumes the first argument is the name of the file to close.
# 
# Will close the file for iterative reading,
#
sub close_iterate_file {
  close($iterate_to_handle{$_[0]});
  $iterate_open{$_[0]} = undef;
}

# Assumptions:
#   Assumes the function read_file has been called for this filename.
#
#   Assumes this file has not been closed.
#
# Will read the next message from file into buffer $iterate_buff.
#
sub read_iterate_file {
  my($buffer)='';
  my($length)='';
  my($filename)=$_[0];

  $iterate_buff{$filename} = "";
  while(read($iterate_to_handle{$filename},$buffer,1) != 0 && $buffer ne ' ') {
    $length .= $buffer;
  }
  return undef if($length eq '');
  read($iterate_to_handle{$filename},$buffer,$length);
  $iterate_buff{$filename} .= $length . " " . $buffer;
}

# Assumptions:
#   Assumes first argument is a file name
#
# If any such file is open it will be closed and 'true' returned, else undef.
#
sub file_close {
  my $filename= shift;
  my $handle;

  if($handle = $file_to_handle{$filename}) {
    delete $file_to_handle{$filename};
    close $handle;
    return 'true';
  } 
  undef;
}

# Assumptions:
#   Assumes first argument is a command (the same as it was open with)
#
# If any such command is running, its STDIN will be closed and 'true'
# returned, else undef.
#
sub prog_close {
  my $cmd= shift;
  my $handle;

  if($handle = $prog_to_handle{$cmd}) {
    delete $prog_to_handle{$cmd};
    close $handle;
    return 'true';
  } 
  undef;
}

# Assumptions:
#   Assumes first argument is a host name
#
#   Assumes second argument is a port number
#
# If any connection exists between us and host,port,
# it will be closed and 'true' returned, else undef.
#
sub tcp_close {
  my($handle,$host,$port);

  unless(defined($INIT)) {
    warn "Comm:tcp_close called before call to Comm::init\n";
    return undef;
  }

  ($host) = gethostbyname($_[0]);
  $port = $_[1];
  if($handle = $hostport_to_handle{$host,$port}) {
    delete $hostport_to_handle{$host,$port};
    &close_handle_gracefully($handle);
    return 'true';
  } 
  if($handle = $hostport_to_handle{$_[0],$port}) {
    warn "Error: hostport_to_handle index by partially qualified name $_[0]\n";
  }
  undef;
}

# Assumptions:
#   Assumes one argument, a tcp socket handle which is to be
#   closed.
#
# If handle is bound, closes it without loss of any messages that
# might be pending, and returns 'true', else returns undef.
#
sub close_handle_gracefully {
  my $handle = shift;
  if(defined($bound{$handle})) {
    &fill_tcp_buff($handle) if(&is_ready_to_read($handle,0));
    &close_handle($handle);
  } 
}

# Assumptions:
#   Assumes one argument, a file handle which is to be
#   closed.
#
# If handle is bound, closes it and returns 'true',
# else returns undef.
#
sub close_handle {
  my($host,$port,$key);
  my $handle = shift;
  if($bound{$handle}) {
    close($handle);
    delete $bound{$handle};
    delete $handle{$descriptor{$handle}};
    delete $descriptor{$handle};
    delete $clients{$handle};
    delete $handle_to_host{$handle};
    delete $handle_to_port{$handle};
    'true';
  }
}

# Assumptions:
#   Assumes the first argument is the port number on which
#   to accept connections.
#
#   Assumes the second argument is the maximum number of
#   clients to have queued.  (This is passed to 'listen').
#   Or if it is not defined, that $MAX_CLIENT is to be
#   used for this value.  Zero is not allowed as a value
#   for the maximum number of clients.  If supplied,
#   $MAX_CLIENT will be used.
#
#   Assumes that "use Fcntl" is stated at the top of this
#   module.
#
#   Assumes that init() has been run.
#
# This function sets up file handle TCP_S as a generic socket
# waiting to make connections.  This should be called before
# any calls to accept_tcp_client().  This should only be called
# once.  Returns 'true' on success, else undef.
#
sub become_tcp_server {
  my($port_num,$max_client,$proto,$this);
  ($port_num,$max_client) = @_;
    $max_client = $MAX_CLIENT unless(defined($max_client));
 
  unless ($bound{'TCP_S'}) {
    ($_,$_,$proto) = getprotobyname('tcp');
    unless(defined($proto)) {
      warn "in become_tcp_server, could not determine protocall 'tcp'\n";
      return undef;
    }
    
    $this = pack($SOCKADDR,AF_INET,$port_num,"\0\0\0\0");
    unless(defined($this)) {
      warn "in become_tcp_server, could not pack our address\n";
      return undef;
    }
    unless(socket(TCP_S,AF_INET,SOCK_STREAM,$proto)) {
      warn "in become_tcp_server, socket: $!\n";
      return undef;
    }
    unless(bind(TCP_S,$this)) {
      my $save_error = $!;	# call to uname will overwrite $!
      my $errmsg = '^G <'.&host_name();
      chomp $errmsg;
      $errmsg .= ":$port_num>";
      warn "in become_tcp_server, bind: $errmsg $save_error\n";
      close(TCP_S);
      return undef;
    }
    ($_,$TCP_PORT,$_) = unpack($SOCKADDR,getsockname('TCP_S'));
    if(($port_num > 0) && ($TCP_PORT != $port_num)) {
      warn "bound to wrong port number in &Comm::become_tcp_server\n";
    }
    # fcntl(TCP_S,F_SETFL,O_NDELAY) || die "Can't make non-blocking\n";
    unless(listen(TCP_S,$max_client)) {
      warn "in become_tcp_server, listen: $!\n";
      close(TCP_S);
      return undef;
    }
    $bound{'TCP_S'} = 'tcp';
    $handle{fileno(TCP_S)} = 'TCP_S';
    $descriptor{'TCP_S'} = fileno(TCP_S);
  }
}

# Assumptions:
#   Assumes the first argument is the port number on which
#   to accept connections.
#
#   Assumes that "use Fcntl" is stated at the top of this
#   module.
#
#   Assumes that init() has been run (or that this is run
#   by init()).
#
# This function sets up file handle UDP_S as a generic socket
# to receive UDP packets.  This should only be called once.
# Returns true if we are a UDP server after call, undef else.
#
sub become_udp_server {
  my($port_num,$proto,$this,$port_num);
  $port_num = shift;

  unless(defined($port_num)) {
    warn "no port number passed to &Comm::become_udp_server\n";
    return undef;
  }
  unless ($bound{'UDP_S'}) {
    ($_,$_,$proto) = getprotobyname('udp');
    $this = pack($SOCKADDR,AF_INET,$port_num,"\0\0\0\0");
    unless(socket(UDP_S,AF_INET,SOCK_DGRAM,$proto)) {
      warn "socket UDP_S: $!\n";
      return undef;
    }
    unless(bind(UDP_S,$this)) { 
      warn "bind: $!\n";
      close(UDP_S);
      return undef;
    }
    ($_,$UDP_PORT,$_) = unpack($SOCKADDR,getsockname('UDP_S'));
    if(($port_num > 0) && ($UDP_PORT != $port_num)) {
      warn "bound to wrong port number in &Comm::become_udp_server\n";
    }
    
    # fcntl(UDP_S,F_SETFL,O_NDELAY) || die "Can't make non-blocking\n";
    $bound{'UDP_S'} = 'udp';
    $handle{fileno(UDP_S)} = 'UDP_S';
    $descriptor{'UDP_S'} = fileno(UDP_S);
    return 'true';
  }
  else {
    warn "call made to become_udp_server, but we're already a server\n";

    # returns 'true' to indicate that we are a server at end of call
    return 'true';
  }
}

# Assumptions:
#   Assumes that TCP_S is a generic file handle bound to the port
#   on which we are accepting new clients.
#
#   Assumes the first argument is 'blocking' or 'nonblocking'.  IF
#   non-blocking, a client will only be accepted if one is waiting.
#   Otherwise, we block on the accept.  If no argument is given,
#   defaults to non-blocking.
#
#   Assumes if second argument is defined that this function is not
#   to check to see if TCP_S is ready to read, but merely to trust
#   that it is.
#
#   Assumes init() has been called once and that $tcp_next is the
#   suffix to use for the next file handle name.
#
# In non-blocking mode, this function checks to see if a client is
# waiting to be accepted and accepts one if so.  It is accepted
# through the generic socket TCP_S and a new socket is created for
# it.  If in blocking mode, this function waits until a client
# tries to connect, then behaves the same as in non-blocking mode.
# If a client is accepted, it returns its handle, else undef.
#
sub accept_tcp_client {
  my($fhname,$blocking,$temp,$host,$port,$mesg,$addr,$no_check);
  ($blocking,$no_check) = @_;
  die "accept_tcp_client called before init!\n" unless(defined($INIT));

  die "Upon entering accept_tcp_client, $mesg" unless(&am_tcp_server(\$mesg));

  # Accept a client if one is ready, or wait for one if in "blocking" mode
  $blocking = 'nonblocking' unless $blocking eq 'blocking';
  if($no_check || &is_ready_to_read(TCP_S,$blocking)) {
    # create new file handle name and accept client on that handle
    $fhname = 'tcp'.$tcp_next;
    $addr = &accept_connection($fhname,\$mesg);

    # die if accept call did not return an address
    die "In accept_tcp_client, $mesg" unless(defined($addr));

    # GET CLIENT ADDRESS, HOST, AND NAME
    ($host,$port) = &handshake_with_new_client($fhname,$addr);

    # IF THEY FAILED THE HANDSHAKE, DELETE THIS CONNECTION 
    unless(defined($host) && defined($port)) {
      close($fhname);
      delete $bound{$fhname};
      delete $handle{$descriptor{$fhname}};
      delete $descriptor{$fhname};
      delete $clients{$handle}; 
      die "Connection made from non-compliant client!\n";
      return undef;
    }

    # since they passed the handshake, update $tcp_next
    $tcp_next++;

    # if this is a repeat connection, keep buffered information from
    # the last connection that we have not read yet and close last
    # connection if it's still open. 
    my($oldhandle);
    $oldhandle = $hostport_to_handle{$host,$port};
    if(defined($oldhandle)) {
      &close_handle_gracefully($oldhandle);

      # code sanity check
      if(defined($bound{$oldhandle})) {
        die "$oldhandle still bound after call to close_handle_gracefully\n";
      }
    }

    $handle_to_host{$fhname} = $host;
    $handle_to_port{$fhname} = $port;

    $hostport_to_handle{$host,$port} = $fhname;
    &add_to_tcp_order($fhname);

    $fhname;
  }
  else {
    undef;
  }
}

# Assumptions:
#   Assumes first argument is the grids agent's host
#
#   Assumes second argument is the grids agent's port
#
#   Assumes third argument is the number of seconds to
#   wait for a response.  0 means no waiting, undef means
#   indefinite waiting.
#
# This function "pings" for a grids agent at the specified
# host,port.  If a grids agent lives there, 1 is returned,
# else 0.  The function waits for a grids agent to come
# alive at the specified host,port according to the third
# argument passed to this function.
#
sub ping {
  my($host,$port,$timeout) = @_;
  my $stoptime = 0;
  $stoptime = time + $timeout if(defined($timeout));

  unless(defined($INIT)) {
    warn "Comm::ping called before call to Comm::init\n";
    return undef;
  }

  ($host) = gethostbyname($host);
  return 0 unless(defined($host) && defined($port));

  # get whatever handle we may have for this grids agent
  my $handle = $hostport_to_handle{$host,$port};

  # has the destination closed the connection?
  if(defined($handle) && defined($bound{$handle}) && &is_ready_to_read($handle)) {
    &fill_tcp_buff($handle);
  }

  # if we have a handle, and the handle is bound, then the agent is alive
  if(defined($handle) && defined($bound{$handle})) {
    return 1;
  }

  # if we have no handle, try to connect until the time runs out
  else {
    $handle = 'tcp'.$tcp_next;
    while(1) {
      if(&become_tcp_client($host,$port,$handle)) {
        $tcp_next++;
        return 1;
      }
      return 0 if(defined($timeout) && time > $stoptime);
    }
  }
}

# Assumptions:
#   Assumes first argument is the filename
#
#   Assumes second argument is the name for the new file handle
#   this function is to create.
#
# This function opens a file for appending to with the given file handle.
# Upon failure, returns undef.  Else, the file handle name passed in will
# be a valid filehandle for the file
#
sub become_file_client {
  my($filename,$fhname) = @_;

  unless (open($fhname,">>$filename")) {
    warn "could not open $filename for append";
    return undef;
  }
  
  $old = select($fhname); $| = 1; select($old);
  $bound{$fhname} = 'file';
  $descriptor{$fhname} = $filename;
  $handle{$filename} = $fhname;

  $file_to_handle{$filename} = $fhname;

  $fhname;
}

# Assumptions:
#   Assumes first argument is the command to run
#
#   Assumes second argument is the name for the new file handle
#   this function is to create.
#
# This function runs a command such that writes to given filehandle will go
# to the commands STDIN. Upon failure, returns undef.
#
sub become_prog_client {
  my($cmd,$fhname) = @_;

  # print STDERR "starting $cmd\n";
  unless (open($fhname,"| $cmd")) {
    warn "could not run $cmd";
    return undef;
  }
  
  $old = select($fhname); $| = 1; select($old);
  $bound{$fhname} = 'prog';
  $descriptor{$fhname} = $cmd;
  $handle{$cmd} = $fhname;

  $prog_to_handle{$cmd} = $fhname;

  $fhname;
}

# Assumptions:
#   Assumes first argument is the server's host name
#
#   Assumes second argument is the server's port address (numerical)
#
#   Assumes third argument is the name for the new file handle
#   this function is to create.
#
#   Assumes the server is set up to receive clients.
#
# This function creates a connection between the us and a
# specified server.  Upon failure, returns undef.  One common
# reason for failure is attempting to connect to a host which
# has not bound a port for receiving GrIDS messages.  Upon
# success, the file handle name passed in will be a valid
# socket name of a tcp connection between us and the server.
#
sub become_tcp_client {
  my($them,$port,$fhname,$theirname,$oldhandle);
  my($thataddr,$this,$that,$old,$proto);
  ($them,$port,$fhname) = @_;
  ($_,$_,$proto) = getprotobyname('tcp');

  unless(defined($them) && $them) {
    print STDERR "them not true in become_tcp_client; returning undef\n";
    return undef;
  }
  unless(defined($port) && $port) {
    print STDERR "port not true in become_tcp_client; returning undef\n";
    return undef;
  }
  unless($port !~ /^undef/) {
    print STDERR "port = $port in become_tcp_client; returning undef\n";
    return undef;
  }

  unless(defined($HOSTNAME)) {
    warn "in become_tcp_client, HOSTNAME undefined\n";
    return undef;
  }
  ($theirname,$_,$_,$_,$thataddr) = gethostbyname($them);
  unless(defined($thataddr)) {
    warn "in become_tcp_client, cannot determine address for $them\n";
    return undef;
  }
  unless(defined($theirname)) {
    warn "in become_tcp_client, cannot determine hostname for $them\n";
    return undef;
  }

  $this = pack($SOCKADDR,AF_INET,0,$THISADDR);
  unless(defined($this)) {
    warn "in become_tcp_client, cannot pack our address\n";
    return undef;
  }
  $that = pack($SOCKADDR,AF_INET,$port,$thataddr);
  unless(defined($that)) {
    warn "in become_tcp_client, cannot pack their address\n";
    return undef;
  }

  # Use global variable $ERROR_CODE to return which part of the
  # connecting process went wrong, if any.
  unless(socket($fhname,AF_INET,SOCK_STREAM,$proto)) {
    $ERROR_CODE = "socket: $!";
    return undef;
  }
  unless(bind($fhname,$this)) {
    $ERROR_CODE = "bind: $!";
    return undef;
  }
  unless(connect($fhname,$that)) {
    $ERROR_CODE = "connect: $!";
    return undef;
  }
  $old = select($fhname); $| = 1; select($old);
  fcntl($fhname,F_SETFL,O_NDELAY) || die "Can't make non-blocking\n";
  $bound{$fhname} = 'tcp';
  $descriptor{$fhname} = fileno($fhname);
  $handle{$descriptor{$fhname}} = $fhname;
  $clients{$fhname} = 'tcp';
  $handle_to_host{$fhname} = $theirname;
  $handle_to_port{$fhname} = $port;

  # if this is a repeat connection, keep buffered information from
  # the last connection we had with this client and close the last
  # connection if it is still open
  $oldhandle = $hostport_to_handle{$theirname,$port};
  if(defined($oldhandle)) {
    &close_handle_gracefully($oldhandle);

    # code sanity check
    if(defined($bound{$oldhandle})) {
      die "$oldhandle still bound after call to close_handle_gracefully\n";
    }
  }

  $hostport_to_handle{$theirname,$port} = $fhname;
  unless(&handshake_with_server($fhname,$HOSTNAME,$TCP_PORT) eq 'true') {
    warn "handshake with server failed in become_tcp_client\n";
    return undef;
  }
  &add_to_tcp_order($fhname);

  $fhname;
}

# Assumptions:
#   Assumes one argument, a handle for which %handle_to_host
#   and %handle_to_port has values.
#
# Adds this handle to @tcp_order.  If successful, returns 'true',
# else undef;
#
sub add_to_tcp_order {
  my($handle,$host,$port,$key);

  $handle = shift;
  $host = $handle_to_host{$handle};
  unless(defined($host)) {
    warn "Comm::add_to_tcp_order given bad argument $handle\n";
    return undef;
  }
  $port = $handle_to_port{$handle};
  unless(defined($port)) {
    warn "Comm::add_to_tcp_order given bad argument $handle\n";
    return undef;
  }

  $key = $host."\xff".$port;
  push(@tcp_order,$key);
  'true';
}
  
# Assumptions:
#   Assumes one or two arguments, a handle for which %handle_to_host
#   and %handle_to_port has values, or the host and port.
#
# Removes this handle from @tcp_order the first place it finds it.
# If successful, returns 'true', else undef.
#
sub remove_from_tcp_order {
  my($handle,$host,$port,$key,$count);

  if($#_ == 0) {
    $handle = shift;
    unless($host = $handle_to_host{$handle}) {
      warn "Comm::remove_from_tcp_order given bad argument $handle\n";
      return undef;
    }
    unless($port = $handle_to_port{$handle}) {
      warn "Comm::remove_from_tcp_order given bad argument $handle\n";
      return undef;
    }
  } elsif($#_ == 1) {
    ($host,$port) = @_;
  } else {
    warn "remove_from_tcp_order given wrong number of arguments\n";
    return undef;
  } 

  $key = $host."\xff".$port;

  for($count = 0; $count <= $#tcp_order; $count++) {
    if($tcp_order[$count] eq $key) {
      $tcp_order[$count] = pop(@tcp_order);
      return 'true';
    }
  }
  return undef;
}

# Assumptions:
#   Assumes first argument is a reference to a string
#   which is to hold the return value.
#
#   Assumes second argument is a reference to a list of
#   strings.
#
# Takes the list of strings and converts it to one
# string which can be sent over a socket.  This
# one string is returned in the first argument.
#
# WARNING: the reference is not respected.  Whatever
# you send in will be hacked after the call returns.
#
sub encode_packet {
  $[ = 0;
  my($stringref,$arrayref) = @_;
  unless(defined($stringref)) {
    warn "stringref undefined in call to Comm::encode_packet\n";
    return;
  }
  unless(defined($arrayref)) {
    warn "arrayref undefined in call to Comm::encode_packet\n";
    return;
  }

  my @array = @$arrayref;		# make a copy to avoid hosing it

  unless(defined(@array)) {
    warn "array undefined in call to Comm::encode_packet\n";
    return;
  }
  for($_ = 0; $_ <= $#array; $_+=2) {
    my $length;
    if(defined($_) && defined($array[$_])
       && defined(length($array[$_]))) {
         $length = length($array[$_]);
    } else {
      $length = "u";		# code for the undefined value
    }
    splice(@array,$_,0,$length.' ');
  }
  no strict;
  if(defined(@array)) {
    $$stringref = join("",@array);
  } else {
    $$stringref = "";
  }
  use strict;
  # substr($$stringref,0,0) = length($$stringref)." ";
  $$stringref = length($$stringref)." ".$$stringref;
}

# Assumptions:
#   Assumes the first argument is a string reference.
#
#   Assumes the second argument is list reference.
#
# Takes the one string and converts it to a list of
# strings (making the assumption that the string was
# encoded using encode_packet) and stores these strings
# in the list referenced in the second argument.
# Returns the unused portion of the string.
#
sub decode_packet {
  $[ = 0;
  my($packetref,$listref) = @_;

  # return undef if they didn't send us a pointer to a packet string
  return undef unless defined($packetref);

  # return undef if they gave us a pointer to undef. 
  return undef unless defined($$packetref);

  # return the packet if they didn't give us a place to put the 
  # decoded packet
  return $$packetref unless(defined($listref));

  @$listref = ();		# zero out the return list

  my($right,$left);
  if($$packetref =~ /^([0-9]*) /) {
    $right = $1 + length($1);           # $right = index of last character
    $left = length($1) + 1;             # $left = index of first character
  } else {
    return ($$packetref);
  }

  # check to make sure we have the whole packet
  return $$packetref unless($right + 1 <= length($$packetref));

  while($left < $right) {
    if(substr($$packetref,$left) =~ /^([0-9u]+) /) {
      $left += length($1) + 1;
      if($1 eq 'u') {
        push(@$listref,undef);
      } else {
        push(@$listref,substr($$packetref,$left,$1));
        $left += $1;
      }
    }
  }
  my $returnval = substr($$packetref,$right+1);
  return $returnval if(length($returnval) > 0);
  undef;
}

# Assumptions:
#   Assumes first argument is the file handle which is to
#   be tested to see if it is ready to be read.
#   
#   Optional second argument is a string.  'blocking' can
#   be given which specifies that this function is not to
#   return until the handle is ready to be read.  If a number
#   is given as the second argument, then it is interpreted
#   as the number of seconds we are willing to wait for
#   this handle to be ready to be read.
#
# Returns the file handle if the file handle is ready
# to be read, else undef, unless 'blocking' is specified,
# in which case it blocks until the handle is ready and
# then returns the handle.
#
sub is_ready_to_read {
  print "entering is_ready_to_read\n" if(defined($DEBUG));
  my($rin,$ReadyCount,$TimeLeft,$handle,@RReady,$blocking);
  ($handle,$blocking) = @_;
  print "in is_ready_to_read, testing $handle\n" if(defined($DEBUG));
  $blocking = 0 unless(defined($blocking));
  if($blocking eq 'blocking') {
    $blocking = undef;
  } elsif($blocking eq 'nonblocking') {
    $blocking = 0;
  }
  print "in is_ready_to_read, blocking = $blocking\n" if(defined($DEBUG));
  $rin = "";
  vec($rin,$descriptor{$handle},1) = 1;
  ($ReadyCount,$TimeLeft) = select($rin,undef,undef,$blocking);
  if($ReadyCount > 0) {
    @RReady = split(//,unpack("b*",$rin));
    if($RReady[$descriptor{$handle}] eq '1') {
      $handle;
    }
    else {
      undef;
    }
  }
  else {
    undef;
  }
}

# Assumptions:
#   Assumes first argument is the file handle which is to
#   be tested to see if it is ready to be written.
#
#   Assumes the second argument, if given, is 'blocking'
#   or 'nonblocking'.  This function defaults to non-blocking.
#
# Returns the file handle if the file handle is ready
# to be written, else undef.
#
sub is_ready_to_write {
  my($win,$ReadyCount,$TimeLeft,$handle,@WReady);
  $handle = shift;
  $blocking = shift;
  $win = "";
  vec($win,$descriptor{$handle},1) = 1;
  if(defined($blocking) && $blocking eq 'blocking') {
    ($ReadyCount,$TimeLeft) = select(undef,$win,undef,undef);
  } else {
    ($ReadyCount,$TimeLeft) = select(undef,$win,undef,0);
  }
  if($ReadyCount > 0) {
    @WReady = split(//,unpack("b*",$win));
    if($WReady[$descriptor{$handle}] eq '1') {
      $handle;
    }
    else {
      undef;
    }
  }
  else {
    undef;
  }
}

# Assumptions:
#   Assumes first argument is the amount of time to
#   block waiting for a handle to be ready.  undef
#   means indefinite waiting, 0 means no waiting.
#   Value may be rounded to an integer by some
#   operating systems.
#
#   Assumes remaining arguments are a list of file
#   handles to test for read readiness.
#
# Returns the input list minus the handles which are not
# read ready.  Order is maintained.
#
sub are_ready_to_read {
  print "entering are_ready_to_read\n" if(defined($DEBUG));
  my($blocking,$rin,$ReadyCount,$TimeLeft,@RReady,@Return,@Input,$temp);
  $blocking = shift;
  @Input = @_;

  print "in are_ready_to_read, blocking = $blocking\n" if(defined($DEBUG));
  print "in are_ready_to_read, input = @Input\n" if(defined($DEBUG));
  $rin = &fhbits(@Input);
  ($ReadyCount,$TimeLeft) = select($rin,undef,undef,$blocking);
  if($ReadyCount > 0) {
    @RReady = split(//,unpack("b*",$rin));
    foreach (@Input) {
      if($RReady[$descriptor{$_}] eq '1') {
        push(@Return,$_);
      }
    }
  } 
  if($ReadyCount == -1) {
    $WAS_INTERUPTED = 'true';
    if($! =~ /^Bad file number/) {
      die "$!\n";
    }
  }
  @Return;
}

# Assumptions:
#   Assumes the arguments are a list of file handles for which
#   a vector is desired.
#
# Makes a bit vector from the file handle list that will be used
# in system call SELECT.  Each file handle's file descriptor number
# is turned on in the bit vector, while all other file descriptor
# numbers (lower in value than the highest file handle value for
# which we're making the vector) have their bits turned off.
#
sub fhbits {
  local(@list) = @_;
  local($bits);
  $bits = "";
  for (@list) {
    vec($bits, $descriptor{$_},1) = 1 if defined($bound{$_});
  }
  $bits;
}

# Assumptions:
#   Assumes optional argument, a reference to a string.
#   This string will be filled with an error string if
#   undef is returned.
#
# Returns 'true' if we appear to be a tcp_server, undef
# else.
#
sub am_tcp_server {
  my $error_ref = shift;
  unless(defined($TCP_PORT)) {
    $$error_ref = "TCP_PORT undefined\n" if defined($error_ref);
    return undef;
  }
  unless(defined($bound{'TCP_S'})) {
    $$error_ref = "bound{'TCP_S'} undefined\n" if defined($error_ref);
    return undef;
  }
  unless($bound{'TCP_S'} eq 'tcp') {
    $$error_ref = "bound{'TCP_S'} = $bound{'TCP_S'}\n" if defined($error_ref);
    return undef;
  }
  unless(defined($tcp_next)) {
    $$error_ref = "tcp_next undefined\n" if defined($error_ref);
    return undef;
  }
  unless(defined($INIT)) {
    $$error_ref = "init has not been run\n" if defined($error_ref);
    return undef;
  }
  unless($INIT == 1) {
    $$error_ref = "INIT = $INIT\n" if defined($error_ref);
    return undef;
  }
  my $fd = fileno(TCP_S);
  unless($fd >= 0) {
    $$error_ref = "TCP_S has negative file number\n" if defined($error_ref);
    return undef;
  }
  unless(defined($handle{$fd})) {
    $$error_ref = "file number to handle undefined for TCP_S\n" if defined($error_ref);
    return undef;
  }
  unless($handle{$fd} eq 'TCP_S') {
    $$error_ref = "file number to handle incorrect for TCP_S\n" if defined($error_ref);
    return undef;
  }
  'true';
}

# Assumptions:
#   Assumes first argument is the name of the file handle for the new
#   connection.
#
#   Assumes second optional argument is a reference to a string where
#   this function may place error messages if this function fails.
#
#   Assumes we are a tcp server.  (&am_tcp_server should be run before
#   this function to make sure.)
#
# Tries to accept a new connection and initialize variables for that
# connection.  Upon success, the accepted address is returned, else undef.
#
sub accept_connection { my($handle,$error_ref) = @_;
  die "accept_connection called without a file handle\n" unless(defined($handle));
  my $address = accept($handle,TCP_S);
  unless($address) {
    $$error_ref = "accept failed\n" if(defined($error_ref));
    return undef;
  }
  my($af,$port,$inetaddr) = unpack($SOCKADDR,$address);
  my $old = select($handle); $| = 1; select($old);
  $bound{$handle} = 'tcp';
  $descriptor{$handle} = fileno($handle);
  $handle{$descriptor{$handle}} = $handle;
  $clients{$handle} = 'tcp';
  fcntl($handle,F_SETFL,O_NDELAY) || die "Can't make $handle non-blocking\n";
  $inetaddr;
}

# Assumptions:
#   Assumes one argument, the name of the handle from which to recv
#
# This function recvs as much from the given handle as possible and
# returns the string which was recv'd.
#
sub recv_from_handle {
  my($bytes_read,$handle,$buff,$message);
  $handle = shift;
  die "handle not given to function recv_from_handle\n" unless(defined($handle));
  die "READSIZE undefined in function recv_from_handle\n" unless(defined($READSIZE));

  $message = "";
  do {
    $buff = "";
    unless(defined(recv($handle,$buff,$READSIZE,0))) {
      return $message unless($message eq "");
      return undef;
    } else {
       $message .= $buff;
    }
  } while(length($buff) > 0);
  return $message;
}

# Assumptions:
#   Assumes first argument is the handle for communicating with the client
#
#   Assumes the second argument is the address returned by the accept
#   system call which opened this handle.
#
# This function receives, over the network from the client, the client's
# host name and port for accepting connections (if any).  This information
# is checked against the address which was passed in for (insufficient) security.
# (HOST,PORT) is returned upon success, else undef.
#
sub handshake_with_new_client {
  print "entering handshake_with_new_client\n" if(defined($DEBUG));
  my($handle,$addr,$mesg);
  ($handle,$addr) = @_;
  print "in handshake_with_new_client, handle = $handle\n" if(defined($DEBUG));
  die "CONNECT_TIME undefined in function handshake_with_new_client\n"
    unless(defined($CONNECT_TIME));
  die "handle not given to handshake_with_new_client\n" unless(defined($handle));
  if(&is_ready_to_read($handle,$CONNECT_TIME)) {

    # Make sure they're alive and about to send data
    # (This works as a sanity check.  If the client doesn't
    # send the correct string, chances are we have a non
    # GrIDS agent at the other end.
    #
    $mesg = &recv_from_handle($handle);
    unless(defined($mesg)) {
      warn "handshake_with_new_client received no message from client\n";
      return undef;
    }
    unless($mesg =~ /^hostname and portnumber:/) {
      warn "handshake_with_new_client received non-complient message from client\n";
      return undef;
    }

    # parse their host name and port number
    $mesg =~ /^hostname and portnumber: ([^\xff]*)\xff([^\xff]*)\xff([\x0-\xff]*)$/;
    ($host) = gethostbyname($1);
    if($2 ne 'undef') {
      $port = $2;
    } else {
      $port = 'undef'.$undef_next++;
    }

    # verify their address
    my $real_addr; 
    ($_,$_,$_,$_,$real_addr) = gethostbyname($host);
    unless($addr eq $real_addr) {
      warn "Address Spoofing:  Machine claiming to be $host should map to $real_addr,
but is communicating from address $addr ";
      return undef;
    }

    # accept any messages they may have already sent
    if(defined($tcp_buff{$host."\xff".$port})) {
      $tcp_buff{$host."\xff".$port} .= $3;
    } else {
      $tcp_buff{$host."\xff".$port} = $3;
    }

    # return host and port of new client
    return ($host,$port);
  }

  # return undef because client sent nothing
  warn "in handshake_with_new_client, handle not ready to read\n";
  return undef;
}

# Assumptions:
#   Assumes the first argument is the handle for communicating with
#   the server.
#
#   Assumes the second argument is our host name.
#
#   Assumes the third argument is our TCP port address.
#
# This function sends our host name and TCP port number (if any)
# to the server.  If the information is sent, 'true' is returned,
# else undef
#
sub handshake_with_server {
  my($handle,$host,$port) = @_;
  my($mesg,$portstring);
  die "CONNECT_TIME undefined in function handshake_with_server\n"
    unless(defined($CONNECT_TIME));
  die "handshake_with_server called without handle\n" unless(defined($handle));
  die "handshake_with_server called without hostname\n" unless(defined($host));

  # send "hostname and portnumber: $host\xff$port\xff" to the server
  if(defined($port)) {
    $portstring = $port;
  } else {
    $portstring = "undef";
  }
  if(&is_ready_to_write($handle)) {
    print $handle "hostname and portnumber: $host\xff$portstring\xff";
    print "hostname and portnumber: $host\xff$portstring\xff\n" if(defined($DEBUG));
  } else {
    warn "file handle $handle not ready for writing in handshake_with_server\n";
    return undef;
  }
      
  'true';
}

1;
