# This file implements a department module.  This class is part of the 
# organizational hierarchy, and represents a single department in the 
# hierarchy.  The things contained are largely specified in the detail
# design report for GrIDS in the "Organizational Hierarchy Server" section
# of the Hierarchy chapter.

# Fields in this object are

# name			scalar	the unique department name.

# parent		scalar	the name of the parent department.  Will be
#				the string 'null' iff this is the root
#				department.
				
# dept_children		array	the names of the departments beneath this one

# host_children		array	the names of the hosts beneath this one

# aggregator_host	scalar	the name of the aggregator host for the dept

# aggregator_port	scalar	the port of the aggregator for the dept

# manager_host		scalar	the name of the software manager host

# manager_port		scalar	the port of the software manager

# Unofficial fields are

# view_serial		scalar	the organizational hierarchy server maintains
#				this for its own purposes.  It is not 
#				manipulated inside this class.  It is mentioned
#				here to avoid naming clashes.  It does not
#				appear in I/O of the hierarchy.
				
#============================================================================#

package Department;

#============================================================================#

# Function: new

# This function will create a new department and return it.

# Inputs:
#  Actual Arguments:
#   name, parent, manager_host, manager_port, agg_host, agg_port
#  Global:
#   None

# Return Value(s):
#  The new department, or an error message.

# Global variable modifications:
#  None
 
# Example of usage:
#  $my_dept = new 
#   Department($name,$parent,$manager_host,$manager_port,$agg_host,$agg_port);
# or
#  $my_dept = new Department($string)
#   where string is a <dept-node> string in the organizational hierarchy grammar.

sub new
{
  shift;
  my($self,$raw_h_kids,$raw_d_kids);
  my($name,$parent,$manager_host,$manager_port,$aggregator_host,
  							$aggregator_port) = @_;
  my($name_idx);
  local(@host_children) = ();
  local(@dept_children) = ();
  
  if(@_ == 1)
   {
    # If we were passed a string instead of a list, check the string for
    # the list elements.  Example format:
    # dept,ROOT,null,{},{},manager:54,aggregator:72;
    $_ = $name;
    unless(($name,$parent,$raw_h_kids,$raw_d_kids,
	$manager_host,$manager_port,$aggregator_host,$aggregator_port) = 
    /^dept\,(\w+),(\w+),\{([\w\.\,]*)\},\{([\w\,]*)\},([\w\.]+):(\d+),([\w\.]+):(\d+)$/)
      {return "Bad dept entry: $_";}
    @host_children = split(/,/,$raw_h_kids);
     {
      foreach $name_idx (@host_children)
       {return "Bad host children in: $_" unless($name_idx);}
     }
    @dept_children = split(/,/,$raw_d_kids);
     {
      foreach $name_idx (@dept_children)
       {return "Bad dept children in: $_" unless($name_idx);}
     }
   }
  elsif(@_ != 6)
   { return "Bad number of parameters in new Department";}

  $self =
   {
    'name' 	=> $name,
    'parent'	=> $parent,
    'host_children'	=> \@host_children,
    'dept_children'	=> \@dept_children,
    'manager_host'	=> $manager_host,
    'manager_port'	=> $manager_port,
    'aggregator_host'	=> $aggregator_host,
    'aggregator_port'	=> $aggregator_port,
   };
  bless $self;
  return 'New department not verified' unless $self->verified;
  return $self;
}

#============================================================================#

# Function: verified
 
# This function will return false if every element of the message
# is not defined.
 
# Inputs:
#  Actual Arguments:
#   an array containing department information
#  Department Arguments (contained in array):
#   name, parent, manager_host, manager_port, add_host, agg_port
#  Global:
#   None
 
# Return Value(s):
#  False (0) if all elements are not defined,
#  True (1) otherwise
 
# Global variable modifications:
#  None
 
# Example of usage:
#  unless $my_dept->verified
 
sub verified
{
  my $self = shift;
  return 0 unless defined $self->{'name'};
  return 0 unless defined $self->{'parent'};
  return 0 unless ref $self->{'host_children'} eq 'ARRAY';
  return 0 unless ref $self->{'dept_children'} eq 'ARRAY';
  return 0 unless defined $self->{'manager_host'};
  return 0 unless defined $self->{'manager_port'};
  return 0 unless defined $self->{'aggregator_host'};
  return 0 unless defined $self->{'aggregator_port'};
  
  return 1;
}

#============================================================================#
#
# Function:  compare
#
# Compares this Department to another.
#
# Inputs:
#  Actual Arguments:
#	Key name
#	Name for our hierarchy
#	Name for other hierarchy
#	Other Host object
#
# Return Value(s): A list of two elements
#  	0 on failure, 1 on success.
#	a warning string on failure, ill-defined on success
#
# Example of usage 
#  ($result,$warning) =  $d->compare('hostname','ohs','local',$local_d);
#
#----------------------------------------------------------------------------#   

sub compare
{
  my($self, $key, $our_name, $their_name, $them) = @_;
  my $success = 1;
  my $warning = '';

  unless($self->verified())
   {
    $warning .= "\n\tDepartment $key from $our_name had undefined fields";
    $success = 0;
   }
  unless($them->verified())
   {
    $warning .= "\n\tDepartment $key from $their_name had undefined fields";
    $success = 0;
   }
  return ($success,$warning) unless $success;

  unless($self->{'name'} eq $them->{'name'})
   {
    $warning .= "\n\tDepartment $key from $their_name had name = $them->{'name'}, while instance from $our_name had name = $self->{'name'}";
    $success = 0;
   }

  unless($self->{'parent'} eq $them->{'parent'})
   {
    $warning .= "\n\tDepartment $key from $their_name had parent = $them->{'parent'}, while instance from $our_name had parent = $self->{'parent'}";
    $success = 0;
   }

  unless($self->{'aggregator_host'} eq $them->{'aggregator_host'})
   {
    $warning .= "\n\tDepartment $key from $their_name had aggregator_host = $them->{'aggregator_host'}, while instance from $our_name had aggregator_host = $self->{'aggregator_host'}";
    $success = 0;
   }

  unless($self->{'aggregator_port'} eq $them->{'aggregator_port'})
   {
    $warning .= "\n\tDepartment $key from $their_name had aggregator_port = $them->{'aggregator_port'}, while instance from $our_name had aggregator_port = $self->{'aggregator_port'}";
    $success = 0;
   }

  unless($self->{'manager_host'} eq $them->{'manager_host'})
   {
    $warning .= "\n\tDepartment $key from $their_name had manager_host = $them->{'manager_host'}, while instance from $our_name had manager_host = $self->{'manager_host'}";
    $success = 0;
   }

  unless($self->{'manager_port'} eq $them->{'manager_port'})
   {
    $warning .= "\n\tDepartment $key from $their_name had manager_port = $them->{'manager_port'}, while instance from $our_name had manager_port = $self->{'manager_port'}";
    $success = 0;
   }

  my $my_dept_chilren = join(' ',sort @{$self->{'dept_children'}});
  my $my_host_chilren = join(' ',sort @{$self->{'host_children'}});
  my $their_dept_chilren = join(' ',sort @{$them->{'dept_children'}});
  my $their_host_chilren = join(' ',sort @{$them->{'host_children'}});

  unless($my_dept_chilren eq $their_dept_chilren)
   {
    $warning .= "\n\tDepartment $key from $their_name had dept_children = $their_dept_chilren, while instance from $our_name had dept_children = $my_dept_chilren";
    $success = 0;
   }

  unless($my_host_chilren eq $their_host_chilren)
   {
    $warning .= "\n\tDepartment $key from $their_name had host_children = $their_host_chilren, while instance from $our_name had host_children = $my_host_chilren";
    $success = 0;
   }

  return ($success,$warning);
}

#============================================================================#

# Function: output
 
# This function will display the host information
 
# Inputs:
#  Actual Arguments:
#   an array containing department information
#  Department Arguments (contained in array):
#   name, parent, manager_host, manager_port, add_host, agg_port
#  Global:
#   None
 
# Return Value(s):
#  None
# Global variable modifications:
#  None
 
sub output
{
  my($self, $format) = @_;
  if ($format eq 'prettyprint') {
    print "dept, $self->{'name'}, $self->{'parent'} {\n\t";      #}
    if (@{$self->{'host_children'}}) {
      print join (",\n\t", @{$self->{'host_children'}});         #{
      print "\n\t},\n\t{";                                       #}
      }
    else {                                                       #{
      print "},\n\t{";                                           #}
      }
    print join (",\n\t", @{$self->{'dept_children'}});           #{
    print "},\n";
    print "$self->{'manager_host'}:$self->{'manager_port'}, "
        . "$self->{'aggregator_host'}:$self->{'aggregator_port'}\n\n";
    return;
    }
  elsif ($format eq 'parse_config_file') {
    my $parent =  $self->{'parent'}  || '0';        # ROOT dept has no parent
    print "\t{DEPT_NAME => '$self->{'name'}', "
        . " PARENT_DEPT => '$parent',\n"
        . "\tSM_HOST => '$self->{'manager_host'}', "
        . " AGG_HOST => '$self->{'aggregator_host'}',\n"
        . "\tSPECIFIC_HOSTS => [\n";
    foreach $child (@{$self->{'host_children'}}) {
      print "\t\t\t'$child',\n";
      }
    print "\t\t\t],\n\t},\n";
    return;
    }
##############################################################################
##############################################################################
  # else ...

  print <<"End";
dept, $self->{'name'}, $self->{'parent'},
{
End
  print join(",\n",@{$self->{'host_children'}});
  print <<"End";
 
},{
End
  print join(",\n",@{$self->{'dept_children'}});
  print <<"End";
 
},
$self->{'manager_host'}:$self->{'manager_port'}, $self->{'aggregator_host'}:$self->{'aggregator_port'};
End

}


#============================================================================#

# Function: string_print
 
# This function accepts an array with host information and returns the
# information in a string format.
 
# Inputs:
#  Actual Arguments:
#   an array containing department information
#  Department Arguments (contained in array):
#   name, parent, manager_host, manager_port, add_host, agg_port
#  Global:
#   None
 
# Return Value(s):
#  String containing Department information
# Global variable modifications:
#  None
 
sub string_print
{
  my($self) = @_;
  return "dept,$self->{'name'},$self->{'parent'},{".
  join(",",@{$self->{'host_children'}})."},{".
  join(",",@{$self->{'dept_children'}}).
  "},$self->{'manager_host'}:$self->{'manager_port'},".
  "$self->{'aggregator_host'}:$self->{'aggregator_port'}";
}

#============================================================================#

1; # end of package
