# This file implements a Host module.  This class is part of the 
# organizational hierarchy, and represents a single host in the 
# hierarchy.  The things contained are largely specified in the detail
# design report for GrIDS in the "Organizational Hierarchy Server" section
# of the Hierarchy chapter.  

# This data structure is mainly used by Hierarchy.pm.  That module
# in turn, is used by both the OHS and the user interface.

# Fields in this object are

# name			scalar	the unique host name.

# parent		scalar	the name of the parent department.  
				
# port			scalar	the port of the module controller for this host

#============================================================================#

package Host;

#============================================================================#

# Function: new

# This function will create a new host and return it.

# Inputs:
#  Actual Arguments:
#   host_name, parent_dept, port_number
#  Global:
#   None

# Return Value(s):
#  The new host, or an error message.

# Global variable modifications:
#  None

# Example of usage:
#  $my_host = new Host($host_name,$parent_dept,$port_number);

sub new
{
  shift;
  my($self);
  my($name,$parent,$port) = @_;
  if(@_ == 1)
   {
    # We were actually passed a string instead of an array.  Parse the string
    # to get our arguments.
    $_ = $name;
    unless(($name,$parent,$port) = /^host,([\w\.]+),(\w+),(\d+)$/)
      {return "Bad host entry: $_";}
   } # end if
  elsif(@_ != 3)
   {return "Bad number of params in call to new Host";}
  bless $self =
   {
    'name' 	=> $name,
    'parent'	=> $parent,
    'port'	=> $port,
   };# end bless
  return "Did not verify" unless $self->verified;
  return $self;
} # end new

#============================================================================#

# Function: verified

# This function will return false if every element of the message
# is not defined.

# Inputs:
#  Actual Arguments:
#   refernce to an array containing host information
#  Host Arguments (contained in array):
#   host_name, parent_dept, port_number
#  Global:
#   None

# Return Value(s):
#  False (0) if all elements are not defined,
#  True (1) otherwise

# Global variable modifications:
#  None

# Example of usage:
#  unless $my_host->verified 

sub verified
{
  my $self = shift;
  
  return 0 unless defined $self->{'name'};
  return 0 unless defined $self->{'parent'};
  return 0 unless defined $self->{'port'};
  
  return 1;
}

#============================================================================#
#
# Function:  compare
#
# Compares this Host to another.
#
# Inputs:
#  Actual Arguments:
#	Key name
#	Name for our hierarchy
#	Name for other hierarchy
#	Other Host object
#
# Return Value(s): A list of two elements
#  	0 on failure, 1 on success.
#	a warning string on failure, ill-defined on success
#
# Example of usage 
#  ($result,$warning) =  $h->compare('hostname','ohs','local',$local_h);
#
#----------------------------------------------------------------------------#   

sub compare
{
  my($self, $key, $our_name, $their_name, $them) = @_;
  my $success = 1;
  my $warning = '';

  unless($self->verified())
   {
    $warning .= "\n\tHost $key from $our_name had undefined fields";
    $success = 0;
   }
  unless($them->verified())
   {
    $warning .= "\n\tHost $key from $their_name had undefined fields";
    $success = 0;
   }
  return ($success,$warning) unless $success;

  unless($self->{'name'} eq $them->{'name'})
   {
    $warning .= "\n\tHost $key from $their_name had name = $them->{'name'}, while instance from  $our_name had name = $self->{'name'}";
    $success = 0;
   }
  unless($self->{'parent'} eq $them->{'parent'})
   {
    $warning .= "\n\tHost $key from $their_name had parent = $them->{'parent'}, while instance from $our_name had parent = $self->{'parent'}";
    $success = 0;
   }
  unless($self->{'port'} == $them->{'port'})
   {
    $warning .= "\n\tHost $key from $their_name had port = $them->{'port'}, while instance from $our_name had port = $self->{'port'}";
    $success = 0;
   }

  return ($success,$warning);
}

#============================================================================#

# Function: output
 
# This function will print the host information
 
# Inputs:
#  Actual Arguments:
#   an array containing host information
#  Host Arguments (contained in array):
#   host_name, parent_dept, port_number
#  Global:
#   None
 
# Return Value(s):
#  None 
# Global variable modifications:
#  None

sub output
{
  my($self) = @_;
  print "host, $self->{'name'}, $self->{'parent'}, $self->{port};\n";
}

#============================================================================#

# Function: string_print

# This function accepts an array with host information and returns the 
# information in a string format.
 
# Inputs:
#  Actual Arguments:
#   an array containing host information
#  Host Arguments (contained in array):
#   host_name, parent_dept, port_number
#  Global:
#   None
 
# Return Value(s):
#  String containing Host information.
# Global variable modifications:
#  None
 
sub string_print
{
  my($self) = @_;
  return "host,$self->{'name'},$self->{'parent'},$self->{port}";
}

#============================================================================#

1; # end of package
