#!/pkg/bin/perl -w

package Timeout_recur;

use Timeout;

# This is a subclass of the timeout manager, and the reader should
# check the documentation there before reading this.  This class
# contains the original functionality of the Timeout manager, and adds
# the ability to manage events which should occur on a regular basis
# rather than on a one-time basis.  To do this, one new public method
# is added ('add_recurrent'), and other public methods are overridden.

# Note: it is possible that the 'timed_out' function is not called for
# a long time, during which numerous instances of a recurrent event
# become due.  In that circumstance, the semantics are that only the
# *last* such event is returned from the 'timed_out' call.

# Recurrent events automatically regenerate so as to appear multiple
# times across polls (calls to timed_out), but never more than once in
# a single poll result.

# add_recurrent has two argument forms.  In both forms, the first
# argument indicates the time interval between occurrences of an event
# and the second argument uniquely identifies the event.  If a third
# argument is given, it is used as the time of the first occurrence of
# the event; the default is for the first occurrence to occur
# "interval" seconds after the call to add_recurrent.

# Conceptually, recurrent events are scheduled every "interval"
# seconds an infinite number of times into the future but events that
# would occur multiple time between two polls have all but one
# occurrence ignored.  Mathematically stated, if an recurrent event e
# occurred at time t, then t= "time of first occurrence of the e" +
# k*"interval(e)" for some integer k.  Note that a call

# Calls to add with the same event identifier as a recurrent one will
# have an unspecified effect.

# Recurrent events as well as non-recurrent ones are removed by the
# "delete" method.

# The software manager depends on this class to implement its polling 
# of children.

#==========================================================================#

@ISA = ('Timeout');

#==========================================================================#

# The new method creates a new instance of a Timeout_recur object.
#
# Timeout_recur::new: class -> Timeout_recur
#
# e.g. $tm= Timeout_recur->new
#
sub new {
  shift;
  my ($self)= Timeout->new(@_);  # inherit all Timeout's data plus add
                                 # two new ones:
  $self->{'Recur_int'}= (); # stores the interval for the given
                            # recurring event
  $self->{'Recur_time'}= (); # stores the last scheduled time for a
                             # recurring event (for use in figuring
                             # out when to schedule the next one;
  return bless $self;
}


# add_recurrent adds a recurring event to the manager as described
# above.  It the event already exists, it is overridden.

# Usage:

# $my_timeout_recur->add_recurrent($interval,$event);
# $my_timeout_recur->add_recurrent($interval,$event,$start_time);

# Parameters:

# $interval is the time in seconds between recurrences of the event.
# $event is the scalar for the event in the same sense used in Timeout.pm

# $start_time is the time at which the first event is to occur.  As
# with the first argument to "add", if this value is less than 100000,
# then it is an offset from the current system time otherwise it
# indicates a Unix time.  The default if $interval seconds from "now".

# Return value:

# The return value is not significant and should not be used.

sub add_recurrent
{
 my($self,$interval,$event,$start) = @_;
 if (defined($start)) {
   $start+= time if $start < 100000;
 } else {
   $start= time+$interval;
 }
 $self->add($start,$event);

 $self->{'Recur_int'}{$event}= $interval;
 $self->{'Recur_time'}{$event}= $start;
 return 0;
}

# This delete method essentially intercepts calls to Timeout::delete,
# deleting the event if it recurring otherwise passing it on to
# Timeout::delete
#
# Timeout_recur::delete: Timeout_recur x event -> 
#
# e.g. $tm->delete($recurring_event);
# e.g. $tm->delete($non_recurring_event);
#
sub delete {
  my($self,$event) = @_;
  if (defined($self->{'Recur_int'}{$event})) { # a recurring event
    delete $self->{'Recur_int'}{$event};
    delete $self->{'Recur_time'}{$event};
  }
  $self->Timeout::delete($event); # needed for recurring and non-recurring
}

# this method overrides the Timeout::timed_out method to return both
# recurring and non-recurring events that have occurred since this
# method was last called. The timed_out function checks the time from
# the system clock and the schedule and returns all of the events that
# have come due since this function was last called in a time-ordered
# ordered list.
#
# Timeout_recur::timed_out: Timeout_recur -> {event}
#
# e.g. foreach $new_event ($tm->timed_out) { ...
#
sub timed_out {
  my($self)= shift;
  
  # note these two lines must occur in this order and should be as
  # close together as possible; strictly speaking, the same "now"
  # should be used in both Timeout's and our "timed_out" methods, but
  # this would entail a redesign of the Timeout class
  @events= $self->Timeout::timed_out(@_);
  my($now)= time;

  foreach (@events) {
    if (defined($self->{'Recur_int'}{$_})) { # a recurrent event so re-add
      do {  # calc next event time (needs to be in future)
	$self->{'Recur_time'}{$_}+= $self->{'Recur_int'}{$_};
      } until ($self->{'Recur_time'}{$_} > $now);
      $self->add($self->{'Recur_time'}{$_},$_);
    }
  }
  return @events;
}


#==========================================================================#
