#!/pkg/bin/perl -w

# $Id: Buffer.pm,v 1.16 1996/10/15 23:47:37 hoagland Exp $
#
# %rule_buffers:  $rule_buffers{RULESETNAME} is the alpha buffer
#                 for RULESET.
#
# %beta_buffer:  holds all reports for beta time to be used
#                      to check for and eliminate redundancy.
#

use Timeout;
use Ruleset;
use Graph;
require "add_report.pl";
package Buffer;

sub init_buffer {
  $beta_buffer = Timeout->new;
  $gamma_buffer= Timeout->new;
  $::cvars->{'beta'}= 30 unless defined($::cvars->{'beta'}) && $::cvars->{'beta'} =~ /^\d+$/; # use a default value if not set up (yet)
  $::cvars->{'gamma'}= 20 unless defined($::cvars->{'gamma'}) && $::cvars->{'gamma'} =~ /^\d+$/; # use a default value if not set up (yet)
  foreach (Ruleset->names) {
    new_ruleset($_);
  }
}

sub new_ruleset {
  $rule_buffers{$_[0]} = Timeout->new;
  $alpha{$_[0]} = Ruleset->named($_[0])->buffer_time;
}

sub remove_ruleset {
  delete $rule_buffers{$_[0]};
  delete $alpha{$_[0]};
}

# Assumptions:
#   Assumes ..../code/common/Timeout.pm is being used.
#
#   Assumes Ruleset is the relevant ruleset.
#
#   Assumes %alpha is a hash of alphas for all rulesets.
#   Recall that alpha is the buffer time for the ruleset,
#   as described in the techrep, section 2.2.
#
#   Assumes %rule_buffers is a hash of Timeouts used as
#   alpha buffers. (Does not assume that this is initialized.)
#
#   Assumes $beta_buffer is a Timeout used as the beta buffer.
#
#   Assumes first argument is a string to be fed to
#   one or more rulesets (after being buffered and converted to
#   a graph).
#
#   Assumes remaining arguments are a list of all the names of the
#   rulesets to add the string to.  If no ruleset names
#   are listed, assumes all rulesets in existence are
#   requested.
#
# Converts text input to graph format (by calling Graph->new($text)),
# then buffers the graph reference in the buffers for each ruleset
# listed in the arguments.
#
sub add_buffer {
  my($text,@rulesets) = @_;

  my $reftext= $text;
  $reftext =~ s/\bid=\"([^\"]+)\"//; # take out "id" field for beta buffer

  unless(&is_redundant($reftext)) {
#    print "$1 was not redundant\n";
    my $graph = Graph->new('',$text);
    # hand off to gamma buffer
    &add_to_gamma_buffer($graph,@rulesets);
  } else {
#    print "$1 was redundant\n";
  }
  &add_redundant($reftext);
}

# adds the given Graph to the specified rulesets.  If not rulesets are
# specified, it is added to them all
sub add_to_alpha_buffers {
  my($graph,@rulesets)= @_;
  my($ruleset);

  @rulesets= Ruleset->names unless(@rulesets);
  foreach $ruleset (@rulesets) {
    if (defined($rule_buffers{$ruleset})) { # make sure we know about this rs
      $rule_buffers{$ruleset}->add($alpha{$ruleset} + &graph_time($graph),$graph);
      $::DEBUG && print "buffering ",$graph->id," for time ",$alpha{$ruleset} + &graph_time($graph)," for $ruleset\n";
    }
  }
}


# graph_time determines the time field for this Graph.  It first looks to
# global attributes; if it finds it, it uses it.  Otherwise it uses
# the highest it finds on the nodes and edges.  If it can't find any
# there, then it uses the current system time.  The time for the graph
# is returned.
#
# Buffer::graph_time: Graph -> time
#
# e.g. $graph->time;
#
sub graph_time {
  my ($graph)= shift;
  my($time,$newt);
  $time= $graph->attr_val('time');
  return $time if defined($time) && $time;
  $time=0;  # look on nodes and  edges for a time field since no global one
  foreach ($graph->edges,$graph->nodes) {
    $time= $newt if defined($newt=$_->attr_val('time')) && $time < $newt;
  }
  $time= time unless $time; # couldn't find a time field anywhere
  return $time;
}

# Assumptions:
#   Assumes no arguments.
#
#   Assumes ..../code/common/Timeout.pm is being used.
#
# Takes all buffered reports which are ready to be sent
# to the rulesets and sends them (in graph form, not text).
#
sub out_buffer {
  my($ruleset,$graphref,$event);
  
  foreach $event ($gamma_buffer->timed_out) {
    my ($report,@rulesets)= @{$event};
    &resolve_depts($report); # hope everything gets resolved okay

    # for reports not involving children in this department, throw them away,
    # otherwise add them to alpha buffers
    my($children)= 0;
    foreach ($report->nodes) {
      $children++ if defined($::cvars->{'children'}{$_->{'name'}});
    }
    &add_to_alpha_buffers($report,@rulesets) if $children;
  }

  foreach $ruleset (Ruleset->all) {
    my ($rsname)= $ruleset->name;
    if (defined($rule_buffers{$rsname})) {
      foreach $graphref ($rule_buffers{$rsname}->timed_out) {
	$::DEBUG && print "unbuffering ",$graphref->id," for ",$rsname,"\n";
	&::add_report($graphref,$ruleset);
      }
    }
  }
}

# Assumptions:
#   Assumes first argument is a string.  This string is
#   assumed to be the text of a graph report.
#
sub add_redundant {
  my $text = shift;

  $beta_buffer->add($::cvars->{'beta'},$text);
  $beta_buffer{$text} = 'true';
}

# Assumptions:
#   Assumes one argument, a string (dot language report)
#   from a data source.
#
# Returns 'true' if the input is recognized as a repeat
# report, undef else.  Note that repeat reports may not
# always be recognized as such.  This function only makes
# a best effort of recognizing repetitions.
#
sub is_redundant {
  my($text,$returnval);
  $text = shift;

  # find out if this report is redundant
  $returnval = $beta_buffer{$text};

  # remove all expired data from Timeout and from hash
  foreach $report ($beta_buffer->timed_out) {
    delete $beta_buffer{$report};
  }
  $returnval;
}
  




##########################
## gamma buffer functions
##########################

# add the given graph to the gamma buffer.  This is a report that is destined
# for the specified rulesets (or all of them if none are specified).  This
# function learns as much as it can about the departments for hosts as it can
# by looking at edges, then tries to resolve the source and destination if
# needed, If they cannot be immediately resolved, then add them to the gamma
# buffer.  Otherwise pass them on to the alpha buffers.
sub add_to_gamma_buffer {
  my($report,@rulesets)= @_;
  my(@edges)= $report->edges;

  unless (@edges) { # not an edge report; add to alpha buffers or throw it
                    # away if it is not a child
    foreach ($report->nodes) { # probably not more than one node, but just in
                               # case ...
      if (defined($::cvars->{'children'}{$_->{'name'}})) { # a child
	&add_to_alpha_buffers($report,@rulesets);
      } else {
	return;
      }
    }
  }

  my($edge);
  foreach $edge (@edges) { # probably not more than one edge, but just in
                           # case ...
    &discover_gamma_info($edge);
  }


  if (&resolve_depts($report)) { # some weren't resolved
    # add to gamma buffer to $gamma seconds
    $gamma_buffer->add($::cvars->{'gamma'},[$report,@rulesets]);
  } else {
    # pass on to alpha buffers
    &add_to_alpha_buffers($report,@rulesets);
  }
}

# given a report Graph, this function resolves the names of the nodes it
# needs to, updating their name in the Graph, and returns a list of the names
# it can't resolve
sub resolve_depts {
  my($report)= shift;
  my(@unresolved)= ();
  my($nodename,$dept,$edge);

  foreach $nodename (map($_->{'name'},$report->nodes)) {
    unless (defined($::cvars->{'children'}{$nodename})) { # not a child
      if (@parentage= &resolve_dept($nodename)) {
	# fill in parentage in source and dest auto-computed attrs in edges
	# as appropriate
	foreach $edge ($report->edges) {
	  if ($edge->src->{'name'} eq $nodename) { 
	    $edge->{'source'}= &::prepend($edge->{'source'},$::cvars->{'department'},@parentage);
	  }
	  if ($edge->dest->{'name'} eq $nodename) { 
	    $edge->{'dest'}= &::prepend($edge->{'dest'},$::cvars->{'department'},@parentage);
	  }
	}
	# change name many places in graph
	$report->rename_node($nodename,$parentage[0]);
      } else {	
	push(@unresolved,$nodename);
      }
    }  
  }
  return @unresolved;
}

# return the parentage of incoming node name or return the empty list.  The parentage
# is returned as a list with a child of this department as the head and the
# immediate parent of the node as the end.
sub resolve_dept {
  return defined($parents_of_node{$_[0]}) ? @{$parents_of_node{$_[0]}} : ();
}

# given an Edge, this routine learns as much as it can about the hierachy
sub discover_gamma_info {
  my($edge)= shift;
  my($top,@srcl,@destl,@parentage);

  ($top,@srcl)= &::list_members($edge->{'source'});
  $top= shift(@srcl) if defined($top) && $top eq $::cvars->{'department'};
  if (defined($top) && defined($::cvars->{'children'}{$top})) { # top is a child
    @parentage= ($top);
    while ($_=shift(@srcl)) { # remember all things under this child as being such
      $parents_of_node{$_}= [@parentage];
      push(@parentage,$_);
    }
  }

  ($top,@destl)= &::list_members($edge->{'dest'});
  $top= shift(@destl) if defined($top) && $top eq $::cvars->{'department'};
  if (defined($top) && defined($::cvars->{'children'}{$top})) { # top is a child
    @parentage= ($top);
    foreach (@destl) { # remember all things under this child as being such
      $parents_of_node{$_}= [@parentage];
      push(@parentage,$_);
    }
  }

#  my ($node);
#  foreach $node (keys %parents_of_node) {
#    print "\$parents_of_node{$node}= [",join(',',@{$parents_of_node{$node}}),"]\n";
#  }
}

1;

