#!/pkg/bin/perl -w

# $Id: Edge.pm,v 1.15 1997/02/01 20:15:31 hoagland Exp $

# This file contains the Edge class and methods.

use GraphObj;
 
# Edge's are edges in a Graph

require 'attr.pl';

package Edge;   

@ISA= qw(GraphObj);

# Edge::new creates a new Edge with the given id, source, destination,
# and attributes.  'source' and 'dest' auto-computed attributes are
# set up/updated as needed and the 'id' one is taken from the
# arguments.
#
# new: class x Node x Node x id x {attr_name x val} -> Edge
#
# e.g. $Edge= Edge->new($src,$dest,$id,'a' => 'b', 'c' => 'd');
#
# assumes that children are stored as indices to the hash ref
# $::cvars->{'children'} and the department is in $::cvars->{'department'}
#
sub new {
  my($class,$src,$dest,$id,%attrs)= @_;
  my $self= bless GraphObj->new($id,%attrs,'_src' => $src,'_dest' => $dest,'id'=>$id);
  my($srcn,$destn)= ($self->{'_src'}{'name'},$self->{'_dest'}{'name'});
  my($dom)= $::cvars->{'department'};

  # set up source and dest (if not already set up)
  my(@srcl)= defined($self->{'source'}) ? &::members($self->{'source'}) : ($srcn);
  @srcl= ($srcn) unless @srcl;
  unshift(@srcl,$dom) if defined($::cvars->{'children'}{$srcn}) && $srcl[0] ne $dom;
  $self->{'source'}= &::create_list(@srcl);

  my(@destl)= defined($self->{'dest'}) ? &::members($self->{'dest'}) : ($destn);
  @destl= ($destn) unless @destl;
  unshift(@destl,$dom) if defined($::cvars->{'children'}{$destn}) && $destl[0] ne $dom;
  $self->{'dest'}= &::create_list(@destl);

  return $self;
}

#######

# Edge::duplicate creates and returns a new Edge with the same id and
# attributes as this Edge and with the given enclosing Graph.  The
# source and destination have the same id as this, but the actual
# Nodes are drawn from the Graph.
#
# Edge::duplicate: Edge x Graph -> Edge
#
# e.g. $Edge2= $Edge->duplicate($graph);
#
sub duplicate {
  my($self,$graph)= @_;
  return Edge->new($graph->node($self->src->id),$graph->node($self->dest->id),$self->id,$self->attr_hash);
}

#######

# Edge::src returns the source Node of this Edge.
#
# src: Edge -> Node
#

# e.g. $src= $edge->src();
#
sub src {
  return $_[0]->{'_src'};
}

# Edge::dest returns the destination Node of this Edge.
#
# dest: Edge -> Node
#
# e.g. $dest= $edge->dest();
#
sub dest {
  return $_[0]->{'_dest'};
}

########

# Edge::change_src resets the source Node of this Edge.
#
# change_src: Edge x Node -> 
#
# e.g. $src= $edge->change_src($node);
#
sub change_src {
  $_[0]->{'_src'}= $_[1];
}

# Edge::change_dest resets the destination Node of this Edge.
#
# change_dest: Edge x Node ->
#
# e.g. $dest= $edge->change_dest($node);
#
sub change_dest {
  $_[0]->{'_dest'}= $_[1];
}

####

# Edge::merge_auto_computed sets the res attributes in terms of the
# cur attributes and those of a new one for auto-computed attributes.
# It is called at the time that the edge combining rules are being on
# the edges. This just copies the 'source', 'dest', and 'id'
# attributes over from cur to res, ignoring the new one entirely.
#
# merge_auto_computed: Edge x Edge ->
#
# e.g. $edge->merge_auto_computed;
#
sub merge_auto_computed {
  my $self=shift;
  $self->res_attrs->set_attrs('source' => $self->attr_val('source'),
			      'dest' => $self->attr_val('dest'),
			      'id' => $self->attr_val('id'));
}

# Edge::auto_computed_to_res moves the auto-computed attributes of the
# Edge (namely 'source' and 'dest') to the res attributes. 
# ** Note that sets source and dest themselves aren't actually copied
# in this implementation, but the reference is copied.  This should be
# okay since the original side is about to be clobbered. **
#
# auto_computed_to_res: Edge ->
#
sub auto_computed_to_res {
  my $self=shift;
  $self->res_attrs->set_attrs('source' => $self->attr_val('source'),
			      'dest' => $self->attr_val('dest'),
			      'id' => $self->attr_val('id'));
}

1;
