#!/pkg/bin/perl -w

# $Id: Function.pm,v 1.7 1997/02/01 20:16:00 hoagland Exp $

# This file contains the Function class and methods.  The Function
# class is a class for functions that are available during the
# execution of ruleset rules.  This class also maintains a registry of
# such functions.

#-------------------------

package Function;   
use strict 'refs';

sub BEGIN {
  %registry=();
}

# Function::named returns the registered Function with the given name
# if one exists otherwise the undefined value.
#
# named: class x name -> Function
#
# e.g. unless (Function->named('foo'_) { ..
#
sub named {
  return $registry{$_[1]};
}

#########

# Function::new creates a new Function with the given prototype and
# adds it to the registry.
#
# new: class x prototype_text -> Function
#
# e.g. Function->new('list foo(set, scalar, list)');
#
sub new {
  my($class,$proto)= @_;
  $_= $proto;

  s/^\s*(\w+)\s*(\w+)\s*\(\s*// || (warn "Function::new: $_ is not a valid prototype" &&  return 0);
  my($return,$name)= ($1,$2);
  s/\s*\).*\s*$//;
  $return =~ /^(set|list|scalar|void)$/ || (warn "Function::new: $_ is not a valid return type in $proto" && return 0);
  
  @types= split(/\s*,\s*/,$_);
  foreach (@types) {
    /^(set|list|scalar)$/ || (warn "Function::new: $_ is not a valid type in $proto" && return 0);
  }
  
  my $self= bless { 'Name' => $name, 'Return_type' => $return, 'Arg_types' => [@types], 'Have_code' => 0 };
  $registry{$name}= $self;
  return $self;
}

# Function::add_code adds the given code to the functions available to
# ruleset rules.  A prototype for the function is required unless
# there is a second argument with a true value.  If the prototype is
# required but not present, then the code is not added.  The function
# returns a true value if there has a prototype given for the function
# (regardless of whether it was required).  The function evals the code
# in the "run" package.
#
# add_code: class x text x boolean -> boolean
#
# e.g. Function->add_code('sub foo {...');
#      Function->add_code('sub bar {...',1);
#
sub add_code {
  my($class,$proto_opt);
  ($class,$code,$proto_opt)= @_;  # note that $code is not local so that it can
                       # be referenced inside the "run" package
  $code =~ /^\s*sub\s+(\w+)/;
  my $name=$1;
  my $fn= Function->named($name);
  return 0 if !(defined($proto_opt) && $proto_opt) && defined($fn);
  
  package run;
  eval $Function::code;
  package Function;

  $fn->{'Have_code'}= 1 if defined($fn);
  return defined($fn);
}

# Function::add_file adds the functions in given file to the functions
# available to the ruleset rules. 
# The file should be formatted as described in the tech report.
#
# add_file: class x filename ->
#
# e.g. Function->add_file($file);
#
sub add_file {
  my($class,$file)= @_;

  open(F,"<$file") || (warn "Function::add_file: could not open $file" && return);

  $_= <F>;
  while (defined($_) && ($_ !~ /^\s*sub/)) {
    Function->new($_) if s/^(\#+|)\s*(void|set|scalar|list)/$2/; # a prototype
    $_= <F>;
  }
  return unless defined($_); # end of file

  # now in routines part of file
  my($routine)= '';
  my($line)= $_;
  while (defined($line)) {
    $routine= $line;
    $line= <F>;
    while (defined($line) && ($line !~ /^\s*sub/)) {
      $routine.= $line;
      $line= <F>;
    }
    Function->add_code($routine,1);
  }
  close F;
  # end of file
}

#########

# Function::name returns the name of this object.
#
# name: Function -> name
#
# e.g. $name= $f->name();
#
sub name {
  return $_[0]->{'Name'};
}

# Function::have_code returns whether or not the code for this
# Function has been checked in.
#
# have_code: Function -> boolean
#
# e.g. $have_code= $f->have_code();
#
sub have_code {
  return $_[0]->{'Have_code'};
}

1;

