#!/pkg/bin/perl -w

# $Id: Graph.pm,v 1.23 1997/02/01 20:16:33 hoagland Exp $

# This file contains the Graph class and methods.

#require 'dump.pl';
require 'parse_dot.pl';
require 'attr.pl';
use GraphObj;
use Node;
use Edge;

package Graph;
use strict 'refs';

# override die and warn to use the main package's
use subs qw(die warn);
sub die {
  &::die(@_);
}
sub warn {
  &::warn(@_);
}

@ISA= qw(GraphObj);

# The Graph class is a container for a connected graph of the GrIDS variety.

sub BEGIN {
  $enonce=0;
  $gid= 1; # graph id for the next graph created
}

# Graph::new creates a new Graph.  If some text is given as an
# argument, then the graph is set up to that state, which is in
# DOT-like format.  If instead a Graph is given as the second
# argument, then the graph is mirrored including its attributes. The
# ruleset that this corresponds to, is given as the first argument.
# If this is not part of a ruleset, '' should be given.
#
# new: class x ruleset [x text | Graph] -> Graph
#
# e.g. $g= Graph->new('worm_detector',"digraph G { a -> b [prot=\"xyz\"]; }");
#
sub new {
  my($class,$ruleset,$arg)= @_;
  my($self);

  $self= bless GraphObj->new($gid++,());
  $self->clear_graph;
  # set up auto-computed attrs; nnodes and nedges are maintained by
  # the operations to add nodes and edges
  $self->{'ruleset'}= $ruleset;
  $self->{'gids'}= &::create_set($self->id);

  return $self unless defined($arg); # set up empty

  if (ref($arg)) { # create from another Graph
    $self->copy_into($arg);
  } else { # create from text
    $self->set_to_text($arg);
  } 
  
  return $self;
}

##########

# Graph::clear_graph overwrites the current Graph with an empty graph,
# resetting the number of nodes and edges to zero, without changing
# the ruleset this graph is associated with.
# Returns itself as a convenience.
#
# Graph::clear_graph: Graph -> Graph
#
# e.g. $g->clear_graph();
#
sub clear_graph {
  my $self=shift;

  my $ruleset= $self->{'ruleset'};
  my $gids= $self->{'gids'};
  $self->remove_attrs($self->attr_names); # remove all attributes except those starting with a '_'
  $self->set_attrs('nnodes' => 0,'nedges' => 0,'ruleset' => $ruleset,'gids' => $gids);

  $self->{'_Nodes'}= ();
  $self->{'_Edges'}= ();
  $self->{'_Edge_ids'}= ();
  return $self;
}
 
# Graph::copy_into overwrites the current Graph, replacing it with a
# virtual copy of the Graph given as an argument.  The new graph
# structure and attributes are the same as the original, but the
# actual Nodes and Edges are different, and the ruleset of the
# original has no bearing on the ruleset of this graph.
#
# Graph::copy_into: Graph x Graph -> 
#
# e.g. $g->copy_into($g2);
#
sub copy_into {
  my($self,$orig)= @_;
  
  my $ruleset= $self->{'ruleset'};
  my $gids= $self->{'gids'};

  # clear out old graph structure and attrs
  $self->clear_graph;

  # ... and set up the new

  foreach ($orig->nodes) {
    $self->add_node($_->duplicate());
  }
  foreach ($orig->edges) {
    $self->add_edge($_->duplicate($self));
  }
  $self->set_attrs($orig->attr_hash);
  $self->{'ruleset'}= $ruleset;
  &::add_to_set($self->{'gids'},&::members($gids)); # merge gids
} 

# Graph::set_to_text overwrites the current Graph, replacing it with
# the graph specified some text in DOT-like format given as an
# argument.
#
# Graph::set_to_text: Graph x text -> 
#
# e.g. $g->set_to_text("digraph G { a -> b [prot=\"xyz\"]; }");
#
sub set_to_text {
  my($self,$src)= @_;
  my(%node_attr,%edge_attr);

  # clear old graph structure
  $self->clear_graph;

  # ... and set up the new

  @_= &parse_dot::get_graphs($src);
  @_ == 1 || warn "one graph only expected in \"$src\" as argument to Graph::set_to_text";
  my($graph)= shift;
  my($name,$nodesref,$edgesref,@attrs)= &parse_dot::graph_parts($graph);
  
  my $ruleset= $self->{'ruleset'};
  my $gids= $self->{'gids'};
  # parse out attrs
  foreach (@attrs) {
    my($attrname,$val)= &parse_dot::split_attr($_);
    $self->set_attr($attrname,&::unflatten_attr($val));
  }
  ($self->{'gids'} ne $gids) && warn "Graph::set_to_text: gids provided in text ignored";
  # restore potentially overwritten auto-computed attrs
  $self->{'gids'}= $gids;
  $self->{'ruleset'}= $ruleset;
  

  # create nodes
  foreach (@{$nodesref}) {
    %node_attr= ();
    my($nodename,@attrs)= &parse_dot::node_parts($_);
    foreach (@attrs) {
      my($attrname,$val)= &parse_dot::split_attr($_);
      $node_attr{$attrname}= &::unflatten_attr($val);
    }
    $self->add_node(Node->new($nodename,%node_attr));
  }
  
  # create edges
  foreach (@{$edgesref}) {
    %edge_attr= ();
    my($srcname,$destname,@attrs)= &parse_dot::edge_parts($_);
    foreach (@attrs) {
      my($attrname,$val)= &parse_dot::split_attr($_);
      $edge_attr{$attrname}= &::unflatten_attr($val);
    }
    my($id)= $edge_attr{'id'};
    unless (defined($id)) {  # just in case an id wasn't given to us
      $id= ++$enonce; 
      warn "Graph::set_to_text: edge id not provided, using $id";
    }
    # create src and dest nodes if not already present
    $src= defined($self->{'_Nodes'}{$srcname}) ? 
      $self->{'_Nodes'}{$srcname} : 
	$self->add_node(Node->new($srcname));
    $dest= defined($self->{'_Nodes'}{$destname}) ?
      $self->{'_Nodes'}{$destname} :
	$self->add_node(Node->new($destname));
    $self->add_edge(Edge->new($src,$dest,$id,%edge_attr));
  }
}

# Graph::set_ruleset changes the ruleset associated with this Graph.
#
# Graph::set_ruleset: Graph x name -> 
#
# e.g. $g1->set_ruleset('worm_detector');
#
sub set_ruleset {
  my($self,$name)= @_;
  $self->{'ruleset'}= $name;
}

#######

# Graph::incorporate_graph adds a Graph into this Graph, ignoring the
# attributes on the intersecting parts of the incoming Graph and
# global attributes. This is essentially a structural merge of two
# Graphs.
#
# Graph::incorporate_graph: Graph x Graph -> Graph
#
# e.g. $g1->incorporate_graph($g2);
#
sub incorporate_graph {
  my($self,$graph)= @_;

  # add non-shared nodes and edges
  my($nodesref,$edgesref)= $self->difference($graph);
  foreach (@{$nodesref}) {
    $self->add_node($_);
  }
  foreach $edge (@{$edgesref}) {
    # need to update some sources and destinations
    $edge->change_src($self->node($edge->src->id))
      if $self->node_is_member($edge->src->id); # edge from this graph
    $edge->change_dest($self->node($edge->dest->id))
      if $self->node_is_member($edge->dest->id); # edge into this graph
    $self->add_edge($edge);
  }
}

# Graph::add_node adds a Node to this Graph. Returns the Node as a
# convenience.
#
# Graph::add_node: Graph x Node -> Node
#
# e.g. $g1->add_node($node);
#
sub add_node {
  my($self,$node)= @_;
  $self->{'nnodes'}++ unless $self->{'_Nodes'}{$node->id};
  $self->{'_Nodes'}{$node->id}= $node;
  return $node;
}

# Graph::add_edge adds an Edge to this Graph. Returns the Edge as a
# convenience.
#
# Graph::add_edge: Graph x Edge -> Edge
#
# e.g. $g1->add_edge($edge);
#
sub add_edge {
  my($self,$edge)= @_;
  $self->{'nedges'}++ unless $self->{'_Edges'}{$edge->id};
  $self->{'_Edges'}{$edge->id}= $edge;
  push(@{$self->{'_Edge_ids'}{$edge->src->id}{$edge->dest->id}},$edge->id);
  return $edge;
}



########

# Graph::nodes returns nodes in this Graph.
#
# Graph::nodes: Graph -> {Node}
#
# e.g. @nodes= $g1->nodes();
#
sub nodes {
  return values %{$_[0]->{'_Nodes'}};
}

# Graph::edges returns edges in this Graph.
#
# Graph::edges: Graph -> {Edge}
#
# e.g. @edges= $g1->edges();
#
sub edges {
  return values %{$_[0]->{'_Edges'}};
}

# Graph::node returns the Node with the given id in this Graph or an
# undefined value.
#
# Graph::node: Graph x id -> Node
#
# e.g. $node= $g1->node('abc');
#
sub node {
  return ${$_[0]->{'_Nodes'}}{$_[1]};
}

# Graph::edge returns the Edge with the given id in this Graph or an
# undefined value.
#
# Graph::edge: Graph x id -> Edge
#
# e.g. $edge= $g1->edge('abc');
#
sub edge {
  return ${$_[0]->{'_Edges'}}{$_[1]};
}

# Graph::edge_from_to returns a list of Edges starting from the given
# node id and ending in a given node id in this Graph.
#
# Graph::edge_from_to: Graph x id x id -> {Edge}
#
# e.g. @edges= $g1->edge_from_to('abc','xyz');
#
sub edge_from_to {
  return map($_[0]->{'_Edges'}{$_},@{$_[0]->{'_Edge_ids'}{$_[1]}{$_[2]}});
}

# remove an edge from the graph but leave all nodes alone
sub remove_edge {
  my($self,$eid)= @_;
  my($edge)= delete $self->{'_Edges'}{$eid};
  return unless defined($edge); # no such edge in here so ignore
  my($srcid,$destid)= ($edge->src->id,$edge->dest->id);
  @{$self->{'_Edge_ids'}{$srcid}{$destid}}= grep($_ ne $eid,@{$self->{'_Edge_ids'}{$srcid}{$destid}});
  $self->{'nedges'}--;
  return $edge;
}

# rename (re-id) the node for all purposes inside the graph, including the Node itself.
sub rename_node {
  my($self,$oldname,$newname)= @_;
  $self->{'_Nodes'}{$newname}= delete $self->{'_Nodes'}{$oldname};
  $self->{'_Nodes'}{$newname}->{'name'}= $newname;
  $self->{'_Nodes'}{$newname}->{'_id'}= $newname;
  $self->{'_Edge_ids'}{$newname}= delete $self->{'_Edge_ids'}{$oldname};
  foreach (keys %{$self->{'_Edge_ids'}}) {
    $self->{'_Edge_ids'}{$_}{$newname}= delete $self->{'_Edge_ids'}{$_}{$oldname} if defined($self->{'_Edge_ids'}{$_}{$oldname});
  }
}
     
# Graph::node_ids returns the node ids of the nodes in this Graph.
#
# Graph::node_ids: Graph -> node_id x node_id x ...
#
# e.g. @nodes= $g1->node_ids();
#
sub node_ids {
  return keys %{$_[0]->{'_Nodes'}};
}

# Graph::edge_ids returns the edge ids of the edges in this Graph.
#
# Graph::edge_ids: Graph -> edge_id x edge_id x ...
#
# e.g. @edges= $g1->edge_ids();
#
sub edge_ids {
  return keys %{$_[0]->{'_Edges'}};
}

# Graph::node_is_member returns whether the given node id is a member
# of this Graph.
#
# Graph::node_is_member: Graph x node_id -> boolean
#
# e.g. if ($g1->node_is_member('d32f')) { ...
#
sub node_is_member {
  return defined(${$_[0]->{'_Nodes'}}{$_[1]});
}

# Graph::edge_is_member returns whether the given edge id is a member
# of this Graph.
#
# Graph::edge_is_member: Graph x edge_id -> boolean
#
# e.g. if ($g1->edge_is_member('d32f')) { ...
#
sub edge_is_member {
  return defined(${$_[0]->{'_Edges'}}{$_[1]});
}

#########

# Graph::dot_text returns the text for this Graphs in the Graph
# Language (which is like DOT).  If an optional true argument is
# given, then "subgraph" is used for the start of the graph text
# rather than the default "digraph".
#
# dot_text: Graph [x boolean ] -> string
#
# e.g. $text= $graph->dot_text();
#
sub dot_text {
  my($self,$use_subgraph)= @_;
  my($node,$edge);
  my($text)= defined($use_subgraph) && $use_subgraph ? "subgraph" : "digraph";
  $text.= " ".($self->id?&::dot_encode($self->id):'G')." {\n";
  my $attrtext= $self->attrs_dot_text(";\n\t");
  $text.= "\t$attrtext;\n" if $attrtext; # only if some global attributes
  foreach $node ($self->nodes) {
    $text.= "\t".&::dot_encode($node->id)." [";
    $text.= $node->attrs_dot_text(',');
    $text.= "];\n";
  }
  foreach $edge ($self->edges) {
    $text.= "\t".&::dot_encode($edge->src->id)." -> ".&::dot_encode($edge->dest->id)." [";
    $text.= $edge->attrs_dot_text(',');
    $text.= "];\n";
  }  
  $text.= "}\n";
  return $text;
}

#########

# Graph::merge_auto_computed sets the res attributes in terms of the
# cur attributes and those of a given new one for auto-computed
# attributes.  It is called at the time that the global combining
# rules are being on the graphs.  This function is called before the
# graph structures are combined, so res, at this point, is the same
# size as cur, so the 'nedges' and 'nnodes' attributes are copied
# directly over from cur.  When new gets joined in structurally, then
# those attribute will get updated again. 'ruleset' gets copied over
# (it should be the same in both graphs). However, 'gids' becomes the
# union of the gids.
#
# merge_auto_computed: Graph x Graph ->
#
# e.g. $graph->merge_auto_computed;
#
sub merge_auto_computed {
  my ($self,$new)=@_;
  $self->res_attrs->set_attrs('nedges' => $self->{'nedges'},
			      'nnodes' => $self->{'nnodes'},
			      'ruleset' => $self->{'ruleset'},
			      'gids' => &::create_set(&::members($self->{'gids'}),&::members($new->{'gids'})));
}

# Graph::auto_computed_to_res moves the auto-computed attributes of
# the Graph (namely 'nedges', 'nnodes', 'ruleset', 'gids') to the res
# attributes.
#
# auto_computed_to_res: Graph ->
#
sub auto_computed_to_res {
  my $self=shift;
  $self->res_attrs->set_attrs('nedges' => $self->{'nedges'},
			      'nnodes' => $self->{'nnodes'},
			      'ruleset' => $self->{'ruleset'},
			      'gids' => $self->{'gids'});
}

#######

# Graph::is_adjacent returns whether or not this Graph has any nodes
# in common with the Graph given in the argument.  Sameness of nodes
# is based on their id.  Note that having the Graph given as the
# argument have fewer nodes than this one should be somewhat more
# efficient that the other way around.
#
# Graph::is_adjacent: Graph x Graph -> boolean
#
# e.g. if ($g1->is_adjacent($g2)) { ...
#
sub is_adjacent {
  my($self,$graph)= @_;
  foreach ($graph->node_ids) {
    return 1 if $self->node_is_member($_);
  }
  return 0;
}

# Graph::intersection returns the node and edge ids that are in the
# intersection of this and the given Graph.  Sameness of nodes and
# edges is based on their id. Note that having the Graph given as the
# argument be smaller than this one should be somewhat more efficient
# that the other way around.
#
# Graph::intersection: Graph x Graph -> node_ids-ref edge_ids-ref
#
# e.g. ($node_idsref,$edge_idsref)= $g1->intersection($g2);
#
sub intersection {
  my($self,$graph)= @_;
  my(@node_ids,@edge_ids)= ();
  foreach ($graph->node_ids) {
    push(@node_ids,$_) if $self->node_is_member($_);
  }
  foreach ($graph->edge_ids) {
    push(@edge_ids,$_) if $self->edge_is_member($_);
  }
  return (\@node_ids,\@edge_ids);
}

# Graph::difference returns the Nodes and Edges that are in the the
# given Graph, but not in this graph.  Sameness of nodes and edges is
# based on their id. Note that having the Graph given as the argument
# be smaller than this one should be somewhat more efficient that the
# other way around.
#
# Graph::difference: Graph x Graph -> nodes-ref edges-ref
#
# e.g. ($node_idsref,$edge_idsref)= $g1->difference($g2);
#
sub difference {
  my($self,$graph)= @_;
  my(@nodes,@edges)= ();
  foreach ($graph->node_ids) {
    push(@nodes,$graph->node($_)) unless $self->node_is_member($_);
  }
  foreach ($graph->edge_ids) {
    push(@edges,$graph->edge($_)) unless $self->edge_is_member($_);
  }
  return (\@nodes,\@edges);
}

1;
