#!/pkg/bin/perl -w

# $Id: GraphObj.pm,v 1.14 1997/02/01 20:17:12 hoagland Exp $

# This file contains the GraphObj class and methods.

# The GraphObj class is a class which, in addition to storing attributes like
# the AttrObj class on which it is based, stores some other information.
# This other information is an ID and an Graph that is "associated" with this
# object. 
#
#  In addition to the regular set of attributes, this class also implements
#  storage of scratch (in progress) attributes known as "res" attributes
#  which can copied over into the regular set of attributes (also known as
#  the "cur" attributes).

#-------------------------

use AttrObj;

package GraphObj;   
use strict 'refs';

# override die and warn to use the main package's
use subs qw(die warn);
sub die {
  &::die(@_);
}
sub warn {
  &::warn(@_);
}


@ISA= qw(AttrObj);

# GraphObj::new creates a new GraphObj with the given attributes and
# id
#
# new: class x id {attr_name x val} -> GraphObj
#
# e.g. $node= GraphObj->new($id,'a' => 'b', 'c' => 'd');
#
sub new {
  return bless AttrObj->new('_id' => $_[1], '_res_attrs' => AttrObj->new, @_[2 .. $#_]);
}

# GraphObj::id returns the id of this object.
#
# id: GraphObj -> id
#
# e.g. $id= $edge->id();
#
sub id {
  return $_[0]->{'_id'};
}

# GraphObj::res_attrs returns the "res" attributes of this object.
#
# res_attrs: GraphObj -> AttrObj
#
# e.g. $res_attrs= $edge->res_attrs();
#
sub res_attrs {
  return $_[0]->{'_res_attrs'};
}

######

# GraphObj::move_res_to_cur moves the "res" attributes of this object
# into the regular attribute area, overriding all old attributes.
# Pseudo-attributes beginning with "_" on the cur side are left alone
# though.
#
# res_attrs: GraphObj -> 
#
# e.g. $graph->move_res_to_cur;
#
sub move_res_to_cur {
  my ($self)= shift;
  $self->clear_attrs;
  $self->set_attrs($self->{'_res_attrs'}->attr_hash);
  $self->{'_res_attrs'}->clear_attrs;
  return $_[0]->{'_res_attrs'};
}

# GraphObj::clear_res clears the "res" attributes of this object.
#
# res_attrs: GraphObj -> 
#
# e.g. $graph->clear_res;
#
sub clear_res {
  my ($self)= shift;
  $self->{'_res_attrs'}->clear_attrs;
}

#######

# GraphObj::set_id resets the id of this object.
#
# set_id: GraphObj x id ->
#
# e.g. $edge->set_id($id);
#
sub set_id {
  $_[0]->{'_id'}= $_[1];
}

#######

# GraphObj::merge_auto_computed is a placeholder for a method that
# derived classes must override.  merge_auto_computed sets the res
# attributes in terms of the cur attributes and those of a new one for
# auto-computed attributes.  It is called at the time that the
# corresponding combining rules are being run.
#
# merge_auto_computed: GraphObj x GraphObj ->
#
sub merge_auto_computed {
  warn("oops, derived class of GraphObj (".ref($_[0]).") must override merge_auto_computed");
}

# GraphObj::auto_computed_to_res is a placeholder for a method that
# derived classes must override.  auto_computed_to_res moves the
# auto-computed attributes of the GraphObj to the res attributes.
#
# auto_computed_to_res: GraphObj ->
#
sub auto_computed_to_res {
  warn("oops, derived class of GraphObj (".ref($_[0]).") must override auto_computed_to_res");
}
