#!/pkg/bin/perl -w

# $Header: /home/cvs/grids/code/engine/execute.pl,v 1.24 1997/03/18 05:49:49 hoagland Exp $

# This file contains code to execute ruleset rules.

require "parse.pl";
use Function;
require 'check_rules.pl';

package execute;

# override die and warn to use the main packages
use subs qw(die warn);
sub die {
  &::die(@_);
}
sub warn {
  &::warn(@_);
}

# init adds the primitive functions and the functions in the files
# given as arguments to the set of functions available to ruleset rule
# exectution
#
# init: files ->
#
# e.g. &execute::init(@files);
#
sub init {
  my ($file);
  foreach $file (@_) {
    Function->add_file($file);
  }
  %fns= (); # storage for functions built to evaluate rules.
            # $fns{$rule}{join(':',@bindkeys)} is a reference to a
            # subroutine for use where the valid binding names are
            # those in @bindkeys
  $rulecount= 1; # used for uniqueness in evaling subroutines
  $run::graph= ''; # quiet warning messages
}

########

# preprocess_ruleset take the source text of a ruleset and breaks it
# down into parts. These parts are individual rules or declaration
# values.  Macros are expanded before being returned.  Assesment rules
# are returned as a reference to a list whose elements are references
# to a (test,action1,action2,...) list.
#
# preprocess_ruleset: text -> set_name x buffer_time x graph_timeout x
#  attr_decl-list-ref x node_precond x edge_precond x
#  global_rule-list-ref x node_rule-list-ref x edge_rule-list-ref x
#  report_global-list-ref x report_node-list-ref x report_edge-list-ref x
#  assesment_rule-list-ref
#
# e.g. ($name,$buffer_time,$graph_timeout,$attr_declsref,
#  $node_precond,$edge_precond,$global_rulesref,$node_rulesref,
#  $edge_rulesref,$assesmentsref)= &preprocess_ruleset($src);
#
sub preprocess_ruleset {
  my(@lines)= grep(!(/^\s*$/ || /^\s*\#/),split("\n",$_[0]));

  my ($line,$map,$macro);

  my($name,$buffer_time,$graph_timeout,$attr_declsref,$node_precond,$edge_precond,$report_globalref,$report_noderef,$report_edgeref,$global_rulesref,$node_rulesref,$edge_rulesref);

  my(@tmp)= @lines; @lines=();
  foreach (@tmp) {
    s/^\s+//; s/\s+$//;
    push(@lines,$_);
  }

  @tmp= @lines; @lines=();
  while ($_= shift(@tmp)) {
    $line= $_;
    while ($line !~ /[\{\}\;]$/)  {
      $line.= ' ' . shift(@tmp);
    }
    push(@lines,$line);
  }
  
  # @lines all now start with a non-whitespace and end in ';', '{', or
  # '}'
  #print join("\n",map("> $_",@lines)),"\n";
  
  $_=shift(@lines);
  /^ruleset\s+([^\s;\{\}]+)/ || die "preprocess_ruleset: name not on first line \"$_\"";
  my $name= $1;
  
  $_=shift(@lines);
   /^buffer\s+(\d+)/ || die "preprocess_ruleset: buffer time not specified at \"$_\""; 	
  my $buffer_time= $1;
  
  $_=shift(@lines);
  /^timeout\s+(\d+)/ || die "preprocess_ruleset: graph timeout not specified at \"$_\"";
  my $graph_timeout= $1;
  
  # find all macros
  $_=shift(@lines);
  /^macros/ || die "preprocess_ruleset: macro section expected at \"$_\"";
  my(%macros)= ();
  while (($_= shift(@lines)) ne '}') {
    s/;$//;
    my($name,$val)= split(/=\s*/,$_);
    $macros{$name}= $val;
  }

  # expand out all macros
  foreach $macro (keys %macros) {
    @_= @lines; @lines= ();
    foreach (@_) {
      s/\b$macro\b/$macros{$macro}/g;
      push(@lines,$_);
    }
  }


  $_=shift(@lines);
  /^attribute\s+declarations/ || die "preprocess_ruleset: attribute declarations section expected at \"$_\"";
  ($attr_declsref,@lines)= &::grab_rule_block(@lines);
  

  $_=shift(@lines);
  /^node\s+precondition (.*)/ || die "preprocess_ruleset: node precondition expected at \"$_\"";
  my $node_precond= $1;
  
  $_=shift(@lines);
  /^edge\s+precondition (.*)/ || die "preprocess_ruleset: edge precondition expected at \"$_\"";
  my $edge_precond= $1;
  $edge_precond =~ s/;$//;
  

  $_=shift(@lines);
  /^report\s+global\s+rules/ || die "preprocess_ruleset: report global rules section expected at \"$_\"";
  ($report_globalref,@lines)= &::grab_rule_block(@lines);
  
  $_=shift(@lines);
  /^report\s+node\s+rules/ || die "preprocess_ruleset: report node rules section expected at \"$_\"";
  ($report_noderef,@lines)= &::grab_rule_block(@lines);

  $_=shift(@lines);
  /^report\s+edge\s+rules/ || die "preprocess_ruleset: report edge rules section expected at \"$_\"";
  ($report_edgeref,@lines)= &::grab_rule_block(@lines);
  

  $_=shift(@lines);
  /^global\s+rules/ || die "preprocess_ruleset: global rules section expected at \"$_\"";
  ($global_rulesref,@lines)= &::grab_rule_block(@lines);
  
  $_=shift(@lines);
  /^node\s+rules/ || die "preprocess_ruleset: node rules section expected at \"$_\"";
  ($node_rulesref,@lines)= &::grab_rule_block(@lines);

  $_=shift(@lines);
  /^edge\s+rules/ || die "preprocess_ruleset: edge rules section expected at \"$_\"";
  ($edge_rulesref,@lines)= &::grab_rule_block(@lines);
  
  $_=shift(@lines);
  /^assessments/ || die "preprocess_ruleset: assesment section expected at \"$_\"";
  my(@assesments)= ();
  while (($_= shift(@lines)) ne '}') {
    s/;$//;
    my($test,$actions)= split(/\s*==>\s*/,$_);
    ($actions,$map)= &::encode_quotes($actions);
    my(@actions)= map(&::restore_quotes($_,$map),&::top_level_pieces($actions));
    push(@assesments,[$test,@actions]);
  }
  
  @lines && warn("preprocess_ruleset: leftover lines:".join("\n",@lines));

  return ($name,$buffer_time,$graph_timeout,
	  $attr_declsref,$node_precond,$edge_precond,
	  $report_globalref,$report_noderef,$report_edgeref,
	  $global_rulesref,$node_rulesref,$edge_rulesref,
	  \@assesments);
}


# separate_rules divides the given node or edge rules into different
# kinds.  The kinds are the combine-indicating rule and other rules.
#
# separate_rules: {rule} -> combine_rule {other_rule}
#
# e.g. ($combine,@rules)= &separate_rules(@edge_rules);
#
sub separate_rules {
  my(@rules)= ();
  my($combine)= undef;

  # ${$bindings}{'res.node'}->set_attr('combine',...)

  foreach (@_) {
    if (!/^\s*(new|cur)/ && /\.combine\s*=[^~=]/) { # a combine rule
      defined($combine) && warn("separate_rules: multiple combine rules found; \"$combine\" ignored");
      $combine= $_;
    } else {
      push(@rules,$_);
    }
  }
  warn "no combine rule found" unless defined($combine);
  return ($combine,@rules);
}

# preprocess_rule takes the given single rule and does some
# preprocessing to make a rule that is executable by a perl eval.  The
# scalar result is returned.
#
# preprocess_rule: rule -> preprocessed_rule
#
# e.g. $rule= &preprocess_rule($rule);
#
sub preprocess_rule {
  $_= shift;
  #print "$_ ==>\n";
  my($qmap,$pmap,$wmap);
  my($bsechar,$qechar); # the encodeing chars used by encode_quotes
                        # for the backslashed chars and the double
                        # quote

  # hide quotes
  ($_,$qmap)= &::encode_quotes($_);
  (undef,$bsechar,$qechar)= @{$qmap};

  # hide regex '/' pairs
  ($_,$pmap)= &::encode_quotes($_,'/');
  
  # hide "eq", "lt", "gt" for their protection
  ($_,$wmap)= &::hide_words($_,'eq','lt','gt','ne');
  
  s/;$//g;

  s/^\s*/ /; # make sure there is an extra character out front

  # protect built-in functions being called
  # defined( => \xEDdefined(
  # time( => \xEDtime(
  s/\b((defined|time)\s*\()/\xED$2(/g;

  # foo ( -> &foo(
  while (s/([^&\w\x80-\xFF])(\w+)\s*\(/defined(Function->named($2)) ? "$1&$2\(" : "$1$2"."\xF0"/e) {};
  s/\xF0/\(/g;

  # take care of {} and [] list and set constructors
  while (s/\[([^\[\]\{\}]*)\]/"\xEEcreate_list(".&member_list($1).')'/e 
	 || s/\{([^\[\]\{\}]*)\}/"\xEEcreate_set(".&member_list($1).')'/e) {};
  s/\xEF/,/g;

  # process references to attribute sets rather than individual attrs
  # a.node => ${$bindings}{'a.node'}   same for edge,global,source,dest
  while (s/([^\.\$\'\&\xEE\xED])(($qechar\d+$qechar|\b[a-zA-Z]\w*\.)*)(\b(node|edge|global|source|dest)\b)([^.])/$1\${\$bindings}{\'$2$4\'}$6/g) {};

  # a.b.c => ${$bindings}{'a.b.'}.c 
  while (s/([^\.\$\'\&\xEE\xED])(($qechar\d+$qechar|\b[a-zA-Z]\w*\.)*)(\b[a-zA-Z]\w*)/$1\${\$bindings}{\'$2\'}.$4/g) {};

  # ${$bindings}{'a.b.'}.c => ${$bindings}{'a.b'}.c 
  s/\.\'/\'/g;

  # ${$bindings}{'new.global'}.a= ... => ${$bindings}{'new.global'}->set_attr('a',&::copy_attr(...))
  # note that assignments only happen at the start of a rule
  s/^(\s*\$\{\$bindings\}\{\'[^\`]*\'})\.(\w+)\s*=([^~=].*)/$1.'->set_attr(\''.$2."\',\xEEcopy_attr($3))"/e;

  # ${$bindings}{'new.global'}.a => ${$bindings}{'new.global'}->{'a'}
  while (s/(\s*\$\{\$bindings}\{\'[^\`]*\'})\.(\w+)/$1\->\{\'$2\'\}/) {};
  
  # :: => .  string concatination
  s/::/./g;

  # \xEE => &::
  s/\xEE/&::/g;

  # delete \xED markers
  s/\xED//g;

  # try to remove extra whitespace for better optimization in evaluate
  # and for uniformity
  s/^\s+//g;
  s/\s+$//g;
  s/([\{\(])\s+/$1/g;
  s/\s+([\)\}])/$1/g;
  s/\s\s+/ /g;

  # unhide words
  $_= &::restore_words($_,$wmap);

  # unhide regex '/' pairs
  $_= &::restore_quotes($_,$pmap);

  # restore quotes
  $_= &::restore_quotes($_,$qmap);
  
  #print "$_\n";

  return $_;
}


# member_list is an internal function to take some text of the inside
# of a set or list constructor and return the code to put inside a
# call to the constructor function, except that \xEF is used in place
# of a comma and \xEE is used in place of '&::'.  It is assumed there
# are no extraneous commas inside the passed in string
#
# member_list: string -> string
#
# e.g. '&::create_list('.&member_list($1).')'
#
sub member_list {
  my(@parts)= split(',',$_[0]);
  return join("\xEF",map(/^\xFF/ || /^\d/ ? $_ : "\xEE".'members('.$_.')',@parts));
}


# rule_sub takes a given rule and returns a reference to a function in
# the 'run' package.  The returned function is invoked with a
# reference to a bindings hash, which binds attribute set names to
# AttrObjs.  The valid attribute set names are given as an argument to
# this function for checking against the rule.  All neccessary
# preprocessing for rules are taken car of by this function.
# 
# rule_sub: rule { bind_key }-> subroutine-ref
#
# e.g. $res= &{&rule_sub($rule)}({'res.global'=> $graph, 'global' => $graph});
#
sub rule_sub {
  my($rule,@bindkeys)= @_;
  my $bindstring= join(':',@bindkeys);
  my($pprule);

#  print "getting subroutine for $rule\n";
  return $fns{$rule}{$bindstring} if defined($fns{$rule}{$bindstring});

  # see if removing extra whitespace from the rule causes a cache hit
  $_= $rule;
  s/^\s+//g;
  s/\s+$//g;
  s/\s*;$//g; # semicolon at end doesn't change semantics
  s/\s\s+/ /g;
  s/([\(\=])\s+/$1/g;
  s/\s+([\)\=])/$1/g;
  if (defined($fns{$_}{$bindstring})) { # it worked!
    return $fns{$rule}{$bindstring}= $fns{$_}{$bindstring}; # remember the rule
                                                     # stripped as well
  }

  #print "rule= $rule\n";
  $pprule= &preprocess_rule($rule);
  #print "pprule= $pprule\n";

  # make sure all the bindings refered to are valid
  my $key;
  $_= $pprule; # make a modifiable copy of the rule
  foreach $key (@bindkeys) {
    s/\$bindings\}\{\'$key\'\}//g;
  }
  my $bad=0;
  while (/\$bindings\}\{\'([\.\w]*)\'\}/g) {
    warn "attribute set '$1' in rule '$rule' is not valid in current context only ".join(',',@bindkeys)." are allowed); '1' used as rule instead";
    $bad++;
  }
  return eval 'sub rule'.$rulecount.' {1;} \&rule'.$rulecount++.';' if $bad;

  # $eval is a global so it can be used in run package
  $eval= 'sub rule'.$rulecount.' { $bindings=$_[0]; return ('.$pprule.'); } \&rule'.$rulecount++.';';
#  print "evaling $eval...\n";
  package run;
  $sub= eval $execute::eval;
  die "eval error for $execute::eval: $@\n" if $@;
  package execute;
  
#  print "$run::sub is the result\n";
  warn "code for rule \"$rule\" is \"$eval\"; eval result was $run::sub";

  return $fns{$rule}{$bindstring}= $run::sub;
}
  

# evaluate executes the given rule and general textual description
# with the given bindings of symbolic names, i.e. begining with
# "new,cur,res", to AttrObj's, returning the result. 
#
# evaluate: rule x text x {text x AttrObj} -> result
#
# e.g. if (&evaluate($rule,'rule to do nothing',
# 		   'new.global' => $report,
#		   'new.node' => $report->node($_),
#	           'cur.global' => $graph,
#		   'cur.node' => $graph->node($_))) { ...
#
sub evaluate {
  my($rule,$descr);
  ($rule,$descr,%bindings)= @_; # bindings is global

#  print "$rule ==>\n";
#  print "Running \"$rule\" $descr\n";
#  print "bindings: ",join(',',map($_.' => '.$bindings{$_},keys %bindings)),"\n";
  
  $run::graph= $bindings{'res.global'}; # make graph avail for alerts    

  my $subref= &rule_sub($rule,keys %bindings); # get a subroutine to call to evaluate this rule
  $result= &{$subref}(\%bindings);
#  print "result= $result\n";
    
  return $result;
}


1;
