#!/pkg/bin/perl

# $Id: log.pl,v 1.8 1996/08/24 09:51:24 crawford Exp $

# The functions in this file assume that a global constant $DUMPSIZE
# exists.  $DUMPSIZE is used as the buffer length for writing to the
# GLF and RLFs.  $invocation is the invocation number for this
# instance of the engine and is internally established. &start_logging should be called before any
# other calls to the debugging logger and &stop_logging should be
# called before the engine exits.

$PATH_PREFIX = $ENV{'GRIDSPATH'} ? "$ENV{'GRIDSPATH'}/log/" : '';

# start_logging initializes the routines the do the debugging logging
# and logs that startup of the engine.  This routine expects $_[0] to
# contain the time of the startup, $_[1] to contain a reference to a
# <file-list>, and $_[2] to contain a reference to a <control-list>. 
sub start_logging {
  my($time,$file_list_ref,$control_list_ref) = @_;
  # get invocation no.
  $invocation= $PATH_PREFIX . $$;
  (-d $invocation) || mkdir("$invocation",0755) || die "could not make debug log directory $invocation";
  &log_startup($time,$invocation,$file_list_ref,$control_list_ref);  
}

# stop_logging logs the shutdown of this instance of the engine and
# flushed all buffers.  This routine should be always be called before
# exiting or data could be lost.  The routine expects $_[0] to contain
# the time of the shutdown.
sub stop_logging {
  &log_shutdown($_[0],$invocation);
  foreach (keys %bufferlength) { # includes GLF
    &flush_RLF($_);  # better flush before quiting!
  }
}

# buffer_GLF writes the string in $_[0] to the
# Global Log File via a buffer.  The buffer length
# is defined by $DUMPSIZE.  The Global Log File
# name is defined by $logname{"GLF"}.
#
sub buffer_GLF {
  buffer_RLF($_[0],'GLF');
}

# buffer_RLF buffers output to a Ruleset Log File
# in the global hash %logfilebuffer.  It expects
# $_[0] to be the string to be buffered and
# $_[1] to be the name of the ruleset.  The buffer
# length is defined by $DUMPSIZE and the log file
# name is defined by $logname{$_[1]}.
#
sub buffer_RLF {
  $logfilebuffer{$_[1]} = '' unless defined $logfilebuffer{$_[1]}; #uninit noise
  $logfilebuffer{$_[1]} .= $_[0];
  $bufferlength{$_[1]} = 0 unless defined $bufferlength{$_[1]};  # quiet uninit
  &flush_RLF($_[1]) if( ($bufferlength{$_[1]} += length($_[0])) > $DUMPSIZE);
}

# Flushes the global log file buffer.
#
sub flush_GLF {
  &flush_RLF('GLF');
}

# Flushes the buffer for the specified ruleset.
# Expects $_[0] to contain the name of the ruleset or 'GLF'.
#

sub flush_RLF {
  my ($filename);
  if ($_[0] eq 'GLF') {
    $filename= $invocation.'/engine.log';
  } else {
    $filename= $invocation.'/ruleset.'.$_[0].'.log';
  }
  unless (open(HANDLE,">>$filename")) {
    warn "could not append to debugging log $filename";
    return;
  }
  print HANDLE $logfilebuffer{$_[0]};
  $logfilebuffer{$_[0]} = "";
  $bufferlength{$_[0]} = 0;
  close(HANDLE);
}


# log_startup logs a start up of the engine in the
# Global Log File.  It expects $_[0] to contain the
# time of the startup, $_[1] to contain the invocation
# number, $_[2] to contain a reference to a <file-list>,
# and $_[3] to contain a reference to a <control-list>.
# (A <file-list> is a list whose even numbered entries
# are filenames and odd numbered entries are 
# revision numbers.  A <control-list> is a list of 
# <control-value>'s, where a control value is of the
# form <id> "=" <id> "\n"; 
# 
sub log_startup {
  my($time,$invocation,$file_list_ref,$control_list_ref) = @_;
  &buffer_GLF(">".$time.";".$invocation."\nversion {\n");
  &log_file_list(@{$file_list_ref});
  &buffer_GLF("}\ncontrol {\n");
  &log_control_list(@{$control_list_ref});
  &buffer_GLF("}\n");
}
  
# log_shutdown logs a shut down of the engine in the
# Global Log File.  It expects $_[0] to contain the
# time of the shutdown and $_[1] to contain the
# invocation number.  log_shutdown does not actually
# perform a shutdown, nor does it flush the log
# buffers.  Hence, the shutdown may not be logged
# if the shutdown does not flush the buffers before
# exiting.
#
sub log_shutdown {
  &buffer_GLF('<'.$_[0].';'.$_[1]."\n");
}

# log_file_id logs a <file-id> in the Global Log File.
# It expects $_[0] to contain the filename and $_[1]
# to contain the revision number.
#
sub log_file_id {
  buffer_GLF($_[0].": ".$_[1]."\n");
}

# log_control_value logs a <control-value> in the
# Global Log File.  It expects $_[0] to contain the
# variable name (the first <id>) and $_[1] to
# contain the value (the second <id>).
#
sub log_control_value {
  buffer_GLF($_[0]."=".$_[1]."\n");
}

# log_control_change logs a <control-change> in the
# Global Log File.  It expects $_[0] to contain the
# name of the control variable whose value is changing,
# and $_[1] to contain the new value.  This function
# should not be used to log ruleset changes.
#
sub log_control_change {
  buffer_GLF("change: ");
  log_control_value($_[0],$_[1]);
}

# log_ruleset_change logs a change to a ruleset
# in the Global Log File.  It expects $_[0] to contain
# the ruleset name and $_[1] to contain the new
# ruleset text.
#
sub log_ruleset_change {
  &buffer_GLF("ruleset change: ");
  &log_control_value($_[0],$_[1]);
}

# log_control_list logs a <control-list> in the
# Global Log File.  It expects @_ to contain pairs
# of variable name and new value.  For even numbered
# k, @_[k] contains a variable name and @_[k+1]
# contains the associated value.  @_ may contain
# zero or more pairs.
#
sub log_control_list {
  my @args = @_;
  while(scalar(@args) > 0)
  {
    my $var = shift(@args);
    my $val = shift(@args);
    log_control_value($var,$val);
  }
}

# log_file_list logs a <file-list> in the
# Global Log File.  It expects @_ to contain pairs
# of filenames and revision numbers.  For even numbered
# k, @_[k] contains a file name and @_[k+1] contains
# the revision number of that file.  @_ may contain
# zero or more pairs.
#
sub log_file_list {
  my @args = @_;
  while(scalar(@args) > 0)
  {
    my $name = shift(@args);
    my $num = shift(@args);
    log_file_id($name,$num);
  }
}

# log_acceptance logs a <report-number> in the
# Ruleset Log File for the specified ruleset.
# It expects $_[0] to contain the report number
# and $_[1] to contain the ruleset name.
#
sub log_acceptance {
  my($rulesetname,$report_id) = @_;
  &buffer_RLF($report_id."\n",$rulesetname);
}

# log_rejectance logs a <report-number> in the
# Ruleset Log File for the specified ruleset.
# It expects $_[0] to contain the report number
# and $_[1] to contain the ruleset name.
#
sub log_rejectance {
  my($rulesetname,$report_id) = @_;
  &buffer_RLF("!".$report_id."\n",$rulesetname);
}

# log_graphspace logs a <graphspace> in the
# Ruleset Log File for the specified ruleset.
# It expects $_[0] to contain the ruleset
# name and $_[1] through $_[n] to contain the
# n graphs in the space.  (You don't specify
# n.)  The n graphs should be text, not references
# to graphs.
#
sub log_graphspace {
  my $rulesetname = shift;
  &buffer_RLF("graphspace {\n",$rulesetname);
  my $graph;
  foreach $graph (@_) {
    &buffer_RLF($graph,$rulesetname);
  }
  &buffer_RLF("\n}\n",$rulesetname);
}

# log_comment logs a comment, or list of comments
# in the Global Log File.  All comments must be
# in text format.
#
sub log_comment {
  foreach $_ (@_) {
    # put a "#" before each interior newline
    s/\n/\n#/;

    # put a "#" before start of string and buffer it.
    &buffer_GLF("#".$_);
  }
}


1;
