#!/pkg/bin/perl -w
#!/pkg/bin/perl -wd
#!/pkg/bin/perl -w:DProf


# $Id: module_controller.pl,v 1.36 1998/02/17 19:38:39 rowe Exp $


# File <module_controller.pl> contains most stuff for Module Controller.
# It is divided into the following sections:

#   %%%%%%%%%%%%%%%%%%%%%%%%%%% GLOBALS & MAIN %%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%% PRIMARY ROUTINES %%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%% TASK MANAGEMENT %%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%% SPECIAL SUB-COMMANDS %%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%% INITIALIZATION %%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%% UTILITIES %%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #



#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%% GLOBALS & MAIN %%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #

use control_vars;
use Comm;
use Clog;
use Fcntl;

$MC_RECV_PORT = 23032;  # "known" TCP port for all module controllers.
                       # What if it's not available when I come up?

$MAX_CLIENT = 3;  # keep open one TCP connection to my dept's Software Manager,
                  # maybe one for Clog,
                  # and should only need one other to handle rest of world.

$MAX_READ = 265000;   # max attempted read length from a file

# Constants for flock call:  (on solaris, see /usr/ucbinclude/sys/files.h)
# $LOCK_EX = 2;
# $LOCK_NB = 4;     # NB = NoBlock; binary-OR this with $LOCK_EX if desired.
                    # In that case, an UN_lock call returns -1 (vs. 0),
                    # and global $errno == EWOULDBLOCK
# $LOCK_UN = 8;


         ### CAUTION:  SYSTEM DEPENDENCIES::

$hostname = `/bin/uname -n` if ( -e "/bin/uname" );
$hostname = `/usr/bin/uname -n` if ( -e "/usr/bin/uname" );

chomp $hostname;

# sunos vs. solaris represent defunct zombie procs differently:
$uname = `/bin/uname -a` if ( -e "/bin/uname" );
$uname = `/usr/bin/uname -a` if ( -e "/usr/bin/uname" );
if ($uname =~ /SunOS \S+ 5[\d\.]+ Generic/) {
  $uname = "solaris";
  $ps_comm = '/usr/ucb/ps ';
  $WNOHANG = 0100;
  }
elsif ($uname =~ /IRIX/) {
  $uname = "irix";
  $ps_comm = '/bin/ps ';
  $WNOHANG = 0100;
  }
else {
  $uname = "sunos";
  $ps_comm = '/bin/ps -';
  $WNOHANG = 1;
  }
# use POSIX "wait_h";   ## doesn't work; so must define WNOHANG manually.

                     # modules/tasks now running:
$CURRENT_TASK_FILE = "mc.$hostname.current.tasks";    # for shared NFS
#  RC 12/20/96 :: shortened "mod.control" prefix, unlike for TASKS & CONSTRAINTS
#  format of each line is one-space-separated items as follows:
#  module dept_ID pid version init_file path_prefix access_control_list


# BUG:?  Should we worry about distinguishing shared NFS for next TASK_FILES?

$STARTUP_TASK_FILE = "mod.control.startup.tasks";   # modules/tasks to startup
#  format of each line is space-separated items as follows:
#  $generic_task_name, $version, $dept_ID, $host_port, $init_file, $path_prefix

$CONSTRAINT_TASK_FILE = "mod.control.constraints.tasks";
                   # executable paths and misc constraints on modules/tasks
#  format of each line is space-separated items as follows:
#         $generic_task_name, $version, $executable_path, 
#         $default_init_file_name, $default_path_prefix

$PATH_PREFIX = $ENV{'GRIDSPATH'} ? "$ENV{'GRIDSPATH'}/log/" : '';
# Every instance of a data source module will need a separate filespace
# for its command, status-response, and log files.  This is a default prefix.

$MC_INIT_FILE = "module_controller.init";  # BUG?  ok/desirable if shared NFS?
# That file initializes our own control vars, especially host:port
# of our parent software manager, and of the aggregator to whom our
# data source modules should report.  Our init file will have same format
# as an ordinary module's command file, and we will write our updated
# status to it if/when we receive SET commands to alter our own control vars.

$MODULE_CONTROLLER_NAME  = "module_controller";    # name of our task itself


# Debugging toggles to tune verbosity level.
# if ($ENV{'mcdebug'}), these will be selectively set/toggled.
$debug{'busyloop'} = 0;
$debug{'raw_response'} = 0;
$debug{'get_response'} = 0;
$debug{'log'} = "mc.$hostname";  # local logfile
$debug{'tasks'} = 0;
$debug{'tcp_recv'} = 1;
$debug{'tcp_send'} = 1;
$debug{'special_kill'} = 0;
$debug{'make_init_file'} = 0;    # create an init file for ourself?
# Save vs. rm cmd & status files of dead kids?  Save vs. rm /tmp init_files?
$debug{'save_zombie_files'} = 0;
$debug{'acl'} = 0;               # test scaffolding for auth/access-control


# Various sleep times.  Should these be writable control vars?
# Current values are initial guesses; each may need tuning for performance.
$timer{'tcp_recv'} = 0.02;
$timer{'cmd_file'} = 0.02;
$timer{'kill_vrfy'} = 0.05;

# Force strict wait until child has enabled its sighandler:
$timer{'strict_task_wait'} = "strict_task_wait";
# Also required if we want process_special_start to return status info.
# This setting seems to make sense permanently; during integration with SW MGR,
# the setting is always:  $timer{'strict_task_wait'} = "strict_task_wait";


# ------------------
# Other Global Vars:
# ------------------
# Permanent globals:
# $_ is used for file reads.
# $errmsg holds/transfers last error msg.  Perhaps it could be made local.
# $req_host, $req_port ::  host & port of requestor who sent last command.
# $mc_vars :: control_vars used by MC itself

  $optimize_alive_to_true = 0;     # speeds up benchmark, but risky vs. robust.
#  $enable_acl;                     # disabled by default, unless set in ENV

  $PARENT_MANAGER = undef;  # null-string (vs. undef) will supersede
                            # value in module_controller.init file
  $PARENT_AGGREGATOR = undef; # null-string (vs. undef) will supersede 
                            # value in module_controller.init file
# ------------------
# Temporary globals for incremental development:
# (perhaps some of these should be permanent, as part of test scaffolding)
# $i
# $header, @body :: header & body of last command received.


         #######################  MAIN  ##########################

&self_init();
&start_tasks();

# Let potential test shells know we're ready to recv input:
system ("touch $CURRENT_TASK_FILE") unless -f $CURRENT_TASK_FILE;

if ($debug{'acl'}) {               # test scaffolding
  my $acl_file = $ENV{'grids_input'} . '.acl';
  # Consistent w/ naming scheme for test data files driven
  # by mc.test.pl -- so, eg, data file 'engine.input' (the GET/SET cmds)
  # should have corresponding acl file 'engine.acl', each line of which
  # should contain the alleged host:port source of corresponding cmd.
  unless ( open (ACL, "<$acl_file") ) {
    print STDERR "Unable to open ACL FILE [$acl_file]\n";
    $my_log->die ("Unable to open ACL FILE [$acl_file]\n");
    }
  @fake_auth_source = ();
  while (<ACL>) {
    next if /^\s*$|^\s*#/;      # ignore possible comments
    chomp;                      # each acl line should be HOST:PORT
    push (@fake_auth_source, $_);
    }
  close (ACL);
  }

&module_controller();
# &kill_all_modules ('*', '*', 'kill_sm');
$my_log->warn ("+++++++++ module_controller: == about to exit ...\n");

         #########################################################



#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%% PRIMARY ROUTINES %%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #


    ######################### module_controller ############################
sub module_controller {

  for (;;) {
    # BLOCKING select waiting for anybody?
    # Priority select on swm_parent for liveness/uptime protocol?
    # If so, handle that separately; else ...

    ($header, @body) = ();
    unless ( ($req_host, $req_port, $header, @body) =
             &Comm::tcp_recv ('blocking') ) {
      # If nobody wants to talk, then don't hog the CPU from child modules.
      select (undef, undef, undef, $timer{'tcp_recv'});  # do in non-block mode
      }
    $my_log->separator() if $debug{'tcp_recv'};
    $my_log->warn ("\n RECD ==> <$req_host, $req_port, $header, "
                   . join ('#', @body) . ">\n\n")
       if $debug{'tcp_recv'};

    if ($header !~ /G|S/i) {
      $errmsg = "ERROR- Wrong grids_msg_type [$header]!";
      $my_log->warn ($errmsg . " from $req_host:$req_port!");
      &Comm::tcp_send ($req_host, $req_port, 'BUG on your end.', $errmsg);
      $my_log->warn
         ("\n SENT ==> <$req_host, $req_port, BUG on your end. $errmsg>\n\n")
         if $debug{'tcp_send'};
      next;
      }

    $channel = "$req_host:$req_port";

    if ($debug{'acl'}) {
      ## then you better have enough lines in $acl_file!
      $debug{'acl'} = $channel;    # Remember the real source: our testdriver
      $channel = shift (@fake_auth_source);
      }

    &process_command ($channel, @body);
    }

  } ######################### module_controller ############################


    ############################### process_command ##########################
### route (or internally handle) the command we just received:
sub process_command {          # big, ugly
### this now takes a *list* as command
  my ($channel, @cmd) = @_;
     #list assignment is greedy, so scalars must precede any unlimited lists.
  # @cmd has form: (GET/SET, Module_Name, Version, Dept_ID,
  #                 var1, val1, var2, val2)
  # $channel is to check ACL -- string of form 'Host:Port'
  my ($errmsg);

  my $command = shift @cmd;
  my $module = shift @cmd;
  my $version = shift @cmd;
  my $dept = shift @cmd;
  my %cmd = @cmd;
  ### $command could be Sub-command ::  START, KILL, INVENTORY,
  my $header =  ($command =~ /^GET$|^INVENTORY$/i) ? 'gr' :
                ($command =~ /^SET$|^START$|^KILL$/i) ? 'sr' :
                                       'BUG' ;  # presumably debug response
  my ($response, @response);

  unless ($module && $version && $dept && $header =~ /^gr|sr$/ ) {
    $errmsg = "ERROR- Bad command syntax sent to module controller.";
    $my_log->warn ($errmsg . " [$command] [$module] [$version] [$dept]");
    &Comm::tcp_send ($req_host, $req_port, $header, $errmsg);
    $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, $errmsg>\n\n")
       if $debug{'tcp_send'};
    return;
    }

  ## If $command contains a SUB-command, we assume *completely* new syntax;
  ## so we convert it to old syntax, then fall thru to semi-original code.
  if ($command !~ /GET|SET/i) {   #BEGIN tmp kludge- convert to old syntax:
    $cmd{'mc_command'} = $command;
    $cmd{'mod_module'} = $module;
    $cmd{'mod_version'} = $version;
    $cmd{'mod_department'} = $dept;
    # $cmd{'mod_parent_aggregator'} = $cmd{'parent_aggregator'};  # Not needed.
    # $cmd{'mod_parent_manager'} = $cmd{'parent_manager'};   #Not needed.
    $module = $MODULE_CONTROLLER_NAME;
    if ($command =~ /INVENTORY/i) {
      $command = 'GET';
      }
    elsif ($command =~ /START|KILL/i) {
      $command = 'SET';
      }
    }  #END tmp kludge
  elsif ( $module ne $MODULE_CONTROLLER_NAME  &&
         ($module eq '*'  ||  $dept eq '*') ) {    # GET|SET on *multiple* kids
    my $func = 'multi_' . lc $command;        # only &multi_set() defined now.
    ###  multi cmd to GET | SET must check &authorized() on each acl.
    ($response, @response) = &{$func} ($header, $module, $version, $dept, %cmd);

    &Comm::tcp_send ($req_host, $req_port, $header,
                    ($response, $module, $version, $dept, @response));
    $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                 . "$response, $module, $version, $dept, "
                 . join ('#', @response) . ">\n\n")
       if $debug{'tcp_send'};
    return;
    }

  if ($module eq $MODULE_CONTROLLER_NAME  &&  exists $cmd{'mc_command'}) {
    my $func = $cmd{'mc_command'};
    ### Sub-command could be::  START, KILL, INVENTORY,
                          # ??  TASK_FILE_ADD, TASK_FILE_DELETE  ??
    $func =~ tr/A-Z/a-z/;
    $func = 'process_special_' . $func;    # build name for indirect funcall

    unless (defined &{$func}) {
      $errmsg = "ERROR- Bad SUB_command [" . $cmd{'mc_command'} . "].";
      $my_log->warn ($errmsg);
      &Comm::tcp_send ($req_host, $req_port, $header,
                      ($errmsg, $module, $version, $dept));
      $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                   . "$errmsg, $module, $version, $dept>\n\n")
         if $debug{'tcp_send'};
      return;
      }

    if ($cmd{'mc_command'} =~ /START/i  ||
        $cmd{'mc_command'} =~ /INVENTORY/i) {
      # only MC's own parent_manager is authorized
      return unless &authorized ($channel, $mc_vars->{'parent_manager'},
                                 $header, join ' ',
                                 $cmd{'mc_command'}, $cmd{'mod_module'},
                                 $cmd{'mod_version'}, $cmd{'mod_department'});
      }
    ###  since KILL is a multi cmd, it must check &authorized() on each acl.

    # pass SUB_dept, etc, parms to SUB_command:
    ($header, $response, $module, $version, $dept, @response) = 
      &{$func} ($command, $cmd{'mod_module'}, $cmd{'mod_version'},
                          $cmd{'mod_department'}, %cmd);

    &Comm::tcp_send ($req_host, $req_port, $header,
                    ($response, $module, $version, $dept, @response));
    $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                 . "$response, $module, $version, $dept, "
                 . join ('#', @response) . ">\n\n")
       if $debug{'tcp_send'};

    if ($func eq 'process_special_kill') {
      if (exists $cmd{'mc_shutdown'}  &&  $cmd{'mc_shutdown'} eq 'suicide') {
        &shutdown();
        }
      if ($module eq '*' && $version eq '*' && $dept eq '*') {
        $my_log->reopen_logfile ($mc_vars->{'department'}, $debug{'log'});
        # reopen Clog logfile, since that entire subdir has been renamed.
        }
      }

    return;
    }
  elsif ($module eq $MODULE_CONTROLLER_NAME) {  # && !exists $cmd{'mc_command'}
    # MC must SET | GET its *own* control_vars:
    # Is this special acl/auth bootstrap case?
    unless ( $cmd{'parent_manager'}  &&  !$mc_vars->{'parent_manager'} ) {
      return unless &authorized ($channel, $mc_vars->{'parent_manager'},
                                 $header, "$command $module $dept");
      }
    if ($command =~ /SET/i) {
      # CANNOT $mc_vars->do_set (\$cmd, \$mc_vars); since %cmd is not ref

      my $src_obj = { 'command', 'SET', 'module', $MODULE_CONTROLLER_NAME,
                      'version', $mc_vars->{'version'},
                      'department', $mc_vars->{'department'},
                      %cmd };

      &control_vars::do_set (\$src_obj, \$mc_vars);
      $mc_vars->update();
      # NEVER try to unlink our command file:   # $mc_vars->complete_command();

      @response = ();    # my var should be undef still
      foreach $key (keys %cmd) {
        push (@response, $key, $$mc_vars{$key});
        }
      }
    elsif ($command =~ /GET/i) {
    # my @result = %{$mc_vars};   # this would get __INFO__ tuple,
      my @result = ();

      foreach $key (keys %{$mc_vars}) {
        push (@result, $key, $$mc_vars{$key})
          unless $key eq '__INFO__';    # gory internals of CVs!
        }

      @response = &get_select (\%cmd, \@result);
      }

    &Comm::tcp_send ($req_host, $req_port, $header,
                     $mc_vars->{'command'},    #eg, OK or warning
                     $module, $version, $dept, @response);
    $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                 . "$mc_vars->{'command'}, $module, $version, $dept, "
                 . join ('#', @response) . ">\n\n")
       if $debug{'tcp_send'};

    return;
    }


# check ACL
# $channel  ===> 'Host:Port' of most recent command


  my ($child, $path_prefix, $acl) = &existing_task ($module, $version, $dept);
  return unless &authorized ($channel, $acl, $header,
                            "$command $module $version $dept");
  if ($path_prefix eq "/dev/null") {
    $errmsg = "ERROR- Cannot find child process for [$module].";
    $my_log->warn ($errmsg);
    &Comm::tcp_send ($req_host, $req_port, $header,
                    ($errmsg, $module, $version, $dept));
    $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                 . "$errmsg, $module, $version, $dept>\n\n")
       if $debug{'tcp_send'};
    return;
    }

  ($response, @response) = &get_or_set ('process_command', $command, $header,
                         $module, $dept, $child, $version, $path_prefix, %cmd);
 
  &Comm::tcp_send ($req_host, $req_port, $header,
                  ($response, $module, $version, $dept, @response));
  $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
               . "$response, $module, $version, $dept, "
               . join ('#', @response) . ">\n\n")
     if $debug{'tcp_send'};

  if ($debug{'get_response'} && $command =~ /GET/i) {
    print "\n============ \@RESPONSE portion from Do_response: ============\n";
    my @copy = @response;
    while (@copy) { 
      print shift @copy, " ===> ", shift @copy, "\n";
      }
    print "============ ================================== ============\n\n";
    }

  return;         # RC - should our caller do tcp_send instead?
  } ############################### process_command ##########################


    ############################### do_response ##########################
sub do_response {
  # Our caller must construct response packet and send it along $channel.
  my ($command, $module, $version, $dept, $path_prefix, %cmd) = @_;
  my ($response_file) = &construct_response_file_name ($path_prefix);
  my $response = '';
  my $header =  $command =~ /GET/i  ? 'gr' : 'sr' ;

  unless ( open (RESPONSE, "<$response_file") ) {
    $errmsg = "ERROR- Unable to open response file [$response_file].";
    $my_log->warn ($errmsg);
    return ($header, $errmsg, $module, $version, $dept);
    }

  ### &try_lock();  # Horrendous timing, plus _flock_ is busted on solaris.
  ### RC 12/20/96: This was deleted entirely.

  read (RESPONSE, $response, $MAX_READ);
  ### flock(RESPONSE,$LOCK_UN);   # we can be awakened by timer/alarm SIGnal
  close (RESPONSE);

  my @result = &gcpf_from_string($response);

  if ($debug{'raw_response'}) {
    print "+++++++++++ Do_response read this list from status file ++++++++\n";
    print join ('#', @result), "\n";
    print "+++++++++++ ++++++++++++++++++++++++++++++++++++++ +++++++++++++\n";
    }

  $response = shift @result;

  # unless ($module eq (shift @result)) {}
  # shift @result;   # ignore version field
  # unless ($dept eq (shift @result)) {}
  $module = shift @result;   # ignore module field ???
  $version = shift @result;   # ignore version field ???
  $dept = shift @result;   # ignore dept field ???

  if ($command =~ /SET/i) {
    unless ( $response =~ /^SET$|^OK/i ) {
      $errmsg =
         "OOPS- Module $module (dept $dept) responded *not* ok: [$response].";
      $my_log->warn ($errmsg);
      }
    return ($header, $response, $module, $version, $dept,
            &gcpf_select("listen_tcp", @result),
            &gcpf_select("listen_udp", @result));
    # does not return a trailing undef value if module declared no listen_tcp

    # bug?  Could return informative stuff in rest of @result?  Or too much?
    }
  # else we must respond to a GET ...
  # Look at %cmd and return values of desired $newvars.
  # What about errors in *individual* vars?  Will data source tag those?


  return  ( $header, $response, $module, $version, $dept,
            &get_select (\%cmd, \@result) );

  } ############################### do_response ##########################



    ############################### get_select ###########################
sub get_select {
  my %cmd = %{$_[0]};
  my @result = @{$_[1]};

  # Check for 2 easy cases of 'wildcard' aliases:

  my $all_scalars = 0;
  my $all_indices = 0;
  foreach $key (keys %cmd) {
    $cmd{$key} = 1;
    if ( $key =~ '\s*ALL_STATEVARS\s*{\s*ALL_STATEVARS\s*}\s*' ) {
      $all_indices = 1;
      # (we may want to change the semantics after we gain experience using it)
      return (@result);
      }
    }  # possible incoming VALS are irrelevant; loop below needs them _defined_


  my %index_table;
  if ($cmd{'ALL_STATEVARS'}) {
    $all_scalars = 1;
    delete $cmd{'ALL_STATEVARS'};
    }  # User may have requested all scalars but only some specific index vars.

  # bug?  always return timestamp tuple, even if not requested?
  my ($var, $val, @response);
  while ( @result  &&  (%cmd || $all_scalars) ) {
    $var = shift (@result);
    $val = shift (@result);
    if ( delete $cmd{$var} ) {
      # then this is one of the requested vars:
      push (@response, $var, $val);
      }
    elsif ( $all_scalars  &&  $var !~ /\{/ ) { # Scalar Var:
      push (@response, $var, $val);
      }                               # } (balance {}s)
    elsif ( $var =~ /\{/ ) {                   # Index Var:
      if ($all_indices) {   # generally requested:
        push (@response, $var, $val);
        }
      else {
        my ($varbase) =  $var =~ /^(\S+)\{/ ;   #get base of index var
                                     # }  (balance {}s)
        if ( $cmd{ $varbase . '{ALL_STATEVARS}' } ) { # specifically requested:
          push (@response, $var, $val);
          }
        elsif ( $all_scalars  ||  $cmd{ $varbase . '{}' } ) {
           #e.g., requesting "rulesets{}" means give those secondary indices
          # return an "index" of all our index vars:
          my ($val) =  $var =~ /\{(\S+)\}/ ;   #get base of index var
          if ($index_table{$varbase}) {
            $index_table{$varbase} .= ", \{$val\}";
            }
          else {
            $index_table{$varbase} = "\{$val\}"
            }
          }
        }
      }             # }  (balance {}s)
    }
  if (%index_table) {  # then append those constructed values to our response:
    while ( ($var, $val) = each %index_table ) {
      push (@response, $var, $val);
      }
    }

  return (@response);
  } ############################### get_select ###########################


#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%% TASK MANAGEMENT %%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #


    ############################### start_tasks ##########################
sub start_tasks {
  print "MC here about to start_tasks; my pid == $$\n" if $debug{'tasks'};
  unless ( open (STARTUP, "<$STARTUP_TASK_FILE") ) {
    $my_log->warn ("Unable to open STARTUP_TASK_FILE [$STARTUP_TASK_FILE]\n");
    return 0;
    }
  while (<STARTUP>) {
    next if /^\s*$|^\s*#/;      # ignore possible comments
    chop;
    &start_one_task ($_);
    }
  close (STARTUP);
  print "MC here AFTER start_tasks;\n" if $debug{'tasks'};
  } ############################### start_tasks ##########################


    ############################### start_one_task ##########################
### single parm incoming format::
### generic_task_name, version, dept_ID, aggregator_host:port,
### init_file_name, path_prefix
sub start_one_task {
  my ($pid, $child, $acl);
  my ($line, $init_set) = @_;   # optional 2nd parm from process_special_start
  my ($task, $version, $dept_ID, $host_port, $init_file, $path_prefix) =
     split (' ', $line);
  my (@status, @junk, $junk, $path_name);
  $init_file = &convert_null ($init_file);
  $path_prefix = &convert_null ($path_prefix);

# First check to see if this task is already "covered" by a pre-existing child.
  ($pid, $path_name, $acl) = &existing_task ($task, $version, $dept_ID);
  if ($pid) {
    ($junk, $errmsg, @junk[0 .. 2], @status) =
        &do_response ("SET", $task, $version, $dept_ID, $path_name,  {});

    $my_log->warn ("MC recovered task ===> <$task> "
                 . "for dept <$dept_ID> as PID <$pid> on listen_tcp <"
                 . (&gcpf_select ("listen_tcp", @status))[1] . ">\n")
      if $debug{'tasks'};

    my $comment = "OK- Not Started because existing task $pid "
            . "already covers command <$line>.";
    return ("$comment\n$errmsg", @status);
    }

  my ($executable_path, $default_init_file_name, $default_path_prefix) =
                                    &allowed_task ($task, $version);
  unless (defined $executable_path) {
    #  global var  $errmsg is set by allowed_task
    $my_log->warn ($errmsg);
    return $errmsg;
    }
  $default_path_prefix = &convert_null ($default_path_prefix);
  $default_init_file_name = &convert_null ($default_init_file_name);

  $init_file = $default_init_file_name
     unless defined ($init_file) && $init_file =~ /^\S+$/;;
  $init_file =~ s/\$ENV\{'GRIDSPATH'\}/$ENV{'GRIDSPATH'}/;
  $path_prefix = $default_path_prefix
     unless defined ($path_prefix) && $path_prefix =~ /^\S+$/;;
  $path_prefix = $PATH_PREFIX
     unless defined ($path_prefix) && $path_prefix =~ /^\S+$/;;
  $path_prefix =~ s/\$ENV\{'GRIDSPATH'\}/$ENV{'GRIDSPATH'}/;

  # We allow either $init_file or $path_prefix to be defined as empty strings.
  # Currently, values in CONSTRAINT file serve as defaults.
  # Instead, should they *override* corresponding values in TASK file?

  $path_prefix .= '/'  if -d $path_prefix
                       && '/' ne substr ($path_prefix, -1, 1);
       # Usually a reasonable assumption to /-append to a _dir_ name;
       # ... else it will become a _file_ prefix!
  $path_prefix .= '.' unless ( $path_prefix eq ''
                   ||  substr ($path_prefix, -1, 1) eq '/' );
                    # don't make it a visually-hidden file!

# { ######## BEGIN replace inefficient post_set
 # Note that, in contrast to genuine post_set, this approach of
 # constructing a one-time, custom init_file will *not* complain
 # about attempts to SET undefined control_vars.  This is correct
 # behavior, because at init time, we cannot know whether the module
 # will ever do an  add_straight() of the currently undefined varname.

  if (keys %$init_set) {     # build new init file to feed it:
    my @init = ();
    my %init = {};
    $acl = $$init_set{'parent_manager'} || $mc_vars->{'parent_manager'};
    ### RC 12/19/96::  CAUTION -- auth/acl/ACL BUG?  Are values of 0 or '' ok?

    if ( -f  $init_file  &&  open (INIT, "<$init_file") ) {
      # then load stuff from normal init file, then overlay that w/ init_set
      my $init_data = '';
      read (INIT, $init_data, $MAX_READ);
      close (INIT);
      @init = &gcpf_from_string($init_data);
      shift @init;  # status
      shift @init;  # module  - we assume these are acceptable values
      shift @init;  # version
      shift @init;  # dept
      %init = @init;
      }

    # now load (perhaps overlay) dynamic init_set init data
    foreach $key (keys %$init_set) {
      $init{$key} = $$init_set{$key};
      }

    $init_file = (split('/', $init_file)) [-1] if $init_file;  # get last substr
    $init_file = "/tmp/$hostname.$$.$init_file";  # make up ~new name

    # Note the above uses our *own* PID, not that of impending child.
    $my_log->warn ("Cannot build new init file <$init_file>")
      unless  open (INIT, ">$init_file");
    @init = ('command', 'SET', 'module', $task,
              'version', $version, 'department', $dept_ID, %init);
    print INIT &gcpf_to_string ('SET', $task, $version, $dept_ID, @init);
    close (INIT);
    }

  # else fall thru and feed it pre-existing init file as $init_file

# } ######## END replace inefficient post_set

my ($fork, $max_forks, $errno) = (0, 2);
for ($fork = 0; $fork < $max_forks; $fork++) {
                      # 8/12/96:  "Not enough space" failures on kanab only
  if (($child = fork()) == 0) {   # child process
    unless (defined $child) {
                            # New Perl5 doc sez failure ==> UNDEF (which == 0)
      $errno = $!;
      warn ("\n MC on $hostname: FORKING ERROR!!!   <UNDEF>\n <$errno>\n");
      $my_log->warn ("MC on $hostname: FORKING ERROR!!!   <UNDEF>\n <$errno>\n");
      sleep 2;
      next;
      }
    $ENV{'GRIDS_STARTER'} = $MODULE_CONTROLLER_NAME;
    # We must tell child its dept_ID, aggregator_host:port, init_file_name,
    # and also its path_prefix (for its log, cmd, and status/response files).
    $path_prefix .= &construct_file_name ($task, $dept_ID, $$);
#
#
#
# Running on non-UCDseclab environments ( perl binaries not in /pkg/bin ) may
# require a PERLLOC environment variable to point explicitly to the perl
# binary.  If so the following comment line must be manipulated. JRowe 7/22/97

    my $perl_cmd = "$ENV{PERLLOC}/perl -w $executable_path $path_prefix".
                   " $init_file $dept_ID $host_port";
#    my $perl_cmd = "$executable_path $path_prefix".
#                   " $init_file $dept_ID $host_port";

    $my_log->warn ("MC on $hostname will exec:\n" .
       "[$perl_cmd]\n");

    unless ( exec
        ( $perl_cmd ) ) {
      # fall thru if exec fails/returns ...
      $errmsg = "Exec failed on [$perl_cmd]!";
      $my_log->die ($errmsg);
      # can't have unemployed children running around loose !-)
      }

    }
  elsif ($child == -1) {  # New Perl5 doc sez failure ==> UNDEF, not -1
    $my_log->warn ("MC on $hostname: FORKING ERROR!!!   <-1>\n <$!>\n");
    sleep 2;
    next;
    }
  elsif ($child) {
    last;
    }
  }

  unless ($child) {
    $errmsg =
          "ERROR- MC on $hostname failed $max_forks attempts to FORK $task!!!\n <$errno>\n";
    $my_log->warn ($errmsg);
    return ($errmsg);
    }

  $path_prefix .= &construct_file_name ($task, $dept_ID, $child);

  if ( open (CURRENTS, ">>$CURRENT_TASK_FILE") ) {
    $init_file = '""' unless $init_file;       # need printable representation
    $path_prefix = '""' unless $path_prefix;   # need printable representation
    $acl = '""' unless $acl;                   # need printable representation
    ### RC 12/19/96::  CAUTION -- auth/acl/ACL BUG?  Are values of 0 or '' ok?
    print CURRENTS
          "$task $dept_ID $child $version $init_file $path_prefix $acl\n"; 
    close (CURRENTS);
    }
  else {
    $my_log->warn ("Cannot append to CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
    }

  if ($timer{'strict_task_wait'} eq "strict_task_wait") {
    for (;;) {  # don't dare signal it until child gets its act together:
      my $fname = &construct_response_file_name ($path_prefix);
      last if  -f $fname  &&  -s $fname;  # If zero size, let child LOCK it!
      select (undef, undef, undef, 0.1);
      # BUG!  need timeout here too!  Set $errmsg if no time to read status
      }
    if (keys %$init_set) {     # delete new custom init file
      unlink $init_file unless $debug{'save_zombie_files'};
      }
    ($junk, $errmsg, @junk[0 .. 2], @status) =
          &do_response ("SET", $task, $version, $dept_ID, $path_prefix, {});
    }

  if ($debug{'tasks'}) {
    $my_log->{'stderr'} = 1;
    $my_log->warn
       ("MC started task ===> <$task> for dept <$dept_ID> as PID <$child> on" .
        " listen_tcp <" . (&gcpf_select ("listen_tcp", @status))[1] . ">\n");
    $my_log->{'stderr'} = $clog_default_stderr;
    }

  return ($errmsg, @status);

  } ############################### start_one_task ##########################


    ############################### existing_task ##########################
### format of CURRENT_TASK_FILE is space-separated lines:
### $task_name $dept_ID $pid $version $init_file $path_prefix $access_control
sub existing_task {
  my ($task_name, $version, $dept_ID) = @_;
  my $search_pat = join (' ', $task_name, $dept_ID);
  my ($pid, $path_prefix, $match_pid, $match_prefix,
                                      $acl, $match_acl, @deletes);
  $errmsg = '';
  $errmsg = "Warning: no CURRENT_TASK_FILE [$CURRENT_TASK_FILE]."
    unless  -f $CURRENT_TASK_FILE;
    # Above will occur on initial startup in a new or clean subdir.
  $errmsg  ||  open (CURRENTS, "<$CURRENT_TASK_FILE")  ||
     ($errmsg = "Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
  if ($errmsg) {
    $my_log->warn ($errmsg);
    return (0, "/dev/null");
    }

  while (<CURRENTS>) {
    if ( ($pid) = /^$search_pat (\d+) / ) {      # should be <= *1* match
      if (&alive ($pid)) {
        ($path_prefix, $acl) = /^$search_pat $pid \S+ \S+ (\S+) (\S+)/;
        $match_pid = $pid unless $match_pid;
        $match_prefix = $path_prefix unless $match_prefix;
        $match_acl = $acl unless $match_acl;
        # Note we return *first* live entry matching task & dept.
        # It should be impossible for a SET cmd to change a dept to one that
        # is already covered by a live child, hence masking that one's entry!
        # However, in future botched upgrades, there could be confusion
        # re ones that differ by a version #, so I'm leaving this code in.
        }
      else {
        push (@deletes, $_);
        }
      }
    }
  close (CURRENTS);
  foreach $entry (@deletes) {
    my $init_file;
    ($task_name, $dept_ID, $pid, $version, $init_file, $path_prefix) =
        $entry =~ /^(\S+) (\S+) (\d+) (\S+) (\S+) (\S+)/;
    $my_log->warn
              ("\nWill delete $pid\'s zombie entry in CURRENT_TASK_FILE.");
    &delete_task_entry ($pid);
    unless ($debug{'save_zombie_files'}) {
      my @files = ( &construct_cmd_file_name ($path_prefix),
                    &construct_response_file_name ($path_prefix) );
      unlink @files;
      }
    }
  return ($match_pid, $match_prefix, $match_acl) if $match_pid;
  return (0, "/dev/null");         # our caller will complain
  } ############################### existing_task ##########################


    ############################ delete_task_entry #########################
### delete task from CURRENT_TASK_FILE
sub delete_task_entry {
  my @lines = ();
  my $pid = shift;
  unless ( open (CURRENTS, "<$CURRENT_TASK_FILE") ) {
    $my_log->warn ("Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
    return;
    }
  while (<CURRENTS>) {
    if ( /^\S+ \S+ $pid / ) {
      next;
      }
    push (@lines, $_);
    }
  close (CURRENTS);
  unless ( open (CURRENTS, ">$CURRENT_TASK_FILE") ) {
    $my_log->warn ("Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
    return;
    }
  while (@lines) {
    print CURRENTS shift (@lines);
    }
  close (CURRENTS);
  } ############################ delete_task_entry #########################


    ##################### replace_current_task_entry ######################
sub replace_current_task_entry {
  my ($child, $field, $replacement) = @_;
  return unless $field =~ /acl|dept/;   # don't anticipate any others yet.
  my @lines = ();

  unless ( open (CURRENTS, "<$CURRENT_TASK_FILE") ) {
    $my_log->warn ("Can't change $field in TASKFILE [$CURRENT_TASK_FILE]\n");
    #  return silently?  pretty hostile!  ACL BUG?
    return;
    }
  while (<CURRENTS>) {
    if ( /^(\S+) (\S+) $child (\S+) (\S+) (\S+) (\S+)$/ ) {
      $_ = "$1 $2 $child $3 $4 $5 $replacement\n" if $field eq 'acl';
      $_ = "$1 $replacement $child $3 $4 $5 $6\n" if $field eq 'dept';
      }
    push (@lines, $_);
    }

  close (CURRENTS);
  unless ( open (CURRENTS, ">$CURRENT_TASK_FILE") ) {
    $my_log->warn ("Can't change $field in TASKFILE [$CURRENT_TASK_FILE]\n");
    return;
    }
  while (@lines) {
    print CURRENTS shift (@lines);
    }
  close (CURRENTS);
  } ##################### replace_current_task_entry ######################


    ############################### allowed_task ##########################
### single parm incoming format::
### generic_task_name, version, dept_ID, report_to_host:port, init_file_name
###    May communicate to its caller via global var  $errmsg.
sub allowed_task {
  my ($task_name, $version, $executable_path,
      $default_init_file_name, $default_path_prefix);
  my ($search_task, $search_version, @junk) = @_;
  $errmsg = '';
  $errmsg = "Error: no CONSTRAINT_TASK_FILE [$CONSTRAINT_TASK_FILE]."
    unless  -f $CONSTRAINT_TASK_FILE;
  $errmsg  ||  open (CONSTRAINTS, "<$CONSTRAINT_TASK_FILE")  ||
    ($errmsg = "Unable to open CONSTRAINT_TASK_FILE [$CONSTRAINT_TASK_FILE]\n");
  if ($errmsg) {
    return();
    }

  while (<CONSTRAINTS>) {
    next if /^\s*$|^\s*#/;      # ignore possible comments
    chop;
    ($task_name, $version, $executable_path,
     $default_init_file_name, $default_path_prefix) = split (' ', $_);
    next unless ($search_task eq $task_name  &&  $search_version eq $version);
    $default_path_prefix = &convert_null ($default_path_prefix);

    if (1 == $executable_path =~ s/\$ENV\{'GRIDSPATH'\}/$ENV{'GRIDSPATH'}/) {
      unless (-f $executable_path) {
        $errmsg = "ERROR- Flawed constraint file entry or \$ENV{'GRIDSPATH'} "
                . "for [$task_name, $version] ==> [$executable_path]";
        return ();  # our caller will complain
        }
      }
    return ($executable_path, $default_init_file_name, $default_path_prefix);
    }
  $errmsg = "ERROR- Task [$search_task], "
          . "version [$search_version] is not allowed.";
  return ();  # our caller will complain
  } ############################### allowed_task ##########################


    ############################# authorized #############################
sub authorized {
  my ($channel, $acl, $header, $cmd) = @_;
  # $channel should be 'HOST:PORT'
  # $acl should be 'HOST:PORT,HOST:PORT,HOST:PORT'  ( >= 1 tuple )
  # To standardize, HOST should be fully qualified.
  return 1 unless $enable_acl;
  return 1 if $acl =~ /$channel/;
  $errmsg = "ERROR- Unauthorized access by [$channel] ==> [$cmd].";
  $my_log->warn ($errmsg);

  my ($req_host, $req_port) = split ':', $channel;
  ## If fake_auth_source during testing, then tell *real* testdriver!
  ($req_host, $req_port) = split ':', $debug{'acl'}   if $debug{'acl'};

  &Comm::tcp_send ($req_host, $req_port, $header, $errmsg);
  $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
               . "$errmsg>\n\n")
     if $debug{'tcp_send'};
  return 0;
  } ############################# authorized #############################

#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%% MULTI-SET|GET COMMANDs %%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #


    ############################# match_tasks ############################
sub match_tasks {          # does *NOT* currently match on version
  my ($module, $version, $department) = @_;
  my @tasks = ();
  my ($mod, $dept, $pid, $vers, $prefix, $acl);
  $errmsg = '';
  $errmsg = "Warning: no CURRENT_TASK_FILE [$CURRENT_TASK_FILE]."
    unless  -f $CURRENT_TASK_FILE;
  $errmsg  ||  open (CURRENTS, "<$CURRENT_TASK_FILE")  ||
     ($errmsg = "Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
  if ($errmsg) {
    $my_log->warn ($errmsg);
    return ();
    }
  while (<CURRENTS>) {
    next unless ($mod, $dept, $pid, $vers, $prefix, $acl) = 
                 /^(\S+) (\S+) (\d+) (\S+) \S+ (\S+) (\S+)/ ;  # maybe garbled
    if ( ($module eq '*' || $module eq $mod)
      && ($department eq '*' || $department eq $dept) ) {
      push (@tasks, "$mod $dept $pid $vers $prefix $acl");
      }
    }
  close (CURRENTS);
  return @tasks;
  } ############################# match_tasks ############################


    ############################## multi_set #############################
sub multi_set {
  my ($header, $module, $version, $dept, %cmd) = @_;
  my @tasks = &match_tasks ($module, $version, $dept);
  return 'WARNING- NO MATCHING TASK' unless @tasks;
  my @tasks_done;         # for rollback of bulk transaction if ERROR
  my ($response, @response, %response) = ('');
  my ($result, @result) = ('');
  my $error = '';
  my ($response_file, $cmd_file);

  foreach $entry (@tasks) {
    my ($mod, $d, $pid, $v, $prefix) = split (' ', $entry);
    next unless &alive ($pid);
    next if $mod eq 'sm';
    next if $mod eq 'engine';   # SW mgr handles these separately

    $response_file = &construct_response_file_name ($prefix);
    system ("mv $response_file $prefix.rollback");
    push (@tasks_done, $entry);
    # Note: By push-ing here, any rollback will include the item causing error.
    # At the least, we need to do an *inverse* "mv" on that item.

    ($result, @result) = &get_or_set ('multi_set', 'SET', $header,
                                      $mod, $d, $pid, $v, $prefix, %cmd);
    if ($result =~ /^OK/) {
      my $pound_parity = ($#result == -1) ? '' : '#' ;
         # for automated vsuite testing by &comp() in mc.test.pl, not here.
      push (@response, "($mod, $d, $v) ==> (#$pound_parity"
                       . join ('#', @result) . ")\n");
      if ($result =~ /warning/i) {      # Warning has precedence, so prepend
        $response = "$result\n" . $response;
        }
      else {
        $response .= "$result\n";
        }
      }
    else {
      $error = "$result\n$entry";
      $error = $result;         # Although the above *is* more informative,
                                # its format is too varied for mc.test.pl
      $response = $error;
      last;
      }
    }

  return 'WARNING- NO MATCHING TASK' unless @tasks_done;
  # 2/1/97:  since @tasks might contain only sm or engine, which we ignore.

  $my_log->warn ("multi_set transaction ERROR:\n$error\n"
               . "Will ROLLBACK transaction") if $error;
  foreach $entry (@tasks_done) {
    my ($mod, $d, $pid, $v, $prefix) = split (' ', $entry);
    if ($error) {   # ROLLBACK transaction after ERROR
      # must make command in file a SET
      $cmd_file = &construct_cmd_file_name ($prefix);
      $errmsg = '';

      open (ROLLBACK, "<$prefix.rollback")  ||
         ($errmsg = "Unable to open ROLLBACK [$prefix.rollback]\n");
      $errmsg  ||  open (CMD, ">$cmd_file")  ||
         ($errmsg = "Unable to open CMD File [$cmd_file]\n");
      if ($errmsg) {
        $my_log->warn ($errmsg);
        $error .= "\nCannot ROLLBACK Set on ($mod, $d, $pid)";
        next;
        }
      $_ = <ROLLBACK>;   # throw away old cmd
      print CMD "SET\xff\n";
      foreach (1..3) {
        # cross module, version, and dept values, which might eq 'command'
        $_ = <ROLLBACK>;
        print CMD $_;
        }
      my $parity = 0;
      while (<ROLLBACK>) {
        print CMD $_;
        if (/^command\xff\n$/  &&  ++$parity % 2) {
          # it's our Field, not some Value
          $_ = <ROLLBACK>;   # throw away old cmd value
          print CMD "SET\xff\n";
          }
        }
      close (ROLLBACK);
      close (CMD);
      kill 'USR1', $pid;
      # Don't wait around to see the results.
      }
    unlink "$prefix.rollback";   # whether or not $error
    }

  return ($response, @response);
  } ############################## multi_set #############################


    ############################# get_or_set #############################
sub get_or_set {
  my ($caller, $command, $header, $module, $dept,
      $child, $version, $path_prefix, %cmd) = @_;

  if ( exists $cmd{'department'}  &&  $cmd{'department'} ne $dept
                                  &&  $command =~ /SET/i ) {
    # Changing a child's dept means extra bookkeeping for us:
    if ( (&existing_task ($module, $version, $cmd{'department'})) [0] ) {
      $errmsg = "ERROR- change_dept would duplicate unique assignment: "
                      . "<$module, $dept, $cmd{'department'}>!";
      return $errmsg;
      }
    else {
      &replace_current_task_entry ($child, 'dept', $cmd{'department'});
      $dept = $cmd{'department'};    # BUG?  will someone notice & complain?
      }
    }

  if (exists $cmd{'parent_manager'}  &&  $command =~ /SET/i) {
    # Changing a child's acl/authorized parent_manager means extra bookkeeping:
### 	$my_log->warn ("   Calling replace_current_task_entry...");
	&replace_current_task_entry ($child, 'acl', $cmd{'parent_manager'})
	    unless $cmd{'parent_manager'} eq "";
    }

  my $cmd_file = &construct_cmd_file_name ($path_prefix);

  if ( -f  $cmd_file  ||  ! open (COMMAND, ">$cmd_file") ) {
    # if file exists, then module failed to complete previous cmd.
    $errmsg = "WARNING- Cannot overwrite or open command file [$cmd_file].";
    $my_log->warn ($errmsg);
    if ($caller eq 'process_command') {
      &Comm::tcp_send ($req_host, $req_port, $header,
                      ($errmsg, $module, $version, $dept));
      $my_log->warn ("\n SENT ==> <$req_host, $req_port, $header, "
                   . "$errmsg, $module, $version, $dept>\n\n")
         if $debug{'tcp_send'};
      }
    return "Would overwrite CMD file for pid $child";
    }

  print COMMAND &gcpf_to_string ($command, $module, $version, $dept, %cmd);
  close (COMMAND);
  my $num = kill 'USR1', $child;
  $my_log->warn ("===========> $num kids signalled! <=============\n")
           unless $num == 1;
  my $busyloop;
  for ($busyloop = 0; -f "$cmd_file"; $busyloop++) {
    select (undef, undef, undef, $timer{'cmd_file'});
    # BUG!  That probably will disable ALARM timer.
    # Hence must maintain *own* count of timer increments (0.02 or whatever)?
    # Better just to get initial timestamp, then sample the diff
    # each time we wakeup, since the select() timer is a MINIMUM sleep!
    print "Waiting for module to delete command file [$cmd_file].",
          " [\$busyloop == $busyloop] ++++ \n" if
      $debug{'busyloop'} && ! ($busyloop % 100);
    }
  print "Kid deleted command file.  ++++ [\$busyloop == $busyloop] ++++ \n"
    if $debug{'busyloop'};

  ($header, $response, $module, $version, $dept, @response) = 
    &do_response ($command, $module, $version, $dept, $path_prefix, %cmd);

  return ($response, @response);
  } ############################# get_or_set #############################




    ############################## multi_get #############################
sub multi_get {
  my ($header, $module, $version, $dept, %cmd) = @_;
  my @tasks = &match_tasks ($module, $version, $dept);
  my ($response, @response, %response, $header) = ('');
  my ($result, @result) = ('');
  my $error = '';
  #  This does ASYNC GET, hence error means matching tasks fail to have a
  #  *unique* value for one of the requested fields.

  foreach $entry (@tasks) {
    my ($mod, $d, $pid, $v, $prefix) = split (' ', $entry);
    next unless &alive ($pid);

    ($header, $result, $mod, $v, $d, @result) =
      &do_response ('GET', $mod, $v, $d, $prefix, %cmd);

    if ($result =~ /^OK/) {
      $response{$entry} = @result;
      $response .= "$result\n";

      # compare each @result to first list loaded into @response (or load this)

      # Merge results.  ERROR if any field intersection has non-unique value.
      # Fields to return may already be determined.  But what if ALL_STATEVARS?

      }
    else {
      $error .= "$result\n$entry";
      $response .= "$result\n$entry";
      }
    }

  return ($response, @response);
  } ############################## multi_get #############################



#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%% SPECIAL SUB-COMMANDS %%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #


    ######################### process_special_kill #########################
### KILL a child.  This is *not* same as a "polite shutdown".
### Called indirectly via::  &{$func} ();
sub process_special_kill {
    my ($command, $module, $version, $dept, %cmd) = @_;
    unless ($command eq "SET") {
      $errmsg = "ERROR- Special KILL SUB_command must be a SET.";
      $my_log->warn ($errmsg);
      return ("gr", $errmsg, $module, $version, $dept);
      }

  # check ACL

  $module = '*' unless $module;
  $dept = '*' unless $dept;
  if ($module eq '*' || $dept eq '*') {
    my $kill_sm =  exists $cmd{'kill_sm'}  &&  $cmd{'kill_sm'} eq 'kill_sm';
    my $deaths = &kill_all_modules ($dept, $module, $kill_sm);
    $errmsg = "OK- KILLED <$deaths> <$module> modules for department <$dept>";
    $errmsg = "WARNING- KILLED <zero> <$module> modules for department <$dept>"
       unless $deaths;
    $my_log->warn ($errmsg);
    return ("sr", $errmsg, $module, $version, $dept);
    }

  my ($child, $path_prefix) = &existing_task ($module, $version, $dept);
  if ($child) {
    kill 'SIGKILL', $child;
    select (undef, undef, undef, $timer{'kill_vrfy'});

    unless ( $child eq waitpid ($child, $WNOHANG) ) {
      $errmsg = "ERROR- Task [$module] (PID $child) refuses to be KILLed:\n";
      $my_log->warn ($errmsg);
      return ("sr", $errmsg, $module, $version, $dept);
      }
    $my_log->warn ("\n MC KILLed task [$module] (PID $child) as ordered.\n\n");
    &delete_task_entry ($child);
    return ("sr", "OK- KILLed as ordered.", $module, $version, $dept);
    }
  else {
    $errmsg = "WARNING- Cannot find child process to KILL for [$module].";
    $my_log->warn ($errmsg);
    return ("sr", $errmsg, $module, $version, $dept);
    }
  } ######################### process_special_kill #########################


    ######################### process_special_start #########################
### START a child dynamically (vs. from our own TASK init file).
### Called indirectly via::  &{$func} ();
sub process_special_start {
    my ($command, $module, $version, $dept, %cmd) = @_;
    my ($errmsg, @status);
    unless ($command eq "SET") {
      $errmsg = "ERROR- Special START SUB_command must be a SET.";
      $my_log->warn ($errmsg);
      return ("gr", $errmsg, $module, $version, $dept);
      }

  # check ACL

  my $init_file_name = delete $cmd{'mod_init_file_name'}  ||  "''";
  my $path_prefix = delete $cmd{'mod_path_prefix'}  ||  "''";
  # Apply 2 init values from MC's own control vars:
                       # $cmd{'mod_parent_aggregator'} # never implemented by UI
  my $host_port = delete $cmd{'parent_aggregator'};
  $host_port = $mc_vars->{'parent_aggregator'} unless defined $host_port;
  $cmd{'parent_manager'} = $mc_vars->{'parent_manager'}
     unless defined $cmd{'parent_manager'};
  # values of 0 and '' are considered meaningful
  ### RC 12/19/96::  CAUTION -- auth/acl/ACL BUG?

  # pass %cmd to %init_set in &start_one_task:
  delete $cmd{'mc_command'};
  delete $cmd{'mod_module'};
  delete $cmd{'mod_version'};
  delete $cmd{'mod_department'};
  ($errmsg, @status) = &start_one_task
     ("$module $version $dept $host_port $init_file_name $path_prefix", \%cmd);

  $errmsg = 'OK- Started.' if $errmsg eq 'OK';
  if ($errmsg =~ /^WARNING- (Attempted SET on UNDEFINED control var: .*)$/) {
    $errmsg = 'WARNING- Started OK, but ' . $1;
    }
  return ("sr", $errmsg, $module, $version, $dept, @status);

  } ######################### process_special_start #########################


    ####################### process_special_inventory #####################
### GET an INVENTORY of living tasks.
### Called indirectly via::  &{$func} ();
sub process_special_inventory {
    my ($command, $module, $version, $dept, %cmd) = @_;
    $module = '*' unless $module;
    $dept = '*' unless $dept;
    $version = '*' unless $version;
    unless ($command eq "GET") {
      $errmsg = "ERROR- Special INVENTORY SUB_command must be a GET.";
      $my_log->warn ($errmsg);
      return ("sr", $errmsg, $module, $version, $dept);
      }

  # check ACL

  $errmsg = '';
  $errmsg = "Warning: no CURRENT_TASK_FILE [$CURRENT_TASK_FILE]."
    unless  -f $CURRENT_TASK_FILE;
    # Above will occur on initial startup in a new or clean subdir.
  $errmsg  ||  open (CURRENTS, "<$CURRENT_TASK_FILE")  ||
     ($errmsg = "Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
  if ($errmsg) {
    $my_log->warn ($errmsg);
    return ("gr", $errmsg, $module, $version, $dept, () );
    }

  my ($task_name, $dept_ID, $pid, $ver, $init_file, $path_prefix, $acl);
  my (@result, @deletes);
  my $num = 1;
  while (<CURRENTS>) {
    ($task_name, $dept_ID, $pid, $ver, $init_file, $path_prefix, $acl) =
      /^(\S+) (\S+) (\d+) (\S+) (\S+) (\S+) (\S+)/;
    unless (&alive ($pid)) {
      push (@deletes, $_);
      next;
      }
    push (@result, "\nINVENTORY: ${\$num++}. ",
                   # fieldname eases automated testsuite pdiff comparison
                   join (' ', $task_name, $dept_ID, $pid, $ver, $acl))
      if ($module eq '*' || $module eq $task_name)
      && ($dept eq '*' || $dept eq $dept_ID);
    }
  close (CURRENTS);
  foreach $entry (@deletes) {
    ($task_name, $dept_ID, $pid, $ver, $init_file, $path_prefix) =
        $entry =~ /^(\S+) (\S+) (\d+) (\S+) (\S+) (\S+)/;
    &delete_task_entry ($pid);
    unless ($debug{'save_zombie_files'}) {
      my @files = ( &construct_cmd_file_name ($path_prefix),
                    &construct_response_file_name ($path_prefix) );
      unlink @files;
      }
    }

  return ("gr", "OK", $module, $version, $dept, @result);

  } ####################### process_special_inventory #####################






#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%% INITIALIZATION %%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #


    ############################### self_init ##########################
### Initialization routine for module controller:
sub self_init {
  $| = 1;       # unbuffer STDOUT
  &debug_verbosity();

  # $SIG{'SIGCHLD'} = 'IGNORE';
  # else our kids become *zombie* procs upon death, and
  # the MC must do an expensive `ps` test (vs. kill 0) to avoid
  # hanging while trying to communicate with a dead child.
  # Well, it doesn't work!  The unix manpages lie again!

  $MC_RECV_PORT = $ENV{'MC_RECV_PORT'}  ||  $MC_RECV_PORT;

  my ($status, $tcp_port) = &Comm::init ($MC_RECV_PORT, undef, $MAX_CLIENT);
  unless ($status eq 'ok'  &&  $tcp_port == $MC_RECV_PORT) {
    ($status, $tcp_port) = &Comm::init (0, undef, $MAX_CLIENT);
    }                           # parm <0> means we'll accept any freeport

  my $cv_prefix = "mc.$hostname";                 # for shared NFS

  $mc_vars = control_vars->new ($cv_prefix, $MC_INIT_FILE || '/dev/null');
  # The above initializes our control vars, but we still must declare them:
  $mc_vars->add_straight ('listen_tcp');
  # add_straight automatically adds   'parent_manager', 'parent_aggregator'
  # MC doesn't use parent_aggregator, but it *does* pass it on to children.  

  $enable_acl = defined($ENV{'enable_acl'}) && $ENV{'enable_acl'} eq 'enable_acl';
  $PARENT_MANAGER = $ENV{'parent_manager'}
         if defined $ENV{'parent_manager'};    # ok to setenv parent_manager ''
  $PARENT_AGGREGATOR = $ENV{'parent_aggregator'}
            if defined $ENV{'parent_aggregator'};
  # ENV vars override values in init_file.
  $mc_vars->{'parent_manager'} = $PARENT_MANAGER
                      if defined $PARENT_MANAGER;
  $mc_vars->{'parent_aggregator'} = $PARENT_AGGREGATOR
                         if defined $PARENT_AGGREGATOR;

  if ($debug{'make_init_file'}) {  # create an init file for ourself:
    # load 'standard' control vars:
    $mc_vars->{'version'} = 'RCS 1.29';
    $mc_vars->{'module'} = $MODULE_CONTROLLER_NAME;
    $mc_vars->{'department'} = $hostname;
    $mc_vars->{'command'} = 'SET';   # so we can re-init from it.

    $PARENT_MANAGER = "krakatoa.cs.ucdavis.edu:4444";  # for debug only
    $PARENT_AGGREGATOR = "AGGREGATOR_HOST:4321";       # for debug only
    $mc_vars->{'parent_manager'} = $PARENT_MANAGER;
    $mc_vars->{'parent_aggregator'} = $PARENT_AGGREGATOR;
    $mc_vars->{'listen_tcp'} = $tcp_port  ||  $MC_RECV_PORT;

    $mc_vars->update();

    # NEVER try to unlink our command file:     # $mc_vars->complete_command();
    # That's actually our init file / status file !
    # If our control vars get changed, and if we want those changes
    # to be mirrored in the init file, then after update we must explicitly:
    system ("cp ${\&construct_response_file_name ($cv_prefix)} $MC_INIT_FILE");
    }
  else {    # testing shells must be able to learn our TCP port:
    $mc_vars->{'listen_tcp'} = $tcp_port  ||  $MC_RECV_PORT;
    $mc_vars->update();
    }

  $my_log = new Clog ('MC',   # abbreviate $MODULE_CONTROLLER_NAME
                      $mc_vars->{'department'}  ||  $hostname,
                      $debug{'log'}
		      );
  $my_log->{'prefix'} = " (PID$$) ";
  $clog_default_stderr = 0;        # should clog print to stderr?

  $my_log->warn ("Module Controller: No INIT file [$MC_INIT_FILE]\n")
     unless $debug{'make_init_file'}  ||  -f $MC_INIT_FILE;

  $my_log->die ("Module Controller: Cannot bind *any* TCP port!\n")
    unless $status eq 'ok';
  unless ($tcp_port == $MC_RECV_PORT) {
    $my_log->warn ("Module Controller: preferred TCP port [$MC_RECV_PORT]" .
                   "unavailable, so I'm using port [$tcp_port].\n");
    $MC_RECV_PORT = $tcp_port;
    }

  print ("MODULE CONTROLLER LISTENING ON PORT $MC_RECV_PORT\n");

  } ############################### self_init ##########################



sub debug_verbosity {
  return unless $ENV{'mcdebug'} && $ENV{'mcdebug'} !~ /^tcp2mc$/;
  if ($ENV{'mcdebug'} =~ /[^#]default.1/) {
    # Debugging toggles to tune verbosity level.
    $debug{'busyloop'} = 1;
    $debug{'raw_response'} = 1;
    $debug{'get_response'} = 1;
    $debug{'log'} = "mc.$hostname";    # local logfile
    $debug{'tasks'} = 1;
    $debug{'tcp_recv'} = 1;
    $debug{'tcp_send'} = 1;
    $debug{'special_kill'} = 1;
    $debug{'make_init_file'} = 0;    # create an init file for ourself?
    }
  # now selectively toggle defaults, or initialize individual flags
  my @flags = split(/\s+/, $ENV{'mcdebug'});
  foreach $key (@flags) {
    $debug{$key} = ! $debug{$key}  if defined $debug{$key};
    }   # selectively override defaults with simple env-toggle
  }



sub shutdown {
  $my_log->die ("MC obeying command to die");
  }



#
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%% UTILITIES %%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #
#   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   #

sub construct_file_name {
  my ($module, $dept, $pid) = @_;
# return "$module.$hostname.$dept";  # for shared NFS; must be unique
  # The above is indeed unique, but since depts can change,
  # we cannot depend on that for a *stable* notion of identity!
  return "$module.$hostname.$pid";   # for shared NFS; must be unique
  } ########################################################################

sub construct_response_file_name {
  return  $_[0] . ".status";
  } ########################################################################

sub construct_cmd_file_name {
  return  $_[0] . ".cmd";
  } ########################################################################


### Convert printed representation of empty string (in a file) to actual empty.
sub convert_null {
  return undef unless defined ($_[0]);
  return '' if $_[0] eq "\"\"";
  return '' if $_[0] eq "\'\'";
  return $_[0];
  }


    ########################## gcpf_to_string ##########################
### Convert body of GET/SET from list to GCPF string, for writing to a file:
sub gcpf_to_string {
  return join ("\xff\n", @_) . "\xff\n";
  # undef list items in @_ may generate complaints re  "uninitialized value".
  } ########################## gcpf_to_string ##########################

    ########################## gcpf_from_string ##########################
### Convert body of GET/SET from GCPF string (eg, from a file) to list:
sub gcpf_from_string {
  return split ("\xff\n", $_[0]);
  } ########################## gcpf_from_string ##########################


    ########################## gcpf_select ##########################
    # searches a list and returns an adjacent varname/value pair as list.
    # first parm is search pattern; remaining parms are a gcpf list.
sub gcpf_select {
  for ($i = 1; $i < $#_; $i += 2) {
    $_[$i] eq $_[0]  &&  return ($_[0], $_[++$i]);
    }
  return ();   # an empty list is *not* same as a list containing undef(s)!
  } ########################## gcpf_select ##########################



    ############################### alive ##########################
### If a child appears in a ps listing, is or is it not a defunct zombie?
sub alive {
    my $pid = shift;
    return $pid if $optimize_alive_to_true;
 
    # "kill 0" alone cannot differentiate *zombie* procs from alive ones!
    if (kill 0, $pid) {  # it's either alive or a zombie
      return $pid if $pid ne waitpid ($pid, $WNOHANG);
      # Caveat:  above test fails unless it's *our* zombie,
      # vs. the child of another live MC.  But if that were case,
      # you'd have bigger problems to worry about ;-)
      }
    else {
      return 0;
      }
 
    # below is old, *slow* (200 millisec) method.
    my $ps = `$ps_comm$pid`;
    $ps = ( split ("\n", $ps) )[1]  ||  '';   # discard column headings
    return $pid if ( $ps =~ /\s*$pid\s+/  &&
                     (($uname eq "sunos" && $ps !~ /defunct/)  ||
                      ($uname eq "solaris" &&  $ps !~ /\s+Z\s+/)) );
    return 0;
  } ############################### alive ##########################



    ########################## kill_all_modules ##########################
### This is a clean-up utility for use during development.
sub kill_all_modules {
  #  $_[0] is dept of modules to kill, or '*' for all.
  my ($dept, $module, $kill_sm) = @_;
  my ($pid, $task_name, $dept_ID, @saves);
  $errmsg = '';
  $errmsg = "Warning: no CURRENT_TASK_FILE [$CURRENT_TASK_FILE]."
    unless  -f $CURRENT_TASK_FILE;
  $errmsg  ||  open (CURRENTS, "<$CURRENT_TASK_FILE")  ||
     ($errmsg = "Unable to open CURRENT_TASK_FILE [$CURRENT_TASK_FILE]\n");
  if ($errmsg) {
    $my_log->warn ($errmsg);
    return 0;
    }

### Note no acl protection yet.  Not clear if we want that,
### since this may be a "doomsday" func, called after
### properly-authorized SW mgrs have hung themselves.

  my $deaths = 0;
  while (<CURRENTS>) {
    next unless ($task_name, $dept_ID, $pid) =
                 /^(\S+) (\S+) (\d+) / ;     # maybe garbled file
    if ( ($module eq '*' || $module eq $task_name)
      && ($dept eq '*' || $dept eq $dept_ID)
      && ($task_name ne 'sm'  || $kill_sm) ) {
      $deaths += kill ('SIGKILL', $pid);
      }
    else {
      push (@saves, $_);
      }
    }
  close (CURRENTS);
  open (CURRENTS, ">$CURRENT_TASK_FILE");   # rewrite any survivors
  while (@saves) {
    print CURRENTS shift @saves;
    }
  close (CURRENTS);
  return $deaths;
  } ########################## kill_all_modules ##########################


          #########################################################
          #########################################################

__DATA__

Data source packets use the general GrIDS Common Packet Format (GCPF)
described in chapter~\ref{comm}.  The GCPF header is either {\em g}
for a GET request, {\em s} for a SET request, or
{\em gr} and {\em sr} for RESPONSES to GET and SET requests.
The format for the body of a command or response packet will contain
various fields, $separated$ by a 255 character.


The $body$ of a GET, SET, or RESPONSE packet will contain the following fields:

\begin{itemize}
\item  SET {\tt |} GET {\tt |} GET-asynch {\tt |} OK {\tt |}
ERROR-$text$ {\tt |} WARNING-$text$
\item  Data Source Name
\item  Version \# (minimum acceptable version; assume upward-compatible)
(RESPONSE packet may insert $actual$ Version \#)
\item  Department ID represented by that module.
\item  StateVarName
\item  Value
\item  StateVarName
\item  Value
\item  StateVarName
\item  Value
\end{itemize}

For GET requests, the $Value$ fields may be empty,
in which case there may be adjacent 255 characters.


If a packet is a RESPONSE to a GET command, then $all$ the requested
StateVarNames and corresponding Values will be returned, unless
there was an error outcome.  If this is a RESPONSE to a SET command,
then $none$ of the original StateVarNames and corresponding Values
will be returned, unless an error occurred.  In that case, the
response packet may include only those StateVar/Value
pairs that help explain the error.

Note that an ERROR-$text$ value (pertaining to the entire RESPONSE)
may appear in the initial field, in addition to other
ERROR-$text$ fields which may appear as Values of StateVars.
Possible $text$ for ERROR fields include:

\begin{itemize}
\item  Failure,
\item  Timeout,
\item  No such data source,
\item  No data source here representing specified Department,
\item  Bad version number,
\item  Set failed (invalid value),
\item  Set failed (read-only variable),
\item  Set/Get failed (no such StateVarName).
\item  Set/Get failed (unauthorized access).
\end{itemize}

Among the set of $required$ StateVars, we have reserved one special
StateVar named $ALL\_STATEVARS$.

(The initial implementation may ignore this feature.)

A command to GET $ALL\_STATEVARS$ for some module should
return a RESPONSE containing the StateVarNames and Values
of all its {\bf scalar} StateVars; for its {\bf indexed}
StateVars, only the $indices$ themselves will be returned
within a single Value field, enclosed in squiggly brackets.

A command to GET $Ruleset$ {$ALL\_STATEVARS$}
for some module should return all Rulesets.

A command to GET $ALL\_STATEVARS$ {$ALL\_STATEVARS$}
for some module should generate an exhaustive enumeration
of both scalar and indexed StateVars.


The Module Controller will have a known set of StateVars.
A request, e.g., to GET $INVENTORY$, would cause the
Module Controller to read its $Task\ File$ and return
a list of every GrIDS module currently running on its host.


\subsection{Launching Recurring Modules}
The Module Controller will have a known StateVar,
e.g., $CHRON\_JOBS$, which will be indexed by the
frequency (in minutes) at which to launch a task,
and the particular task to launch.

This will $not$ be implemented initially!




