# $Id: Hierarchy_interface.pm,v 1.55 1998/02/17 19:22:24 rowe Exp $

# This module is the library included by an interface that would like to
# interact with the hierarchy.  This is a class, and it is possible for
# a process to have multiple objects from the class at one time.  (This
# facilitates integration testing with interleaved transactions from 
# different interfaces.

# Fields in this object

# user		The user-id of the user who created this object
# password	The password of the user who created this object
# dept		The root department of this particular hierarchy subtree
# view_serial	The serial number of the view we have here.
# hierarchy	A Hierarchy object which represents our current understanding
#		of the hierarchy we are viewing and managing.
		
#		SPECIAL NOTE

# Users of this class can see the hierarchy data structure directly.  This
# is necessary so that, for example, it can be rendered on the screen etc.
# However, the Hierarchy object MUST NOT BE altered directly - this will
# lead to disaster.  It must only be changed by using the methods supplied
# in the current class.

#============================================================================#

use strict;

use Hierarchy;
use Clog;
package Hierarchy_interface;
use Comm;
use Grdbm;
use Message_center;

#============================================================================#

# Obtain the database of things we need for GrIDS to run.
$Hierarchy_interface::grids_db = new Grdbm(5,3);

# These are only mainly defined for backward compatability with test
# scripts.  Eventually, we should get shut of this.
$Hierarchy_interface::ohs_location 
	= $Hierarchy_interface::grids_db->{'ohs_host'}.':'.
		$Hierarchy_interface::grids_db->{'ohs_port'};
$Hierarchy_interface::my_location = "jaya.cs.ucdavis.edu:3000";

#============================================================================#

# Constants used by this class.

# Models the set of variables which the user CAN NOT set via a set call.
%Hierarchy_interface::dept_set_not_allowed = (
                        'department',1,
                        'parent',1,
                        'host_child',1,
                        'dept_child',1,
                        'our_aggregator',1,
                        'parent_aggregator',1,
                        'parent_manager',1,
			'aggregator_mc',1,
              );           
 
%Hierarchy_interface::host_set_not_allowed = (
			'rulesets',1,
              );           
 

$global::use_ACL = 0; #This is not needed for the Interface.


#============================================================================#
#
# Function: new
# 
# This function initializes a new Hierarchy_interface object, including
# going to the organizational hierarchy server and getting a copy of the
# appropriate part of the hierarchy.
# 
# Inputs:
#  Actual Arguments:
#   user, password, department, log, output_mode, handle
#  Global:
#   None
# 
# Return Value(s):
#  Hierarchy_interface object
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi = new Hierarchy_interface($user,$password,$dept,$log,
#                                               $output_mode,$handle);
# 
#-------------------------------------------------------------------------#
 
sub new
{
  shift;
  my($user,$password,$dept,$log,$output_mode,$handle,$no_hier) = @_;
  my($self);
 
  bless $self =
   {
    'user'              => $user,
    'password'          => $password,
    'dept'              => $dept,
    'log'               => $log,
    'output_mode'       => $output_mode,
    'handle'            => $handle,
    'view_serial'       => undef,
    'hierarchy'         => undef,
    'ohs_location'      => $Hierarchy_interface::ohs_location,
    'my_location'       => $Hierarchy_interface::my_location,
    'ohs_open'          => 0,
   };
    
  my($host,$port) = split(/:/,$Hierarchy_interface::my_location);
     
  # Initialize Comm.pm if necessary.
  if($self->{'output_mode'} eq 'comm' && !$Comm::INIT)
   {
    my ($status) = Comm::init(0,0);
    unless ($status eq 'ok')
     {
      die "Couldn't get Comm.pm init in Hierarchy_interface.pm\n";
     }
   }
  unless (defined ($no_hier))# May not want to talk to OHS to start Hierarchy (polling)
   {    
    unless($self->refresh())
     {
      $log->warn("Couldn't talk to OHS successfully\n");
      return (0,"Couldn't talk to OHS successfully");
     }
    $log->warn("Successfully obtained hierarchy");
   }
  return (1,$self);
} # end new

#============================================================================#
#
# Function: refresh
# 
# This function goes to the organizational hierarchy server and gets a fresh,
# updated copy of the hierarchy.
# 
# Return Value(s):
#  False (1) if the hv transaction is ok.
#  In the future might return 0 on failure.  Presently dies altogether
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi =  refresh();
#
#----------------------------------------------------------------------------#
 
sub refresh
{
  my($self) = @_;
  my($log) = $self->{'log'};
  my($junk1,$junk2,$header,$body_ref) = (0,0,0,0);
 
  my $messageref = [$self->{'user'}, $self->{'password'}, $self->{'dept'}];
  $self->send_to_OHS('hvr',$messageref);
  my($hostname,$port) = split(/:/,$self->{'ohs_location'});
  my($status,$reply_ref) = receive_messages([[$hostname,$port]],$log);
  die "Could not talk to OHS at all!!" unless $status;
  ($junk1,$junk2,$header,$body_ref) = @{${$reply_ref}[0]};
  #die "Got $header from OHS - cannot continue" unless $header eq 'hv';
  unless ($header eq 'hv')
   {
    print STDERR "Got $header from OHS - malfunctions may occur\n";
    return 0;
   }
  $self->{'view_serial'} = ${$body_ref}[1];
# Modified by J.Rowe to pass the current view serial to the new Hierarchy.
  $self->{'hierarchy'} = new Hierarchy($self->{'log'},\${$body_ref}[2],
                             ${$body_ref}[1]);
 
  Hierarchy::use_ACL($global::use_ACL);# We do not need to use the ACL in the
                                       # interface.  The OHS and SM's handle it.
 
  return 1;
 
} # end refresh
 
#============================================================================#
# 
# Function: new_root
# 
# This creates a new root department with the name 'Root_name'.  The
# software manager and aggregator for the new root are to be run on
# 'Manager_Host' and 'Aggregator_Host' respectively.  Two ports must be 
# specified: the port for the root department software manager (which must 
# already be running) and the port for the module controller of the
# of the aggregator (which we will start).
# Note:  See technical document for a full description of message.
# 
# Return Value(s):
#  hierarchy
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi->new_root($root_name,$man_host,$man_port,$agg_host,$agg_mc_port);
# 
#----------------------------------------------------------------------------#

sub new_root
{
  my($self,$root_name,$man_host,$man_port,$agg_host,$agg_mc_port) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($agg_port,$err,$replies);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("New root: ".join(' ',@_[1..$#_]));

  # Get a root aggregator up.
  ($agg_port,$err) = $self->start_root_aggregator($root_name,
						$agg_host,$agg_mc_port);
  return (0,$err) unless $agg_port;

  # Check if sensible and contact OHS.
  my $htr_ref = ['new_root',$trans_id, $self->{'user'},$self->{'password'},
    $root_name, $man_host,$man_port, $agg_host,$agg_port];
  my($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::new_root_is_sane);
  return ($status,$err) unless $status;

  # We have permission from the OHS.  Now talk to the software managers.
  # We will set its department name to be the $root_name to be supplied to us
  # parent will be null - this is the root department.
  # We will tell it where to find its aggregator.

  my($sdv_ref) = [$root_name,'department',$root_name,
		'parent','null',
		'our_aggregator',	"$agg_host:$agg_port",
		'parent_aggregator',	0,
		'aggregator_mc',	"$agg_host:$agg_mc_port",
		];
  my $sm_messages = [[$man_host,$man_port,'sdv',$sdv_ref]];
  ($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  $self->{'dept'} = $root_name;
  my $htc_ref = ['new_root',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
					$htc_ref,\&Hierarchy::new_root);
} #end new_root

#============================================================================#
#
# Function: change_user
#
# This adds a new user to the ACL.  It asks the OHS first, then tells the SM
# of the department where the user is being added.  It then reports back
# to the OHS so it can update its ACL.
#
# It also remove a user, by passing a null '' password for an existing user.
#
# Return Value(s):
#  hierarchy
#
# Global variable modifications:
#  None
#
# Example of usage:
#  $hi->add_host($Department,$Host,$Port);
#
#----------------------------------------------------------------------------#
 
sub change_user
{
  my($self,$user,$pass,$dept) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};
 
  $log->separator();
  $log->warn("change user: ".join(' ',@_[1..$#_]));
 
  # Check if this is at all sensible
  my($htr_ref) =
    ['change_user',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'},$user, $pass, $dept];
  my($status,$err) = $self->initiate_transaction($htr_ref,
                                \&Hierarchy::change_user_is_sane);
  return ($status,$err) unless $status;

  # We have permission from the OHS.  Now talk to the software manager.
  # Tell it about the new host.
  my($sdv_ref) = [$dept,"local_ACL\{$user\}",$pass];
  my($dept_object) = $self->{'hierarchy'}{$dept};
  my $sm_messages = [[$dept_object->{'manager_host'},
                        $dept_object->{'manager_port'},'sdv',$sdv_ref]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['change_user',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
                                        $htc_ref,\&Hierarchy::change_user);

} # end change_user
 
#============================================================================#
# 
# Function: add_host
# 
# This adds a new host with name 'Host' under an existing department named
# 'Department'.  The 'Port' on which its module controller is running is
# also necessary.
# Note:  See technical document for a full description of message.
# 
# Return Value(s):
#  hierarchy
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi->add_host($Department,$Host,$Port);
# 
#----------------------------------------------------------------------------#

sub add_host
{
  my($self,$dept,$host,$port) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Add host: ".join(' ',@_[1..$#_]));
  
  # Check if this is at all sensible
  my($htr_ref) = 
    ['add_host',$trans_id,$self->{'user'},$self->{'password'},
    	$self->{'view_serial'}, $dept, $host, $port];
  my($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::add_host_is_sane);
  return ($status,$err) unless $status;
     
  # We have permission from the OHS.  Now talk to the software manager.
  # Tell it about the new host.
  my($sdv_ref) = [$dept,"host_child\{$host\}",$port]; 
  my($dept_object) = $self->{'hierarchy'}{$dept};
  my $sm_messages = [[$dept_object->{'manager_host'},
			$dept_object->{'manager_port'},'sdv',$sdv_ref]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['add_host',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
					$htc_ref,\&Hierarchy::add_host);

} # end add_host

#============================================================================#
#
# Function: start_manager
#
# This is an internal function only, that is used to start a new SM.
#
# Return Value(s):
#  The manager port, error messages
#
# Global variable modifications:
#  None
#
# Example of usage:
#  ($man_port,$err,$replies) = 
#     $self->start_manager($parent,$dept_name,$man_host,$agg_host,$agg_port);
#
#----------------------------------------------------------------------------#

sub start_manager
{
  my($self,$parent,$dept_name,$man_host,$agg_host,$agg_port) = @_;
  my($log) = $self->{'log'};
  
  my($parent_dept_object) = $self->{'hierarchy'}{$parent};
  my($man_port);
 
  # Marshall the information we need out of the hierarchy.
  my($man_host_obj) = $self->{'hierarchy'}{$man_host}; # software manager
  my($host_dept_obj) = $self->{'hierarchy'}{$man_host_obj->{'parent'}};
  my($parent_agg_port) = $parent_dept_object->{'aggregator_port'};
  my($parent_agg_host) = $parent_dept_object->{'aggregator_host'};
  my($parent_man_host) = $parent_dept_object->{'manager_host'};
  my($parent_man_port) = $parent_dept_object->{'manager_port'};
  my($agg_host_obj) = $self->{'hierarchy'}{$agg_host};
  my $agg_mc_port  = $agg_host_obj->{'port'};
#
# The following was removed by J.Rowe to solve logical inconsistencies
# in GrIDS messaging when multiple departments are handled by the same
# SM. Now use a seperate SM process for each dept. 5-JUN-97
# RC 6/11/97:  consensus == revert to 1 sm process per dept for now.
#
#  # Check the hierarchy to see if a SM is running on the host.
#  my (@dept_array,$temp_dept);
#  push (@dept_array, $self->{'hierarchy'}{'root_name'});
#  SEARCH_SM: while ($temp_dept = pop(@dept_array))
#   {
#    push (@dept_array, @{$self->{'hierarchy'}{$temp_dept}{'dept_children'}});
#    if ($self->{'hierarchy'}{$temp_dept}{'manager_host'} eq $man_host)
#     {
#      # A SM is already running on that host, so start a new internal SM.
#      $man_port = $self->{'hierarchy'}{$temp_dept}{'manager_port'};
# 
#      my $new_ref = [$dept_name,
#                'parent',               $parent,
#                'our_aggregator',       "$agg_host:$agg_port",
#                'parent_aggregator',    "$parent_agg_host:$parent_agg_port",
#                'parent_manager',       "$parent_man_host:$parent_man_port",
#                'aggregator_mc',        "$agg_host:$agg_mc_port",
#                 ];
# 
#      my $sm_messages = [[$man_host, $man_port,'new',$new_ref]];
#      my ($err,$replies) = $self->communicate_with_sms($sm_messages);
# 
#      return($man_port,$err,$replies);
#     }# end if
#   } # end while
# 
  # No SM was running on the host, so we must start up a new SM.
  my $shv_ref = [$host_dept_obj->{'name'},$man_host,
		'module_controller','version',
                $host_dept_obj->{'name'},'mc_command','START',
                'mod_module',           'sm',
                'mod_department',       $dept_name,
                'mod_version',          'v.v',
                'parent',               $parent,
                'our_aggregator',       "$agg_host:$agg_port",
                'parent_aggregator',    "$parent_agg_host:$parent_agg_port",
                'parent_manager',       "$parent_man_host:$parent_man_port",
                'aggregator_mc',        "$agg_host:$agg_mc_port",
		"inherited_ACL\{$self->{'user'}\}",$self->{'password'},
#'inherited_ACL{first}',$self->{'password'},
		'init_user',		$self->{'user'},
		'init_pass',		$self->{'password'},
                ];
 
  my $sm_messages = [[$host_dept_obj->{'manager_host'},
                        $host_dept_obj->{'manager_port'},'shv',$shv_ref]];
  my ($err,$replies) = $self->communicate_with_sms($sm_messages);
  unless($err)
   {
    my %hash = @{$replies->[0][3]}; # extract body of reply
    $man_port = $hash{'listen_tcp'};
    unless($man_port)
     {
      $err = "No manager port in add_dept- all bets are off";
      $log->warn($err);
      return($err,undef);
      return($man_port,$err,undef);
     }
   }

  return ($man_port,$err,$replies);
}# end start_manager
 
#============================================================================#
# 
# Function: add_dept
# 
# This creates a new department with name 'Department' under the existing
# 'Parent' department.  The software manager and aggregator for the new
# department are to be run on 'Manager_Host' and 'Aggregator_Host'
# respectively.
# 
# Note:  See technical document for a full description of message.
# 
# Return Value(s):
#  hierarchy
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi->add_dept($Parent,$Department,$Manager_Host,$Aggregator_Host);
# 
#----------------------------------------------------------------------------#

sub add_dept
{
  my($self,$parent,$dept_name,$man_host,$agg_host) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Add dept: ".join(' ',@_[1..$#_]));

  # Check if this is at all sensible
  my($htr_ref) = 
    ['add_dept',$trans_id,$self->{'user'},$self->{'password'},
    	$self->{'view_serial'}, $parent, $dept_name, $man_host, $agg_host];
  my($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::add_dept_is_sane);
  return ($status,$err) unless $status;
   
  # We have permission from the OHS.  Now talk to the software managers.
  
  # Contact the software manager for the department of the host where the
  # new software manager is to run, and have it start the new software
  # manager.
  my($man_port,$agg_port);

  # Marshall the information we need out of the hierarchy.
  my($parent_dept_object) = $self->{'hierarchy'}{$parent};
  my($man_host_obj) = $self->{'hierarchy'}{$man_host}; # software manager

  my($host_dept_obj) = $self->{'hierarchy'}{$man_host_obj->{'parent'}}; 
### RC BUGfix NOTE:  The above should be named $man_host_dept_obj, ie:
# my($man_host_dept_obj) = $self->{'hierarchy'}{$man_host_obj->{'parent'}}; 
###                  But in this *add_dept* routine, we never need that!
###                  Instead, what we need is  $agg_host_dept_obj  below ...

  my($parent_agg_host) = $parent_dept_object->{'aggregator_host'};
  my($parent_agg_port) = $parent_dept_object->{'aggregator_port'};
  my($parent_man_host) = $parent_dept_object->{'manager_host'};
  my($parent_man_port) = $parent_dept_object->{'manager_port'};
  my($agg_host_obj) = $self->{'hierarchy'}{$agg_host};
  my $agg_mc_port  = $agg_host_obj->{'port'};

### RC BUGfix NOTE:  *This* routine needs new var below, *not* $host_dept_obj!
  my($agg_host_dept_obj) = $self->{'hierarchy'}{$agg_host_obj->{'parent'}}; 
###                  6/28/97:  I've substituted it in 3 places below.


  # Startup the new aggregator.
  my($shv_ref) = [$agg_host_dept_obj->{'name'},$agg_host,'module_controller','version',
		$dept_name,'mc_command','START',
		'mod_module',		'engine', 
		'mod_department',	$dept_name,
		'mod_version',		'v.v',
		'parent_aggregator',	"$parent_agg_host:$parent_agg_port",
		'parent_manager',	"$parent_man_host:$parent_man_port",
		];
  my $sm_messages = [[$agg_host_dept_obj->{'manager_host'},
			$agg_host_dept_obj->{'manager_port'},'shv',$shv_ref]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);
# 
# Modified by J.Rowe (6-may-97); If an error occurs while starting the
# engine, don't continue, but send a warning and end this transaction
# and return an error.
#  unless($err)
#   {
#    my %hash = @{$replies->[0][3]}; # extract body of reply
#    $agg_port = $hash{'listen_tcp'};
#    unless($agg_port)
#     {
#      $err = "No aggregator port in add_dept- all bets are off";
#      $log->warn($err);
#      goto END_add_dept_;
#     }
#   }
  if ($err)
   {
     $log->warn("After trying to start aggregator the sm said: $err");
     goto END_add_dept;
   }
  else
   {
    my %hash = @{$replies->[0][3]}; # extract body of reply
    $agg_port = $hash{'listen_tcp'};
    unless($agg_port)
     {
      $err = "No aggregator port in add_dept- all bets are off";
      $log->warn($err);
      goto END_add_dept_;
     }
   }

  sleep 5;
  ($man_port, $err,$replies) = 
         $self->start_manager($parent,$dept_name,$man_host,$agg_host,$agg_port);
  unless($err)
   {
    sleep 5;
    # Tell the parent about its new child.
    my $sdv_ref = [$parent,"dept_child\{$dept_name\}",
               "$man_host $man_port"];
    $sm_messages = [[$parent_dept_object->{'manager_host'},
                     $parent_dept_object->{'manager_port'},'sdv',$sdv_ref]];
    ($err,$replies) = $self->communicate_with_sms($sm_messages);
   }
 
  # Complete the transaction with the OHS and update ourselves.
  END_add_dept:

  my $htc_ref = ['add_dept',$trans_id,$man_port,$agg_port];
  return $self->complete_transaction($err,$htr_ref,
					$htc_ref,\&Hierarchy::add_dept);
}# end add_dept

#============================================================================#
#
# Function: move_dept
#
# This moves a department within the hierarchy.  It first checks with the OHS
# on the sanity of the move.  Then it contacts the appropriate software
# managers and updates them.  Afterwhich it informs the OHS that it has
# completed the job and updates its own hierarchy.
#
# Arguments:
#  Department moving to, Deptartment to move
#
# Return Value(s):
#  hierarchy
#
# Global variable modifications:
#  None
#
# Example of usage:
#  $hi->move_dept($Parent,$Department);
#
#----------------------------------------------------------------------------#

sub move_dept
{
  my($self,$new_parent,$dept) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Move dept: ".join(' ',@_[1..$#_]));
 
  # Check if this is at all sensible
  my($htr_ref) =
    ['move_dept',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $new_parent, $dept];
  my($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::move_dept_is_sane);
  return ($status,$err) unless $status;
 
  # We have permission from the OHS.  Now talk to the software managers.
 
  # Remove $dept as a child of $old_parent.
  my $old_parent =  $self->{'hierarchy'}{$dept}{'parent'};
  my($dept_object) = $self->{'hierarchy'}{$old_parent};
  my($sdv_ref) = [$old_parent,"dept_child\{$dept\}",''];
  my $sm_messages = [[$dept_object->{'manager_host'},
                           $dept_object->{'manager_port'},'sdv',$sdv_ref]];
  my ($err,$replies) = $self->communicate_with_sms($sm_messages);

# Modified by J.Rowe (12-may-97): Tell the new parent about it's
# child only after child has been updated to avoid inconsistencies.
#
#  # Tell the new parent about this new child.
#  my($man_host) =  $self->{'hierarchy'}{$dept}{'manager_host'};
#  my($man_port) =  $self->{'hierarchy'}{$dept}{'manager_port'};
  my($new_parent_obj) = $self->{'hierarchy'}{$new_parent};
#  my($sdv_ref) = [$new_parent,"dept_child\{$dept\}","$man_host $man_port"];
# 
#  $sm_messages = [[$new_parent_obj->{'manager_host'},
#                       $new_parent_obj->{'manager_port'},'sdv',$sdv_ref]];
#  my ($err,$replies) = $self->communicate_with_sms($sm_messages);
# 
  # Change parent information in moved department
  my($new_agg_host) =  $new_parent_obj->{'aggregator_host'};
  my($new_agg_port) =  $new_parent_obj->{'aggregator_port'};
  my($new_man_host) =  $new_parent_obj->{'manager_host'};
  my($new_man_port) =  $new_parent_obj->{'manager_port'};
  my($new_parent_agg) = ($new_agg_host.':'.$new_agg_port);
  my($new_parent_man) = ($new_man_host.':'.$new_man_port);
 
  my($dept_object) = $self->{'hierarchy'}{$dept};
  my($sdv_ref) = [$dept,'parent',$new_parent,'parent_aggregator',$new_parent_agg,
                  'parent_manager',$new_parent_man];
 
  $sm_messages = [[$dept_object->{'manager_host'},
                               $dept_object->{'manager_port'},'sdv',$sdv_ref]];
  my ($err,$replies) = $self->communicate_with_sms($sm_messages);


# Placed here by J.Rowe after updating the child above 12-may-97
# Tell the new parent about this new child.
  my($man_host) =  $self->{'hierarchy'}{$dept}{'manager_host'};
  my($man_port) =  $self->{'hierarchy'}{$dept}{'manager_port'};
  my($sdv_ref) = [$new_parent,"dept_child\{$dept\}","$man_host $man_port"];
 
  $sm_messages = [[$new_parent_obj->{'manager_host'},
                       $new_parent_obj->{'manager_port'},'sdv',$sdv_ref]];
  my ($err,$replies) = $self->communicate_with_sms($sm_messages);
 
  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['move_dept',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
					$htc_ref,\&Hierarchy::move_dept);
}# end move_dept
 
#============================================================================#
#
# Function: move_host
#
# This moves a host within the hierarchy.  It first checks with the OHS
# on the sanity of the move.  Then it contacts the appropriate software
# managers and updates them.  Afterwhich it informs the OHS that it has
# completed the job and updates its own hierarchy.
#
# Arguments:
#  Department moving to, Host to move
#
# Return Value(s):
#  hierarchy
#
# Global variable modifications:
#  None
#
# Example of usage:
#  $hi->move_host($Deptment, $host);
#
#----------------------------------------------------------------------------#

sub move_host
{
  my($self,$dept,$host) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};
  my($status,$err,$replies);

  $log->separator();
  $log->warn("Move host: ".join(' ',@_[1..$#_]));

  # Check if this is at all sensible and get permission from the OHS
  my($htr_ref) =
    ['move_host',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $dept, $host];
  ($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::move_host_is_sane);
  return ($status,$err) unless $status;
  
  # Talk to the software managers.
  my $old_parent	= $self->{'hierarchy'}{$host}{'parent'};
  my $old_dept_object	= $self->{'hierarchy'}{$old_parent};
  my $port		= $self->{'hierarchy'}{$host}{'port'};
  my $new_dept_object	= $self->{'hierarchy'}{$dept};
  my $old_sdv_ref 	= [$old_parent,"host_child\{$host\}",''];
  # NOTE WELL!! Order dependency in this next set message!!! See 
  # host_child_change in sm for details.
  my $new_sdv_ref1 = [$dept,'old_dept',$old_parent,];
  my $new_sdv_ref2 = [$dept,"host_child\{$host\}",$port,];
  my $sm_messages = [
			[$old_dept_object->{'manager_host'},
				$old_dept_object->{'manager_port'},
				'sdv',$old_sdv_ref,
			],
			[$new_dept_object->{'manager_host'},
				$new_dept_object->{'manager_port'},
				'sdv',$new_sdv_ref1,
			],
			[$new_dept_object->{'manager_host'},
				$new_dept_object->{'manager_port'},
				'sdv',$new_sdv_ref2,
			],
		    ];
  ($err,$replies) = $self->communicate_with_sms($sm_messages);
 
   # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['move_host',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
					$htc_ref,\&Hierarchy::move_host);
}# end move_host 

#============================================================================#
#
# Function: move_manager
#
# This function is to move the physical location of a "Department's" software
# manager to another 'Host' without changing the actual structure of the
# hierarchy.
# It does this by first starting a new SM on the new host, and getting all of
# the current control variables on the old SM.  Then it kills the old SM,
# tells the parent and children of the department where the new SM is at and
# initialize the new SM with the values of the control variables from the
# old sm.
#
# Return Value(s):
#  None
#
# Global variable modifications:
#  None
#
# Example of usage:
#  $hi->move_manager($Department,$Host);
#
#-------------------------------------------------------------------------#
     
sub move_manager
{
  my($self,$dept,$man_host) = @_;
  my($log) = $self->{'log'};
  my($trans_id) = int(rand(10000000));
  my($return);

  $log->separator();
  $log->warn("Move manager: ".join(' ',@_[1..$#_]));

  # Check if this is at all sensible                 
  my($htr_ref) =
    ['move_manager',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $dept, $man_host];
  my($status,$err) = $self->initiate_transaction($htr_ref,
                                \&Hierarchy::move_manager_is_sane);
  return ($status,$err) unless $status;

  # We have permission from the OHS.  Now talk to the software managers.

  # Marshall the information we need out of the hierarchy.
  my($parent) = $self->{'hierarchy'}{$dept}{'parent'};
  my($agg_host) = $self->{'hierarchy'}{$dept}{'aggregator_host'};
  my($agg_port) = $self->{'hierarchy'}{$dept}{'aggregator_port'};
  
  # Start the new SM on the new host.
  my($man_port, $err,$replies) =
      $self->start_manager($parent,$dept,$man_host,$agg_host,$agg_port);

# Added by J.Rowe (20-May-97). Later, when the old variables are
# dumped into the new manager, the SM will check whether the sdv request
# for inherited variables is coming from the proper parent host
# address. It finds the interface host instead and bails out with
# permission denied. Remove the inherited variables from the list
# before sending to the new manager.  The values will be inherited properly
# when the parent update is done.
#
  # Get all of the current control variables from the old SM.
#  my $new_ref = [$dept,@{$self->get($dept,0,0,'ALL_STATE_VARS','_')}];

  my $new_ref_hash = {@{$self->get($dept,0,0,'ALL_STATE_VARS','_')}};
  my $variable;
  foreach $variable ( grep(/inherited/,(keys %$new_ref_hash)) ) {
    delete( $new_ref_hash->{$variable} );
  }
  my $new_ref_array = [ $dept, %$new_ref_hash ];
#
  # The following commands can be done in parallel.

  # Kill the SM on the old host.
  my $sdv_ref = [$dept,"local_managers\{$dept\}",'M'];
  my($old_host) = $self->{'hierarchy'}{$dept}{'manager_host'};
  my($old_port) = $self->{'hierarchy'}{$dept}{'manager_port'};
  my @sm_messages;
  push (@sm_messages, [$old_host, $old_port,'sdv',$sdv_ref]);


  #  Tell the parent of the Dept where the new sm is
  my $parent        = $self->{'hierarchy'}{$dept}{'parent'};
  my $parent_dept_object   = $self->{'hierarchy'}{$parent};
 
  if ( defined $parent_dept_object ) {
    $sdv_ref = [$parent,"dept_child\{$dept\}",
               "$man_host $man_port"];
 
    push(@sm_messages, [$parent_dept_object->{'manager_host'},
                   $parent_dept_object->{'manager_port'},'sdv',$sdv_ref]);
    }
  else
  {
#
#   Added to handle the local_ACL variables when the dept has no parents
#   from which to inherit them. J.Rowe 6-JUN-1997
     push(@$new_ref_array, "local_ACL\{$self->{'user'}\} $self->{'password'}");
     }

  #  Tell the children of the Dept where the new sm is
  my $child;
  foreach $child (@{$self->{'hierarchy'}{$dept}{'dept_children'}})
   {
    my $child_dept_object  = $self->{'hierarchy'}{$child};
 
    $sdv_ref = [$child,"parent_manager", "$man_host:$man_port"];
 
    push (@sm_messages, [$child_dept_object->{'manager_host'},
                   $child_dept_object->{'manager_port'},'sdv',$sdv_ref]);
   }

  # Send all of the messages at once.
  ($err,$replies) = $self->communicate_with_sms(\@sm_messages);
 
  # Assign control variables in the new sm to those of the old sm.
    @sm_messages = [$man_host,$man_port,'sdv',$new_ref_array];
    ($err,$replies) = $self->communicate_with_sms(\@sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['move_manager',$trans_id,$man_port];
  return $self->complete_transaction($err,$htr_ref,
                                       $htc_ref,\&Hierarchy::move_manager);

}# end move_manager

#=========================================================================#
#
# Function: move_aggregator
#
# This function is to move the physical location of a "Department's" software
# manager to another 'Host' without changing the actual structure of the
# hierarchy.
# It does this by first starting a new SM on the new host, and getting all of
# the current control variables on the old SM.  Then it kills the old SM,
# tells the parent and children of the department where the new SM is at and
# initialize the new SM with the values of the control variables from the
# old sm.
#
# Return Value(s):
#  None
#
# Global variable modifications:
#  None
#
# Example of usage:
#  $hi->move_aggregator($Department,$Host);
#
#-------------------------------------------------------------------------#

sub move_aggregator
{
  my($self,$dept,$agg_host) = @_;
  my($log) = $self->{'log'};
  my($trans_id) = int(rand(10000000));
  my($return);

  $log->separator();
  $log->warn("Move aggregator: ".join(' ',@_[1..$#_]));

  # Check if this is at all sensible
  my($htr_ref) =
    ['move_aggregator',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $dept, $agg_host];
  my($status,$err) = $self->initiate_transaction($htr_ref,
                                \&Hierarchy::move_aggregator_is_sane);
  return ($status,$err) unless $status;

  my($agg_port);
 
  # Marshall the information we need out of the hierarchy.
  my($parent) = $self->{'hierarchy'}{$dept}{'parent'};
  my($parent_dept_object) = $self->{'hierarchy'}{$parent};
  my($agg_host_obj) = $self->{'hierarchy'}{$agg_host}; # software manager
  my($host_dept_obj) = $self->{'hierarchy'}{$agg_host_obj->{'parent'}};
  my($parent_agg_host) = $parent_dept_object->{'aggregator_host'};
  my($parent_agg_port) = $parent_dept_object->{'aggregator_port'};
  my($parent_man_host) = $parent_dept_object->{'manager_host'};
  my($parent_man_port) = $parent_dept_object->{'manager_port'};
#  my($agg_host_obj) = $self->{'hierarchy'}{$agg_host};
#
# Uncommented by J.Rowe: After moving the aggregator, subsequent
# transactions attempt to lookup the new version with the wrong
# port number, thus the following modification.
  my $agg_mc_port  = $agg_host_obj->{'port'};

  # Get rulesets for engine. 
  my ($request) = ['rulesets{ALL_STATE_NAMES}','_'];
  my ($ok,$rslt) = $self->get($dept,'','',@$request);
  my ($request) = [@$rslt];
  my ($ok,$rslt) = $self->get($dept,'','',@$request);

#  Added by J.Rowe: The rulesets in the old aggregator were propogated
#  but not the children, thus the following inclusions.
  my($old_host) = $self->{'hierarchy'}{$dept}->{'aggregator_host'};
  my %old_cldrn = @{$self->get($old_host,'engine',$dept,
                   'children{ALL_STATEVARS}',1)};
#

  # Startup the new aggregator.
  my($shv_ref) = [$host_dept_obj->{'name'},$agg_host,'module_controller','version',
                $dept,'mc_command','START',
                'mod_module',           'engine',
                'mod_department',       $dept,
                'mod_version',          'v.v',
                'parent_aggregator',    "$parent_agg_host:$parent_agg_port",
                'parent_manager',       "$parent_man_host:$parent_man_port",
                %old_cldrn,
                @$rslt
                ];

  my $sm_messages = [[$host_dept_obj->{'manager_host'},
                        $host_dept_obj->{'manager_port'},'shv',$shv_ref]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);
  unless($err)
   {
    my %hash = @{$replies->[0][3]}; # extract body of reply
    $agg_port = $hash{'listen_tcp'};
    unless($agg_port)
     {   
      $err = "No aggregator port in add_dept- all bets are off";
      $log->warn($err);
      goto END_add_dept;
     }   
   }  

  # The following can all be done at once.
  my @sm_messages;

  my($old_agg_host) = $self->{'hierarchy'}{$dept}->{'aggregator_host'};
  my($old_agg_host_obj) = $self->{'hierarchy'}{$old_agg_host}; # software manager
  my($old_host_dept_obj) = $self->{'hierarchy'}{$old_agg_host_obj->{'parent'}};

  # Kill the old aggregator. 
  my($shv_ref) = [$old_host_dept_obj->{'name'},$old_agg_host,
                  'module_controller','version',$dept,
                  'mc_command', 'KILL',
                  'mod_module', 'engine',
                  'mod_version', 'v.v',
                  'mod_department', $dept,
                  ];
  
  push (@sm_messages , [$old_host_dept_obj->{'manager_host'},
                        $old_host_dept_obj->{'manager_port'},'shv',$shv_ref]);
 
  # Tell our SM about the change.
  my($our_dept_obj) = $self->{'hierarchy'}{$dept};

# Aggregator_mc field added to the sdv to resolve later engine lookup 
# problems as mentioned above. J.Rowe
  my $sdv_ref = [$dept,"our_aggregator", "$agg_host:$agg_port",
                       "aggregator_mc",  "$agg_host:$agg_mc_port"];
 
  push (@sm_messages, [$our_dept_obj->{'manager_host'},
                 $our_dept_obj->{'manager_port'},'sdv',$sdv_ref]);
 
#

  #  Tell the children of the Dept where the new aggregator is
  my $child;
  foreach $child (@{$self->{'hierarchy'}{$dept}{'dept_children'}})
   {
    my $child_dept_object  = $self->{'hierarchy'}{$child};

    my $sdv_ref = [$child,"parent_aggregator", "$agg_host:$agg_port"];

    push (@sm_messages, [$child_dept_object->{'manager_host'},
                   $child_dept_object->{'manager_port'},'sdv',$sdv_ref]);
   }

  # Send all of the messages at once.
  ($err,$replies) = $self->communicate_with_sms(\@sm_messages);
 
#

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['move_aggregator',$trans_id,$agg_port];
  return $self->complete_transaction($err,$htr_ref,
                                       $htc_ref,\&Hierarchy::move_aggregator);

}# end move_aggregator

#============================================================================#
# 
# Function: remove_host
# 
# This function is to remove a Host from the hierarchy.  It checks to make
# sure it is sane, then it kills all modules running on the host.  Then it
# removes the host from the hierarchy.
#
# Inputs:
#  Actual Arguments:
#   Host
#  Global:
#   None
# 
# Return Value(s):
#  None
# 
# Global variable modifications:
#  None
# 
# Example of usage:
#  $hi->remove_host($host);
#
#-------------------------------------------------------------------------# 

sub remove_host
{
  my($self,$host) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};
  my($status,$err,$replies);
 
  $log->separator();
  $log->warn("Remove host: ".join(' ',@_[1..$#_]));
 
  # Check if this is at all sensible and get permission from the OHS

  my($htr_ref) =
    ['remove_host',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $host];
  ($status,$err) = $self->initiate_transaction($htr_ref,
                                \&Hierarchy::remove_host_is_sane);
  return ($status,$err) unless $status;
 
  # We have permission from the OHS. Kill all modules on host.

  my($host_obj) = $self->{'hierarchy'}{$host};
 
  my ($status,$replies) =  $self->get($host,'module_controller',
                  $host_obj->{'parent'},
                  'mc_command', 'INVENTORY',
                  'mod_module', '*',
                  'mod_version', '*',
                  'mod_department', '*',
                  );

  unless ($status)
   {
    my $htc_ref = ['remove_host',$trans_id];
    return $self->complete_transaction($replies,$htr_ref,
                                        $htc_ref,\&Hierarchy::remove_host);
   }

  my ($inv, $module_string);
  my $temp_replies = $replies;
  while ($inv = shift (@{$temp_replies}))
   {
    $module_string = shift (@{$temp_replies});
    if ($inv =~ /^\nINVENTORY/)
     {
      my (@module_array) = split (' ',$module_string);

      my ($status,$replies) =  $self->set($host,'module_controller',
                  $host_obj->{'parent'},
                  'mc_command', 'KILL',
                  'mod_module', @module_array[0],
                  'mod_version', @module_array[3],
                  'mod_department', @module_array[1],
                  );
       $log->warn($replies) unless $status;

     }    
   }# end while

  # Remove host as child of its parent.
  my $old_parent        = $self->{'hierarchy'}{$host}{'parent'};
  my $old_dept_object   = $self->{'hierarchy'}{$old_parent};
 
  my($sdv_ref) = [$old_parent,"host_child\{$host\}",''];

  my $sm_messages = [[$old_dept_object->{'manager_host'},
                        $old_dept_object->{'manager_port'},'sdv',$sdv_ref]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['remove_host',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
                                        $htc_ref,\&Hierarchy::remove_host);
}# end remove_host

#============================================================================#
# 
# Function: remove_dept
# 
# This function is to remove an existing 'Department' within the hierarchy.
# Note:  See technical document for a full description of message.
#
# Return Value(s):
#  None        
#   
# Global variable modifications:
#  None        
#  
# Example of usage:
#  $hi->remove_dept($dept);
# 
#-------------------------------------------------------------------------#
 
sub remove_dept
{
  my($self,$dept) = @_;
  my($trans_id) = int(rand(10000000));
  my($return);
  my($log) = $self->{'log'};
  my($status,$err,$replies);
  my (@host_array, @dept_array, %dept_asc);
  my (@temp_host_array, @temp_dept_array, $temp_dept, $temp_host);

  $log->separator();
  $log->warn("Remove dept: ".join(' ',@_[1..$#_]));

  # Check if this is at all sensible and get permission from the OHS

  my($htr_ref) =
    ['remove_dept',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $dept];
  ($status,$err) = $self->initiate_transaction($htr_ref,
                                \&Hierarchy::remove_dept_is_sane);
  return ($status,$err) unless $status;

 # Remove dept as a child of its parent.
  my $old_parent        = $self->{'hierarchy'}{$dept}{'parent'};
  my $old_dept_object   = $self->{'hierarchy'}{$old_parent};

  my(@sdv_ref);
  push(@sdv_ref,[$old_parent,"dept_child\{$dept\}",'']);

  # Tell the SM to terminate.
  my $dept_object   = $self->{'hierarchy'}{$dept};

  push(@sdv_ref, [$dept,"local_managers\{$dept\}",'']);
  my $sm_messages = [[$old_dept_object->{'manager_host'},
                        $old_dept_object->{'manager_port'},'sdv',$sdv_ref[0]]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  my $sm_messages = [[$dept_object->{'manager_host'},
                        $dept_object->{'manager_port'},'sdv',$sdv_ref[1]]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.  

  my $htc_ref = ['remove_dept',$trans_id];
  return $self->complete_transaction($err,$htr_ref,
                                        $htc_ref,\&Hierarchy::remove_dept);

}# end remove_dept

#============================================================================#
# 
# Function: set
# 
# This function is to set department and host variables by the user.
# It begins by determining if we are dealing with host or department vars.,
# then it sends a 'change_variable' message to the OHS.  Next it gets the
# variables and values to be set, does some error checking and sends the
# appropriate message to the SM.  Finally, it sends the htc back to the
# OHS, signifying completion. 
# 
# Arguments:
#  Actual:
#   Dept/host, module name, dept_name, var_name, value, var_name, value, ...
#  Global (to the Hierarchy_interface):
#   $Hierarchy_interface::dept_set_not_allowed{}
#
# Return Value(s):
#  $self
#     
# Global variable modifications:
#  None   
#    
# Example of usage:
#  $hi->set($dept_or_host,$module_name,$dept,$name,$value,$name,$value,...);
# 
#-------------------------------------------------------------------------#

sub set
{
  my($self,$target,$module,$dept) = @_;
  my($trans_id) = int(rand(10000000));
  my($return,$set_dept,$set_type);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Set: ".join(' ',@_[1..$#_]));
 
  # Determine if the $target is a department or a host.
  if (ref $self->{'hierarchy'}{$target} eq 'Department')
   {
    $set_dept = $target;
    $set_type = 'sdv';
   } elsif (ref $self->{'hierarchy'}{$target} eq 'Host') {
    $set_dept = $self->{'hierarchy'}{$target}{'parent'};
    $set_type = 'shv';
   } else {
    $log->warn("Argument is not a department or host: $target");
    return (0, "Argument is not a department or host: $target");
   }
 
  # We must send the module name and dept name if it is a shv. 
  my @set_vals = splice(@_,4);
  
  unshift @set_vals, $target,$module,'version',$dept if $set_type eq 'shv';
  my $cnt = 0;
  $cnt = 4 if $set_type eq 'shv';

  # Check variables and values for errors.
  # Make sure all variables can be set by the user.
  while ($set_vals[$cnt])
   {
    my ($set_var,$index) = $self->parse_var_name ($set_vals[$cnt]);

    if((($set_type eq 'sdv') &&
          ($Hierarchy_interface::dept_set_not_allowed{$set_var})) ||
       (($set_type eq 'shv') &&
          ($Hierarchy_interface::host_set_not_allowed{$set_var})))
     { 
      $log->warn("Variable cannot be set by user: @set_vals[$cnt]"); 
      return(0, "Variable cannot be set by user: @set_vals[$cnt]");
     } 
    # Make sure there is a value with the variable
    unless (defined $set_vals[++$cnt])
     {
      $log->warn("Invalid arguments to set: $set_vals[$cnt - 1] with no value");
      return (0,"Invalid arguments to set: $set_vals[$cnt - 1] with no value");
     } 

    # Special case for rulesets, append root_rule to keep track of where this 
    # ruleset was defined.
    if ($set_var eq 'rulesets')
     { 
#      push @set_vals, 'root_rule{'.$index.'}',1;
     }
    $cnt ++;
   }# end while
 
  unshift @set_vals,$set_dept; # So the SM knows which dept it is dealing with.

  # Tell the OHS what we plan to do
  my($htr_ref) =
    ['change_variable',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'},  $set_dept];
  my($status,$err) = $self->initiate_transaction($htr_ref,
				\&Hierarchy::change_variable_is_sane);
  return ($status,$err) unless $status;
 
  # We have permission from the OHS.  Now talk to the software manager.
  my($dept_object) = $self->{'hierarchy'}{$set_dept};
  my $sm_messages = [[$dept_object->{'manager_host'},
			$dept_object->{'manager_port'},$set_type,\@set_vals]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);

  # Complete the transaction with the OHS and update ourselves.
  my $htc_ref = ['change_variable',$trans_id];
  return $self->complete_transaction($err,$htr_ref,$htc_ref,
						\&Hierarchy::change_variable);
} # end set 

#============================================================================#
#
# Function: get
#
# This function is to get department and host variables by the user.
# It begins by determining if we are dealing with host or department vars.
# Next it gets the variables and values to be gotten, does some error 
# checking and sends the appropriate message to the SM.  The value returned
# is the reply message from the SM.
#
# Arguments:
#  Actual: 
#   Dept/host, module name, dept_name, var_name, value, var_name, value, ...
#
# Return Value(s):
#  Return array from gdvr/ghvr containing the names/values requested, 
#  else undef    
#      
# Global variable modifications:
#  None     
#        
# Example of usage:
#  $hi->get($dept_or_host,$module_name,$dept,$name,$value,$name,$value,...);
#
#-------------------------------------------------------------------------#

sub get
{
  my($self,$target,$module,$dept) = @_;
  my($trans_id) = int(rand(10000000));
  my($return,$get_dept,$get_type);
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Get: ".join(' ',@_[1..$#_]));
 
  # Determine if the $target is a department or a host.
  if (ref $self->{'hierarchy'}{$target} eq 'Department')
   {
    $get_dept = $target;
    $get_type = 'gdv';
   } elsif (ref $self->{'hierarchy'}{$target} eq 'Host') {
    $get_dept = $self->{'hierarchy'}{$target}{'parent'};
    $get_type = 'ghv';
   } else {
    $log->warn("Argument is not a department or host: $target");
    return (0, "Argument is not a department or host: $target");
   }
 
  # Decide if this is a sane thing to do.
  my($htr_ref) =
    ['change_variable',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'},  $get_dept];
  return (0, "Change variable is not sane")
    unless $self->{'hierarchy'}->change_variable_is_sane($htr_ref);
 
  # We Do not need to lock the Hierarchy, so we do not tell the OHS.
 
  # Now talk to the software managers.

  # We must send the module name and dept name if it is a shv. 
  my @get_vals = splice(@_,4);
  
  unshift @get_vals, $target,$module,'version',$dept if $get_type eq 'ghv';
  my $cnt = 0;
  $cnt = 4 if $get_type eq 'ghv';

  # Check variables and values for errors.
  while (@get_vals[$cnt])
   {
    # Make sure there is a value to go with the variable.
    unless (defined $get_vals[++$cnt])
     {
      $log->warn("Invalid arguments to get: $get_vals[$cnt - 1] with no value");
      return (0, "Invalid arguments to get: $get_vals[$cnt - 1] with no value");
     }
    $cnt++;
   }# end while

  unshift @get_vals, $get_dept;

  #  Now talk to the software manager.
  my($dept_object) = $self->{'hierarchy'}{$get_dept};
  my $sm_messages = [[$dept_object->{'manager_host'},
			$dept_object->{'manager_port'},$get_type,\@get_vals]];
  my($err,$replies) = $self->communicate_with_sms($sm_messages);


  # Complete the transaction with the OHS and update ourselves.
  return (0,$err) if $err;
  return (1,$replies->[0][3]); # reference to body of message

} # end get

#============================================================================#
#============================================================================#

#				CONSISTENCY CHECKS

# Useful functions to see if everything is as it should be.

#============================================================================#
#============================================================================#

sub is_consistent
{
  my($self) = @_;
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Is consistent: ".join(' ',@_[1..$#_]));

  return $self->{'hierarchy'}->is_consistent();
}

#============================================================================#

sub matches_ohs
{
  my($self) = @_;
  my($log) = $self->{'log'};

  $log->separator();
  $log->warn("Matches_ohs: ".join(' ',@_[1..$#_]));

  my $local_h = $self->{'hierarchy'}; # Cache old copy
  delete $local_h->{'ACL'};
  $self->refresh(); # Get copy from ohs
  return $self->{'hierarchy'}->compare('ohs','local',$local_h);
}

#============================================================================#

sub matches_managers
{
  my($self) = @_;
  my($trans_id) = int(rand(10000000));
  my($log) = $self->{'log'};
  $log->separator();
  $log->warn("Matches managers: ".join(' ',@_[1..$#_]));
 
  # This is going to take a while and would get fucked up if anyone
  # changed the hierarchy under us.  So we will lock first.
  my($htr_ref) =
    ['change_variable',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $self->{'hierarchy'}->{'root_name'}];
  return 0 unless $self->{'hierarchy'}->change_variable_is_sane($htr_ref);
  $self->send_to_OHS('htr', $htr_ref);
  my $return = $self->receive_htp();
  unless($return eq $trans_id)
   {
    $log->warn("Transaction $trans_id rejected: $return");
    return 0;
   }
 
  # We have permission from the OHS.  Now talk to the software managers.
  my @names = sort grep($_ ne '_log' && $_ ne 'root_name',
					keys %{$self->{'hierarchy'}});
  my $success = 1; 
  my($key,$key_object,$key_type);

  foreach $key (@names)
   {
    $key_object = $self->{'hierarchy'}{$key};
    $key_type = ref $key_object;

    if($key_type eq 'Host')
     {
      # Here we talk to the module controller and ask it whether it knows the
      # information it is supposed to.
     }
    elsif($key_type eq 'Department')
     {

      my %reply = @{$self->get($key,0,0,'parent',1,'department',1,
		'our_aggregator',1,'parent_aggregator',1,'parent_manager',1,
		'dept_child{ALL_STATE_NAMES}',1,
		'host_child{ALL_STATE_NAMES}',1)};

      unless($reply{'department'} eq $key)
       {
        $success = 0;
	$log->warn("Manager for $key department thought its name".
		" was $reply{'department'}");
       }
      unless($reply{'parent'} eq $key_object->{'parent'})
       {
        $success = 0;
	$log->warn("Manager for $key department thought its parent".
		" was $reply{'parent'}, but interface thought it should be".
		" $key_object->{'parent'}");
       }
      my $local_our_aggregator = 
	  "$key_object->{'aggregator_host'}:$key_object->{'aggregator_port'}";
      unless($reply{'our_aggregator'} eq $local_our_aggregator)
       {
        $success = 0;
	$log->warn("Manager for $key department thought its aggregator".
		" was $reply{'our_aggregator'}, but interface thought it".
		" should be $local_our_aggregator");
       }

      my $parent_name = $key_object->{'parent'};
      unless($parent_name eq 'null')
       {
        my $parent_object = $self->{'hierarchy'}{$parent_name};
        my $local_parent_aggregator = "$parent_object->{'aggregator_host'}:".
		"$parent_object->{'aggregator_port'}";
        unless($reply{'parent_aggregator'} eq $local_parent_aggregator)
         {
          $success = 0;
	  $log->warn("Manager for $key department thought its parent ".
		"aggregator".
		" was $reply{'parent_aggregator'}, but interface thought it".
		" should be $local_parent_aggregator");
         }
        my $local_parent_manager = "$parent_object->{'manager_host'}:".
		"$parent_object->{'manager_port'}";
        unless($reply{'parent_manager'} eq $local_parent_manager)
         {
          $success = 0;
	  $log->warn("Manager for $key department thought its parent ".
		"manager".
		" was $reply{'parent_manager'}, but interface thought it".
		" should be $local_parent_manager");
         }
       }
      my $reply_dept_children = 
	join('',sort map {$1 if /^dept_child\{(.*)\}$/;} (keys %reply));
      my $reply_host_children = 
	join('',sort map {$1 if /^host_child\{(.*)\}$/;} (keys %reply));
      my $local_dept_children = 
	join('',sort @{$key_object->{'dept_children'}});
      my $local_host_children = 
	join('',sort @{$key_object->{'host_children'}});
      unless($reply_dept_children eq $local_dept_children)
       {
        $success = 0;
	$log->warn("Manager for $key department thought its dept children".
		" were $reply_dept_children, but interface thought they".
		" should be $local_dept_children");
       }
      unless($reply_host_children eq $local_host_children)
       {
        $success = 0;
	$log->warn("Manager for $key department thought its host children".
		" were $reply_host_children, but interface thought they".
		" should be $local_host_children");
       }
     }
    else
     {
      $success = 0;
      $log->warn("Bad element $key of type $key_type");
     }
   }

  # Now let the OHS know that we completed successfully.
  my $htc_ref = ['change_variable',$trans_id];
  $self->send_to_OHS('htc', $htc_ref);
 
  # Finally, update our own data structure.
  $self->{'view_serial'}++;
#
  return $success;

} # end matches_managers 

#============================================================================#

sub matches_aggregators
{
  my($self) = @_;
  my($trans_id) = int(rand(10000000));
  my($log) = $self->{'log'};
  $log->separator();
  $log->warn("Matches aggregators: ".join(' ',@_[1..$#_]));
 
  # This is going to take a while and would get fucked up if anyone
  # changed the hierarchy under us.  So we will lock first.
  my($htr_ref) =
    ['change_variable',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $self->{'hierarchy'}->{'root_name'}];
  return 0 unless $self->{'hierarchy'}->change_variable_is_sane($htr_ref);
  $self->send_to_OHS('htr', $htr_ref);
  my $return = $self->receive_htp();
  unless($return eq $trans_id)
   {
    $log->warn("Transaction $trans_id rejected: $return");
    return 0;
   }
    
  # We have permission from the OHS.  Now talk to the software managers.
  my @names = sort grep($_ ne '_log' && $_ ne 'root_name',
                                        keys %{$self->{'hierarchy'}});
  my $success = 1;
  my($key,$key_object,$key_host,$key_type);
    
  foreach $key (@names)
   {
    $key_object = $self->{'hierarchy'}{$key};
    $key_type = ref $key_object;

    if($key_type eq 'Department')
     {
      $key_host = $key_object->{'aggregator_host'};
  
      unless ($key_object->{'parent'} eq 'null')
       {
        my %reply = @{$self->get($key_host,'engine',$key,
                    'parent_aggregator',1,'parent_manager',1)};

        my $our_parent_aggregator = 
              ("$self->{'hierarchy'}->{$key_object->{'parent'}}->{'aggregator_host'}:".
               "$self->{'hierarchy'}->{$key_object->{'parent'}}->{'aggregator_port'}");

        unless($reply{'parent_aggregator'} eq $our_parent_aggregator)
         { 
          $success = 0;
          $log->warn("Aggregator for $key department thought its parent aggregator". 
                  " was $reply{'parent_aggregator'}."); 
         }

        my $our_parent_manager = 
              ("$self->{'hierarchy'}->{$key_object->{'parent'}}->{'manager_host'}:". 
               "$self->{'hierarchy'}->{$key_object->{'parent'}}->{'manager_port'}"); 
 
        unless($reply{'parent_manager'} eq $our_parent_manager)
         {  
          $success = 0;   
          $log->warn("Aggregator for $key department thought its parent manager". 
                  " was $reply{'parent_manager'}."); 
         } 
       }# end unless

      my @int_children = (@{$key_object->{'dept_children'}},
                   @{$key_object->{'host_children'}});

      my %agg_children = @{$self->get($key_host,'engine',$key,
                   'children{ALL_STATEVARS}',1)};
      my ($int_name);
      while ($int_name = shift (@int_children))
       {
        unless ($agg_children{"children{$int_name}"} == 1)
         {
          $log->warn("Aggregator for $key department does not know about ".
               "child $int_name.");
          $success = 0;
         }
       }# end while

     } # end if
   }# end foreach

  # Now let the OHS know that we completed successfully.
  my $htc_ref = ['change_variable',$trans_id];
  $self->send_to_OHS('htc', $htc_ref);
 
  # Finally, update our own data structure.
  $self->{'view_serial'}++;
  return $success;

}# end matches_aggregators 
  
#============================================================================#
# 
# Function Name: rulesets_consistent
# 
# This function checks to see if the ruleset(s) for a department and its 
# children are consistent.
#
# If the ruleset name argument is blank, then the function will check all
# of the rulesets for the department.
#
# Inputs:
# $dept_name - Department name being checked
#  Optional
#  $rule_name - Name of ruleset being checked
#  $correct_rule - The actual correct ruleset
#
# Return Value(s):
#   0,1 depending on success
#
# Examples of usage:
#  ($h->rulesets_consistent('ROOT'))
#  ($h->rulesets_consistent('ROOT','easy'))
#
#----------------------------------------------------------------------------#

sub rulesets_consistent
{
  my($self, $dept_name, $rule_name, $correct_rule) = @_;
  my($trans_id) = int(rand(10000000));
  my($log) = $self->{'log'};
  my $success = 1;

  $log->warn("Rulesets Consistent test begun\n");

  if(defined $dept_name)
  {$log->warn("$dept_name is defined:  $dept_name\n");}
  if(defined $rule_name)
  {$log->warn("rule_name is defined:  $rule_name\n");}
  if(defined $correct_rule)
  {$log->warn("$correct_rule is defined:  $correct_rule\n");}
  if(defined $correct_rule)
   {
    my @sm_rule = @{$self->get($dept_name,0,0,'rulesets{'.$rule_name.'}','1')};
    unless (@sm_rule[1] eq $correct_rule)
     {
      $log->warn("$dept_name's SM ruleset $rule_name is incorret.");
      $success = 0;
     }

    my @agg_rule = @{$self->get($self->{'hierarchy'}->{$dept_name}->{'aggregator_host'},
                                'engine', $dept_name,'rulesets{'.$rule_name.'}','1')};
    unless (@agg_rule[1] eq $correct_rule)
     {
      $log->warn("$dept_name's aggregator ruleset $rule_name is incorret.");
      $success = 0;
     }

    my @children = @{$self->{'hierarchy'}->{$dept_name}->{'dept_children'}};
    my $child;
    while ($child = shift(@children))
     { $success *= $self->rulesets_consistent($child, $rule_name, $correct_rule); }

    return $success;

   }# end if

  $log->separator();
  $log->warn("Rulesets consistent: ".join(' ',@_[1..$#_]));
 
  # This is going to take a while and would get fucked up if anyone
  # changed the hierarchy under us.  So we will lock first.
  my($htr_ref) =
    ['change_variable',$trans_id,$self->{'user'},$self->{'password'},
        $self->{'view_serial'}, $self->{'hierarchy'}->{'root_name'}];
  return 0 unless $self->{'hierarchy'}->change_variable_is_sane($htr_ref);
  $self->send_to_OHS('htr', $htr_ref);
  my $return = $self->receive_htp();
  unless($return eq $trans_id)
   {
    $log->warn("Transaction $trans_id rejected: $return");
    return 0;
   }
 
  # We have permission from the OHS.  Now talk to the software managers.

  my %check_rules;
  if (defined $rule_name)
  {$log->warn("Rulename is defined $rule_name\n");
    $check_rules{"rulesets\{$rule_name\}"} = 1;}
  else
   {
    $log->warn("Rulename is NOT defined\n");
    #my @rules = @{$self->get($dept_name,0,0,'rulesets{ALL_STATE_NAMES}',1)};
    #my @inherited_rules = 
    #       @{$self->get($dept_name,0,0,'inherited_rulesets{ALL_STATE_NAMES}',1)};
    %check_rules = @{$self->get($dept_name,0,0,'rulesets{ALL_STATE_NAMES}',1)};
   } 

  my ($junk,$rule,$full_name,@sm_rule, @agg_rule);
  foreach $full_name (keys (%check_rules))
   {
  $log->warn("Rulename is $full_name\n");

    ($junk,$rule) = $self->parse_var_name ($full_name);

    @sm_rule = @{$self->get($dept_name,0,0,'rulesets{'.$rule.'}','1')};
    @agg_rule = @{$self->get($self->{'hierarchy'}->{$dept_name}->
                  {'aggregator_host'}, 'engine',
                  $dept_name,'rulesets{'.$rule.'}','1')};

    if (@sm_rule[1] ne @agg_rule[1])
     {
      $log->warn("Initial Aggregator and sm do not agree about ruleset ".
                 "$rule_name, cannot proceed.");
      $success = 0;
     } 
    else
     {$success *= $self->rulesets_consistent($dept_name,$rule,@sm_rule[1]);}
   }# end while

  # Now let the OHS know that we completed successfully.
  my $htc_ref = ['change_variable',$trans_id];
  $self->send_to_OHS('htc', $htc_ref);
 
  # Finally, update our own data structure.
  $self->{'view_serial'}++;
  return $success;

} # end rulesets_consistent 


#============================================================================#
#============================================================================#

#				COMMUNICATION INTERFACE

#============================================================================#
#============================================================================#

# Function:  send_to_OHS
 
# This function sends a header and a message to the OHS.
 
# Return Value(s):
#  None
 
# Example of usage:
#  $self->send_to_OHS('hvr',$messageref);
 
sub send_to_OHS
{
  my($self,$header,$messageref) = @_;
  my $log = $self->{'log'};
  my $serial = $self->{'view_serial'};
  $log->warn("Before sending to OHS, my local serial is $serial\n") if ( defined($serial));
  if($self->{'output_mode'} eq 'comm')
   {
    my($hostname,$port) = split(/:/,$self->{'ohs_location'});
    send_messages([[$hostname,$port,$header,$messageref]],$self->{'log'})
   }
  else
   {
    die "Non comm modes presently not working sorry."
   }

}# end send_to_OHS

#============================================================================#

# Function:  initiate_transaction
 
# This function initiates a transaction.  It checks whether this is
# sensible in our internal data structures and then sets things up with
# the OHS.
 
# Return Value(s):
#   (1,$self) on success
#   (0,$err) on failure
#   $err which is false if successful, and explanatory if failure.
 
sub initiate_transaction
{
  my($self,$htr_ref,$sanity_func) = @_;
  my($log) = $self->{'log'};
  my $err = '';
  my($status);

  # Check sanity internally
  return(0,"Transaction insane!!") 
	unless &$sanity_func($self->{'hierarchy'},$htr_ref);
  
  # Talk to OHS
  my($hostname,$port) = split(/:/,$self->{'ohs_location'});
  my $ohs_mesgs = [[$hostname,$port,'htr',$htr_ref]];

  my ($send_status, %results) = send_messages($ohs_mesgs,$self->{'log'});
  return (0,"FAILED in sending to OHS") unless $send_status;

  # Now listen
  my ($recv_status,$replies) = receive_messages($ohs_mesgs,$log);
  return (0,"Receive from OHS failed!!!") unless $recv_status;

  # Ok - we got a reply.  But was it good news?
  my($host1,$port1,$header,$body_ref) = @{$replies->[0]};
  $err .= "OHS says NO: $header @$body_ref\n" unless($header eq 'htp');

  my($our_trans,$ohs_trans) = ($htr_ref->[1],$body_ref->[0]);
  $err .= "Transactions ids disagree!?! $our_trans vs $ohs_trans\n" 
					unless($our_trans eq $ohs_trans);
  if($err)
   {
    $log->warn($err);
    return (0,$err);
   }
  return (1);

}# end initiate_transaction

#============================================================================#

# Function:  complete_transaction
 
# This function completes a transaction.  It replies appropriately to
# the OHS, and then updates our internal data structures.
 
# Return Value(s):
#   (1,$self) on success
#   (0,$err) on failure
#   $err which is false if successful, and explanatory if failure.
 
sub complete_transaction
{
  my($self,$err,$htr_ref,$htc_ref,$update_func) = @_;
  my $header = 'htc'; #optimistic

  my($hostname,$port) = split(/:/,$self->{'ohs_location'});

  if($err)
   {
    $header = 'hte';
    $htc_ref = [$htc_ref->[1],$err]; # re-arrange into hte format
   }

  my $ohs_mesgs = [[$hostname,$port,$header,$htc_ref]];
  my $send_status = send_messages($ohs_mesgs,$self->{'log'});
  $err .= "FAILED in sending to OHS" unless $send_status;

  return (0,$err) if $err;

  # Finally, update our own data structure.
  $self->{'view_serial'}++; 
  return (1,$self) if &$update_func($self->{'hierarchy'},$htr_ref,$htc_ref);
  
  # If we get here it's very bad news.  
  return (0, "ERROR!!!! updated OHS? but own update failed!!!");

}# end complete_transaction

#============================================================================#

sub communicate_with_sms
{
  my($self,$mesgs_ref,$blocking) = @_;
  my($log) = $self->{'log'};
  my($err) = '';
  my($one_mesg_ref,$one_reply_ref);

  $blocking = "blocking" unless defined $blocking;
 
  # Initial error checking and preparation
  unless(defined $mesgs_ref && ref $mesgs_ref eq 'ARRAY')
   {
    $err = 'Bad $mesgs_ref argument in communicate_with_sms: '.$mesgs_ref;
    $log->warn($err);
    return($err,undef,undef);
   }
  foreach $one_mesg_ref (@$mesgs_ref)
   {
    unless(defined $one_mesg_ref && ref $one_mesg_ref eq 'ARRAY')
     {
      $err = 'Bad $one_mesg_ref argument in communicate_with_sms: '
							.$one_mesg_ref;
      $log->warn($err);
      return($err,undef,undef);
     }
    # Add to the body the username and password which all sdvs need.
    unshift(@{$one_mesg_ref->[3]},$self->{'user'},$self->{'password'});
   }

  my %done_send; 
  my $key;

  # Now talk.
  my ($send_status,%done_send) = send_messages($mesgs_ref,$log);
  foreach $key (keys %done_send){
        #print STDERR ("sm_comm  $key, $done_send{$key} - \n");
  }
  #return ("Sends to SM failed!!!", undef,$return_send_ref) unless $send_status;


  my $receive_mesgs_ref;
  my $num_receive_mesgs = 0;
  my @pos_receive_mesgs;  
  my @return_successes;  


  my $len = scalar(@$mesgs_ref);
  my $start = 0;
  my $end = $len;
  my $i; 

  for ($i = $start; $i < $end; $i += 1)
   {
    if ($done_send{$i} == 1)
     {
      push(@$receive_mesgs_ref,@$mesgs_ref[$i]);
      push(@return_successes,1);
      push(@pos_receive_mesgs,$i);
      $num_receive_mesgs++ ;
     }
    else
     {
      push(@return_successes,0);
     } 
   }

  # SHOULD NOT RETURN HERE, MUST REMOVE BAD MESSAGES AND THEN WAIT TO 
  # RECEIVE FROM THE GOOD ONES.

  return ("Sends to SM failed!!!", undef,%done_send) if $num_receive_mesgs == 0;

  my $num = scalar @$mesgs_ref;

  my %done; 
  # Now listen
  my ($recv_status,$replies,%done) = receive_messages($receive_mesgs_ref,$log,$blocking);

  foreach $key (keys %done){
        #print STDERR ("sm_comm  $key, $done{$key} - \n");
  }

  $len = scalar(@pos_receive_mesgs);
  $start = 0;
  $end = $len;
 
  for ($i = $start; $i < $end; $i += 1)
   {   
    my($host,$port,$header,$body_ref) = @$replies[$i];
    @return_successes[@pos_receive_mesgs[$i]] = $done{$i};
   }

  $len = scalar(@return_successes); 
  $start = 0; 
  $end = $len; 
 
  for ($i = $start; $i < $end; $i += 1) 
   {    
#   print STDERR ("FINAL -- $i, @return_successes[$i]\n");
   } 

  my $status = $send_status * $recv_status; 

  return ("Communications failed!!!", undef,@return_successes) unless $status;

  my $pos = 0;

  # Ok - we got replies.  But were they good news?
  foreach $one_reply_ref (@$replies)
   {
    my($host,$port,$header,$body_ref) = @$one_reply_ref;
    unless($header =~ /^(sdvr|shvr|gdvr|ghvr)$/)
     {
      $err .= "ERROR from $host:$port -- $header @$body_ref\n";
      $log->warn($err);
      @return_successes[@pos_receive_mesgs[$pos]] = 0;
     }
    $pos ++;
   }
  return($err,$replies,@return_successes);
}

#============================================================================#

# Function:  send_to_SM

# This function sends a header and a message to the software manager on the
# specified 'Host' using the given 'Port'.

# Return Value(s):
#  None

# Example of usage:
#  $self->send_to_SM($man_host,$man_port,'sdv',$sdv_ref);

sub send_to_SM
{
  my($self,$hostname,$port,$header,$messageref) = @_;

  if($self->{'output_mode'} eq 'comm')
   {
    $self->{'log'}->warn("Interface about to send $header:@$messageref to $hostname:$port\n");
    unshift @$messageref,$self->{'user'},$self->{'password'};
    my $comm_return = Comm::tcp_send($hostname,$port,$header,@$messageref);
    unless ($comm_return eq 'true')
     {
      $self->{'log'}->warn("tcp_send failed:$comm_return");
      return;
     }
   }
  else
   {
    my $output = "$header#$self->{'user'}#$self->{'password'}#"
					.join('#',@$messageref)."\n";
    if($self->{'output_mode'} eq 'stdio')
     {
      print $output;
     }
    elsif($self->{'output_mode'} eq 'file')
     {
      my $location = "$hostname:$port";
      open(SEND,">>pipes/$location");
      print SEND "$self->{'my_location'}#$output";
      close(SEND);
     }
   }

}# end send_to_SM

#============================================================================#
#
# Function:  receive_htp
#
# This function receives either an htp or hte from the OHS.
#
# Return Value(s):
#  htp, hte or undefined in case of error
#
# Example of usage:
#  $return = $self->receive_htp();
#
#----------------------------------------------------------------------------#

sub receive_htp
{
  my($self) = shift;
  my($header,$body_ref) = $self->receive();
  if($header eq 'htp') 
   {return ${$body_ref}[0];}
  elsif($header eq 'hte')
   {
    $self->{'log'}->warn("Transaction rejected: $header @$body_ref");
    return undef;
   }
  else
   {
    $self->{'log'}->warn("Bad htp reply: H:$header B:@$body_ref");
    return undef;
   }
}#end receive_htp

#============================================================================#
#
# Function:  receive_sdvr
# 
# This function receives either an sdvr or sdve message.
# 
# Return Value(s):
#  sdvr, sdve or undefined in case of error
# 
# Example of usage:
#  $return = $self->receive_sdvr();
# 
#----------------------------------------------------------------------------#
 
sub receive_sdvr
{
  my($self) = shift;
  my($header,$body_ref) = $self->receive('#');
  if($header eq 'sdvr')
   {
    return (1,$body_ref);
   }
  elsif($header eq 'sdve')
   {return (${$body_ref}[0],undef);}
  else
   {
    $self->{'log'}->warn("Bad sdvr reply: H:$header B:@$body_ref");
    return (undef,undef);
   }
}

#============================================================================#
#
# Function:  receive_shvr
#
# This function receives either an shvr or shve message.
# 
# Return Value(s):
#  shvr, shve or undefined in case of error
# 
# Example of usage:
#  $return = $self->receive_shvr();
# 
#----------------------------------------------------------------------------#

sub receive_shvr
{
  my($self) = shift;
  my($header,$body_ref) = $self->receive('#');
  if($header eq 'shvr')
   {return (1,$body_ref);}
  elsif($header eq 'shve')
   {return (${$body_ref}[0],undef);}
  else
   {
    $self->{'log'}->warn("Bad shvr reply: H:$header B:@$body_ref");
    return undef;
   }
}

#============================================================================#
# 
# Function:  receive_gdvr
# 
# This function receives either an gdvr or gdve message.
# 
# Return Value(s):
#  gdvr, gdve or undefined in case of error
# 
# Example of usage:
#  $return = $self->receive_gdvr();
# 
#----------------------------------------------------------------------------# 
 
sub receive_gdvr
{
  my($self) = shift;
  my($header,$body_ref) = $self->receive('#');
  if($header eq 'gdvr')
   {return $body_ref;}
  elsif($header eq 'gdve')
   { 
    $self->{'log'}->warn("gdv message failed: ${$body_ref}[0]");
    return undef;
   }
  else
   {
    $self->{'log'}->warn("Bad gdvr reply: H:$header B:@$body_ref");
    return undef;
   }
}

#============================================================================#
#
# Function:  receive_ghvr
#
# This function receives either an ghvr or ghve message.
#
# Return Value(s):
#  ghvr, ghve or undefined in case of error
#
# Example of usage:
#  $return = $self->receive_ghvr();
#
#----------------------------------------------------------------------------#

sub receive_ghvr
{
  my($self) = shift;
  my($header,$body_ref) = $self->receive('#');
  if($header eq 'ghvr')
   {return $body_ref;}
  elsif($header eq 'ghve')
   {
    $self->{'log'}->warn("ghv message failed: ${$body_ref}[0]");
    return undef;
   }
  else
   {
    $self->{'log'}->warn("Bad ghvr reply: H:$header B:@$body_ref");
    return undef;
   }
}

#============================================================================#
#
# Function:  receive
# 
# This function receives any message.  The message is parsed for the 
# a delimeter.  That delimeter is ' ' by default, but to use the '#'
# for SM messages, pass in an argument (any argument).
# 
# Return Value(s):
#  The received header and message 
#
# Example of usage:
#  $self->receive();
# 
#----------------------------------------------------------------------------#
 
sub receive
{
  my($self) = shift;

  if($self->{'output_mode'} eq 'comm')
   {
    my($host,$port,$header,@array);
    until(defined $host && defined $port)
     {
      ($host,$port,$header,@array) = Comm::tcp_recv('blocking');
      unless($host && $port)
       {
        $self->{'log'}->warn("tcp_recv failed in receive");
       }
     }
    $self->{'log'}->warn("Interface received $header from $host:$port\n");
    return ($header, \@array);
   }
  else
   {
    no strict 'refs';
    my $handle = $self->{'handle'};
    my $buffer = undef;
    my $header;
    do
     {
      seek($handle,0,1);
      $buffer = <$handle>;
     }
    until $buffer;
    chomp $buffer;
    if($_[0])
     { 
      # Getting input from a software manager.  Need to use # convention.
      # Aaargh - this stinks.  We *must* rewrite the I/O stuff.
      @Hierarchy_interface::temp_array = split('#',$buffer); 
      my $return_address;
      if($self->{'output_mode'} eq 'file')
       {$return_address = shift @Hierarchy_interface::temp_array;}
      else
       {$return_address = ':'}
      my($host,$port) = split(':',$return_address); 
      $header = shift @Hierarchy_interface::temp_array; 
      $self->{'log'}->warn("Interface received $header ".
	join(' ', @Hierarchy_interface::temp_array)." from $host:$port\n");
     }
    else
     {
      # Getting return from OHS.  For historical reasons we have a different
      # convention here.  Oh this makes me cringe. 
      @Hierarchy_interface::temp_array = split(' ',$buffer); 
      $header = shift @Hierarchy_interface::temp_array; 
     }
    return ($header ,\@Hierarchy_interface::temp_array)
   }

}# end receive

#============================================================================#

# Function:  new_manager

# NOT EXTERNALLY CALLABLE (except maybe for testing and debugging).
 
# This function sets up a new software manager.  It should *only* be called
# during use with "output_mode" equal to "file".  In the full system, this
# task would be done by an appropriate module controller.

# Return Value(s):
#  None 

# Example of usage:
#  my ($man_port,$agg_port) =			 
#	$self->new_manager($man_host,$agg_host,$parent,$dept_name);

sub new_manager
{
  my($self,$man_host,$agg_host,$parent,$dept_name) = @_;
  my $man_port = int(rand(31000))+1000;
  my $agg_port = int(rand(31000))+1000;

  # Set control variables for new manager

  my($sdv_ref) = ['department', $dept_name, 
		'parent' ,$parent, 
		'our_aggregator', "$agg_host:$agg_port",
		'parent_aggregator', 0,
		'parent_manager', 0,
		]; 

  $self->send_to_SM($man_host,$man_port,'sdv',$sdv_ref);

  # Set up the root software manager on the appropriate file.
  my $command = <<'EndofStuff';
../../sm/sm "global::output_mode  = 'file'"
EndofStuff
  chomp $command;
  $command .= " \"global::my_address = \'$man_host:$man_port\'\" < pipes/$man_host:$man_port > $dept_name".'_sm_output &';
  system($command);

  my ($return) = $self->receive_sdvr();
  unless($return eq 1)
   {
    $self->{'log'}->warn("Sdv message failed in new_manager:$return");
    return undef;
   }

  return ($man_port,$agg_port);
}

#============================================================================#
#
# Function Name: parse_var_name
#
# Figure out whether a control variable is indexed or straight and split
# it into pieces if necessary.
#
# Inputs:
# $raw_variable - the raw control variable name.
#
#  Global:
#   None
#
# Return Value(s):
#   $variable - the name of the control variable
#   $index - the index if there is one, or undef if there isn't.
#
# Global variable modifications:
# None.  
#
# Example of usage:
#  ($variable,$index) = parse_var_name($raw_variable);
#
#----------------------------------------------------------------------------#

sub parse_var_name
{
  my($self, $variable) = @_;

  if($variable =~ /^\w[\w\:\-\,\.]*$/)
   {
    # The variable is a straight one.
    return ($variable);
   }
  elsif($variable =~ /^(\w[\w\:\-\,\.]*)\{(\w[\w\:\-\,\.]*)}$/)
   {
    # an indexed variable
    return ($1,$2);
   }
  else
   {
    # Trouble
    $self->{'log'}->warn("Bad variable name $variable\n");
    return undef;
   }
}
 
#============================================================================#
#
# Function Name: start_root_aggregator
#
# Start the root aggregator.
#
# Inputs:
# $root_name - the name of the root department.
# $agg_host - the host to run on.
# $agg_mc_port - the port of the appropriate module controller.
#
#  Global:
#   None
#
# Return Value(s):
#   $agg_port - port the aggregator is started on, zero if failure
#   $error - error message if there was a failure.
#
# Global variable modifications:
# None.  
#
# Example of usage:
#  ($variable,$index) = parse_var_name($raw_variable);
#
#----------------------------------------------------------------------------#

sub start_root_aggregator
{
  my($self,$root_name,$agg_host,$agg_mc_port) = @_;
  my($log) = $self->{'log'};
  my($agg_port);

  # We can start this up via a module controller, but still not through the 
  # regular channels because we don't have enough infrastructure yet.

  my @set = ('SET','module_controller','v.v',$root_name,'mc_command','START',
	'mod_module','engine', 'mod_department' ,$root_name,
        'mod_version','v.v',
	'parent_aggregator',0,
	'parent_manager',0,);

  my $mc_message = [$agg_host,$agg_mc_port,'s',\@set];
  my $send_status = send_messages([$mc_message],$log);
  return (0, "Send to MC failed in start_root_aggregator") 
						unless $send_status;
  my($recv_status,$replies) = receive_messages([$mc_message],$log);
  return (0, "Receive from MC failed in start_root_aggregator") 
						unless $recv_status;
  my($host1,$port1,$header1,$body_ref) = @{$replies->[0]};
  my %reply_hash = @{$body_ref}[4..$#$body_ref];
  $agg_port = $reply_hash{'listen_tcp'};
  unless($agg_port)
   {
    my $err = "No agg_port in start_root_aggregator: got "
						.join(' ',%reply_hash);
    $log->warn($err); return($agg_port,$err);
   }
  return ($agg_port,undef);
}

1; # end of package
