#!/pkg/bin/perl -w

#******************************************************************************
# GrIDS User Interface
# $Id: ui.pl,v 1.47 1998/02/17 19:22:31 rowe Exp $
#******************************************************************************


#******************************************************************************
# # The UI has three main sections:
#
#      Global constants, variables and procedures
#         - initialize and start-up the ui
#         - provide initial access control
#
#      Maintenance:
#         - Modify the hierarchy
#             + start a new hierarchy
#             + add hosts/departments
#             + move hosts/departments
#             + remove hosts/departments
#         - get info about a host or department
#         - change which department is being monitored
#         - view/edit/submit/kill rulesets running on an engine
#         - view and change the functioning of a data source
#         - change the visual layout of the hierarchy elements
#
#      Monitoring:
#         - receives and displays alerts from the engine
#         - allows queries to the engine and display of the returned reports
#         - viewing ruleset text for rulesets on the running on the engine
#
#******************************************************************************
require 5.002;

use English;
use Tk;
use Tk::DropSite qw(Sun);
use Tk::DragDrop qw(Sun);
require Tk::TextUndo;
require Tk::FileSelect;
#use FileSave;
require Tk::Menubar;
use Tk::ErrorDialog;
use Tk::Dialog;
use Hierarchy_interface;
use Clog;
use Comm;

use ui_prefs;

use strict "vars";


require 'check_rules.pl';
require 'parse_dot.pl';

srand;

#------------------------------------------------------------------------------------
#- before we do anything else, validate access via username/password 
#------------------------------------------------------------------------------------
my $welcome = "Welcome to GrIDS, ";
my ($username,$password) = &access_control; #- terminates ui if unsuccessful                
my $info_text = $welcome.$username;




#------------------------------------------------------------------------------------
# Global Variable Definitions
#------------------------------------------------------------------------------------
my ($proto, $host, $port, $header, @body); #- return parameters from msg_recv()
my ($level, $ruleset, $comment, $graph );  #- data returned from alerts

my $hier_frame;

my $monitor_frame;
my $monitor_left_frame;
my $monitor_right_frame;
my $monitor_grph_frame;
my $monitor_list_frame;
my $monitor_rule_frame;
my $monitor_hier_frame;

my $h_canvas;               #- canvas for displaying hierarchy items
my $h_scale = $ui_prefs::STD_HSCALE;  #- current scaling factor used in displaying the main hierarchy
  
my $hs_canvas;              #- canvas for displaying hierarchy items
my $hs_scale = $ui_prefs::STD_SSCALE; #- current scaling factor used in displaying the main hierarchy

my $g_canvas;               #- canvas for displaying graphs in main window pane 
my $g_scale = $ui_prefs::STD_GSCALE;  #- current scaling factor used in displaying the main graph
my $alert_comment = '';     #- holds alert comment for display with $g_canvas
my $alert_time = '';        #- holds time of current alert

my $ruleset_canvas;         #- widget reference for alert/ruleset canvas
my %rulesets = ();          #- hash of rulesets currently 
                            #-     being monitored by the UI

my $display_mode  = 'manager';
my $active_frame  = 'graph';

my @alert_log;          #- list of hashes of last $MAX_ALERTS alerts

#my $rsMonitorW;         #- toplevel window for displaying alerts/rulesets as they occur
my $alertListW;         #- toplevel window for displaying list of alerts received
my $rs_listbox;         #- listbox for active rulesets

#- Standard Font Definitions -----
my $SNfont    = '-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*';
my $SBfont    = '-*-Helvetica-Bold-R-Normal--*-120-*-*-*-*-*-*';
my $MNfont    = '-*-Helvetica-Medium-R-Normal--*-140-*-*-*-*-*-*';
my $MBfont    = '-*-Helvetica-Bold-R-Normal--*-140-*-*-*-*-*-*';
my $LNfont    = '-*-Helvetica-Medium-R-Normal--*-180-*-*-*-*-*-*';
my $LBfont    = '-*-Helvetica-Bold-R-Normal--*-180-*-*-*-*-*-*';


#- standard colors for backgrounds or icon/bitmap fills -----
my $paper1 = '#fff5e1';
my $paper2 = '#fffde9';

#- colors used to indicate ruleset alert levels
my %status_color = (    
		    3 => 'red',         #- maximum alert level
		    2 => 'yellow',     
		    1 => 'lightgray',   #- least significant alert level
		    0 => 'lightblue'    #- no alerts recieved [ !! change to in log? ]
		   );


my $rs_xsize   =  48;     #- width of a ruleset icon
my $rs_ysize   =  64;     #- height of a ruleset icon
my $rs_sep     =  40;     #- seperation between ruleset icons
my $rsw_xsize  = 250;     #- current width of the ruleset window
my $rsw_ysize  = 750;     #- current height of the ruleset window

my $rp_size    =  64;     #- height/width of a report icon
my $rp_sep     =  64;     #- seperation between report icons
my $rpw_xsize  = 200;     #- current width of the report window
my $rpw_ysize  = 300;     #- current height of the report window

my %qreport =  ();        #- global hash to hold result of query reports

my $hierarchy;            #- the heirarchy datastructure
my $hi;

#my $dept;                #- ... and associated values needed to 
my $log;                  #-  initialize it.


my %datasource = get_available_datasources();
foreach (sort keys %datasource) {
  my $p = $datasource{$_}; 
};



#------------------------------------------------------------------------------------
#- default host/ports
#------------------------------------------------------------------------------------
my ($ohs_host,$ohs_port) = split(/:/,$Hierarchy_interface::ohs_location);

my $root_sm_host = 'olympus';
my $root_sm_port = '11168';

my $engine_host;
my $engine_port;

my $ui_host;              
my $ui_port;

#- determine host the UI is running on -----
$ui_host = `/usr/bin/uname -n`;

# get fully qualified ui host name/addr
chomp($ui_host);
my($ourname,$ouraddr);
($ourname,$_,$_,$_,$ouraddr) = gethostbyname($ui_host);


#------------------------------------------------------------------------------------
#- hierarchy node info:
#-   holds info about screen location of hosts/depts, which dept is being monitored,
#-   which host/dept is selected on the display, etc.
#------------------------------------------------------------------------------------
my %h_node = ();
$h_node{'_root'} = 'null';




#- multi-listbox for alerts ---------- 
my ($lb0,$lb1,$lb2,$lb3,$lb4,$lb5 );  #- widget references for the alert list windo'}\n",
                                      #- defined globally because used by main loop

#- item information hash for dragable objects -----
my %iinfo = ();		
$iinfo{'areaX1'} = 0;
$iinfo{'areaY1'} = 0;
$iinfo{'areaX2'} = 0;
$iinfo{'areaY2'} = 0;
$iinfo{'restore_cmd'} = '';



#------------------------------------------------------------------------------------
#- setup the host_name and find an available port for communications
#------------------------------------------------------------------------------------

#- initialize the Comm package -----
# !!! change this to an init that will give me a port number !!! -----
$ui_port = int(rand(9000)+1000);
while(&Comm::init($ui_port,$ui_port) =~ /bad/) {
  $ui_port = int(rand(9000)+1000);
}
print "Alert monitor on host:$ui_host, port:$ui_port\n";




#------------------------------------------------------------------------------------
#- setup a log object for use with the OHS -----
#------------------------------------------------------------------------------------
$log = new Clog('UI','N/A',"ui_log");
$log->{'central_log'} = 0; # No central logging - presently broken.
my $save_log = $log;


#====================================================================================
#===== Create Main Window ===========================================================
#====================================================================================

#------------------------------------------------------------------------------------
#- Main window initialization 
#------------------------------------------------------------------------------------
my $MW = MainWindow->new;
my $MW_title = "GrIDS System Interface (v0.0a)";

$MW->title($MW_title);
$MW->iconname('GrIDS');
$MW->geometry( '800x600' );
#$MW->geometry( '1024x768' );
$MW->geometry( '+10+10' );
$MW->OnDestroy( [ \&exitUI ]);   #- make sure program terminates when main window quits

my $ppi = $MW->fpixels('1i');     #- the number of pixels per inch on the display

#- set-up the small gifs to use as icons -----
$MW->Photo(  'deptgif',     -file => "$ENV{'GRIDSPATH'}/bin/dept.icon.gif" );
$MW->Photo(  'hostgif',     -file => "$ENV{'GRIDSPATH'}/bin/host.icon.gif" );
$MW->Photo(  'eye',         -file => "$ENV{'GRIDSPATH'}/bin/focus.gif" );
  
$MW->Bitmap('monitor',
	    -file     => "$ENV{'GRIDSPATH'}/bin/monitor.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/monitor.xbm",
	   );

$MW->Bitmap('manager',
	    -file     => "$ENV{'GRIDSPATH'}/bin/tool.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/tool.xbm",
	   );

$MW->Bitmap('zoom',
	    -file     => "$ENV{'GRIDSPATH'}/bin/zoom.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/zoom.xbm",
	   );

$MW->Bitmap('find',
	    -file     => "$ENV{'GRIDSPATH'}/bin/search.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/search.xbm",
	   );

$MW->Bitmap('rotate',
	    -file     => "$ENV{'GRIDSPATH'}/bin/rotate.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/rotate.xbm",
	   );

$MW->Bitmap('print',
	    -file     => "$ENV{'GRIDSPATH'}/bin/lineprinter.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/lineprinter.xbm",
	   );

$MW->Bitmap('bugs',
	    -file     => "$ENV{'GRIDSPATH'}/bin/bugs.xbm",
	    -maskfile => "$ENV{'GRIDSPATH'}/bin/bugs.xbm",
	   );


#-------------------------------------------------------------------------------
#- Construct the main window menubar ---
#-------------------------------------------------------------------------------
my $menubar = $MW->Frame;
$menubar->pack( -side => 'top', -fill => 'x' );


#-------------------------------------------------------------------------------
# Create and pack the button frame
#-------------------------------------------------------------------------------
my $btn_frame = $MW->Frame(-relief      => 'sunken', 
			   -borderwidth => '2',
			   -width       => '0.50i',
			   -height      => '5.75i',
			   -background  => 'gray'
			  );
$btn_frame->pack( -side => 'left', -fill => 'y', -expand => 0 );


my $mode_btn_frame = $btn_frame->Frame(-relief      => 'ridge', 
				       -borderwidth => '2',
				       -background  => 'gray'
				      );
$mode_btn_frame->pack( -side => 'top', -pady => '1m',-fill => 'none', -expand => 0 );


my $func_btn_frame = $btn_frame->Frame(-relief      => 'ridge', 
				       -borderwidth => '2',
				       -background  => 'gray'
				      );
$func_btn_frame->pack( -side => 'top', -pady=> '1m',-fill => 'none', -expand => 0 );


my $xtra_btn_frame = $btn_frame->Frame(-relief      => 'ridge', 
				       -borderwidth => '2',
				       -background  => 'gray'
				      );
$xtra_btn_frame->pack( -side => 'top', -pady=> '1m',-fill => 'none', -expand => 0 );

my $bottom_btn_frame = $btn_frame->Frame(-relief      => 'ridge', 
					 -borderwidth => '2',
					 -background  => 'gray'
					);
$bottom_btn_frame->pack( -side => 'bottom', -pady=> '1m',-fill => 'none', -expand => 0 );


my $btn_frame_manager = add_check_button($mode_btn_frame, 'manager', 'manager');
$btn_frame_manager->configure(-command=>[\&change_to_manager_mode]);

my $btn_frame_monitor = add_check_button($mode_btn_frame, 'monitor', 'monitor');
$btn_frame_monitor->configure(-command=>[\&change_to_monitor_mode]);

my $btn_frame_zoom    = add_button($func_btn_frame, 'zoom');
$btn_frame_zoom->bind('<1>',[\&zoom,+1]);
$btn_frame_zoom->bind('<2>',[\&zoom,-1]);

my $btn_frame_find    = add_button($func_btn_frame, 'find');
$btn_frame_find->configure(-command=>[\&find]);

my $btn_frame_rotate  = add_button($func_btn_frame, 'rotate');
$btn_frame_rotate->configure(-command=>[\&rotate]);

my $btn_frame_print   = add_button($xtra_btn_frame, 'print');
$btn_frame_print->configure(-command=>[\&hardcopy]);

my $btn_frame_bugs    = add_button($bottom_btn_frame, 'bugs', 'bottom' );
$btn_frame_bugs->configure(-command=>[\&bug_report]);


#-------------------------------------------------------------------------------
# Create and pack the information frame
#-------------------------------------------------------------------------------
my $info_frame = $MW->Frame(-relief      => 'ridge', 
			    -borderwidth => '2',
			    -width       => '8.50i',
			    -height      => '0.25i',
			    -background  => 'white'
			   );
$info_frame->pack( -side => 'bottom', -fill => 'x', -expand => 0 );

my $info_label = $info_frame->Label(-font=>$SNfont,
				    -anchor=>'sw',
				    -bg=>'white',
				    -textvariable=>\$info_text
				   );
$info_label->pack( -side => 'left', -anchor => 'w' );


#-------------------------------------------------------------------------------
# Create and pack the hierarchy canvas frame
#-------------------------------------------------------------------------------
$hier_frame = $MW->Frame(-relief      => 'sunken',
			    -bd          => '2',
			    -width       => '8.50i',
			    -height      => '5.75i',
			    -background  => 'gray'
			   );
#$hier_frame->pack( -side => 'bottom', -fill => 'both', -expand => 'yes' );


#-------------------------------------------------------------------------------
# Create the canvas
#-------------------------------------------------------------------------------
$h_canvas = $hier_frame->Canvas(-background => $paper1, 
				   -scrollregion => [ '0c', '0c', '80c', '60c' ],
				   -width =>  '25c',
				   -height => '20c',
				   -relief => 'sunken'
				  );



#-------------------------------------------------------------------------------
#- Construct the monitor window frame elements
#-------------------------------------------------------------------------------
$monitor_frame = $MW->Frame(-relief      => 'sunken',
			    -bd          => '2',
			    -width       => '8.50i',
			    -height      => '5.75i',
			    -background  => 'gray'
			   );

$monitor_left_frame = $monitor_frame->Frame(-relief => 'flat',
					    -width       => '3.00i',
					    -height      => '7.75i',
					    -background  => 'gray'
					   );
$monitor_right_frame = $monitor_frame->Frame(-relief => 'flat',
					     -width       => '5.50i',
					     -height      => '7.75i',
					     -background  => 'gray'
					    );
$monitor_rule_frame = $monitor_left_frame->Frame(-relief => 'ridge',
						 -bd          => '2',
						 -width       => $rsw_xsize,
						 -height      => $rsw_ysize,
						 -background  => 'gray'
						);
$monitor_hier_frame = $monitor_left_frame->Frame(-relief => 'ridge',
						 -bd          => '2',
						 -width       => $rsw_xsize,
						 -height      => '3.00i',
						 -background  => 'gray'
						);
$monitor_grph_frame = $monitor_right_frame->Frame(-relief => 'ridge',
						  -bd          => '2',
						  -width       => '5.50i',
						  -height      => '3.00i',
						  -background  => 'gray'
						 );
$monitor_list_frame = $monitor_right_frame->Frame(-relief => 'ridge',
						  -bd          => '2',
						  -width       => '5.50i',
						  -height      => '3.00i',
						  -background  => 'gray'
						 );
$monitor_left_frame->pack(  -side => 'left', -fill => 'y',    -expand => 0 );
$monitor_right_frame->pack( -side => 'left', -fill => 'both', -expand => 1 );

$monitor_rule_frame->pack(  -side => 'top',  -fill => 'both', -expand => 1);
$monitor_hier_frame->pack(  -side => 'top',  -fill => 'both', -expand => 0, -anchor=>'sw' );
$monitor_grph_frame->pack(  -side => 'top',  -fill => 'both', -expand => 1);
$monitor_list_frame->pack(  -side => 'top',  -fill => 'both', -expand => 0 );



#-------------------------------------------------------------------------------
#- Construct the 'File' pull-down menu ---
#-------------------------------------------------------------------------------
my $mb_file = $menubar->Menubutton(-text => ' File ', 
				   -underline => 1
				  );
$mb_file->separator;
$mb_file->command(-label     => 'Quit',
		  -underline => 0,
		  -accelerator => '^Q',
		  -command   => [ 'destroy', $MW ]		  
		 );
$mb_file->pack( -side => 'left' );
		
  
#-------------------------------------------------------------------------------
#====================================================================================
#====================================================================================
#====================================================================================
#====================================================================================
#- Construct the 'Edit' pull-down menu ---
#-------------------------------------------------------------------------------
my $mb_edit = $menubar->Menubutton(
        -text => ' Edit ', 
	-underline => 1
);
$mb_edit->command(
	-label       => 'Undo',
	-underline   => 0,
	-accelerator => '^Z',
        -command     => [ \&Undo , 'Undo' ]		  
);
$mb_edit->command(
	-label     => 'Redo',
	-underline => 0,
	-accelerator => '^Y',
        -command   => [ \&Redo , 'Redo' ]		  
);
$mb_edit->separator;
$mb_edit->command(
	-label     => 'New Host',
	-underline => 4,
	-accelerator => '^H',
        -command   => [ \&makeNode, ( $h_canvas, '100', '100', 'host' ) ]		  
);
$mb_edit->command(
	-label     => 'New Department',
	-underline => 4,
	-accelerator => '^D',
        -command   => [ \&makeNode, ( $h_canvas, '100', '100', 'dept' ) ]		  
);
$mb_edit->pack( -side => 'left' );



#-------------------------------------------------------------------------------
#- Construct the 'Tools' pull-down menu ---
#-------------------------------------------------------------------------------
my $mb_tool = $menubar->Menubutton(
        -text => ' Tools ', 
	-underline => 1
);
$mb_tool->command(
	-label     => 'Find...',
	-underline => 0,
	-accelerator => '^F',
        -command   => [ \&find, '' ]		  
);
$mb_tool->separator;
$mb_tool->command(
	-label     => 'Screwdriver',
	-underline => 0,
        -command   => [ \&unimplemented ]		  
);
$mb_tool->command(
	-label     => 'Sledge Hammer',
	-underline => 7,
        -command   => [ \&unimplemented ]		  
);
$mb_tool->pack( -side => 'left' );


#-------------------------------------------------------------------------------
#- Construct the 'View' pull-down menu ---
#-------------------------------------------------------------------------------
my $mb_view = $menubar->Menubutton(
        -text => ' View ', 
	-underline => 1 
);
$mb_view->command(
	-label     => 'Zoom In',
	-underline => 5,
	-accelerator => '^1',
        -command   => [ \&zoom, -1 ]		  
);
$mb_view->command(
	-label     => 'Zoom Out',
	-underline => 5,
	-accelerator => '^2',
        -command   => [ \&zoom, +1 ]		  
);
$mb_view->command(
	-label     => 'Rotate',
	-underline => 0,
	-accelerator => '^R',
        -command   => [ \&unimplemented ]		  
);
$mb_view->separator;
$mb_view->command(
	-label     => 'Reorganize Display',
	-underline => 2,
	-accelerator => '^O',
        -command   => [ \&reorganize ]
);
$mb_view->separator;
$mb_view->command(
	-label     => 'User Account Managment',
	-underline => 0,
	-accelerator => '^U',
        -command   => [ \&account_mgmt ]
);
$mb_view->pack( -side => 'left' );



#-------------------------------------------------------------------------------
#- Construct the 'Help' pull-down menu ---
#-------------------------------------------------------------------------------
my $mb_help = $menubar->Menubutton(
        -text => 'Help', 
	-underline => 0 
);
$mb_help->command(
	-label     => 'Help...',
	-underline => 0,
	-accelerator => '^?',
        -command   => [ \&showHelp, 'Help' ]		  
);
$mb_help->separator;
$mb_help->command(
	-label     => 'About GrIDS...',
	-underline => 0,
        -command   => [ \&aboutGrids , 'GrIDS' ]		  
);
$mb_help->pack( -side => 'right' );


#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
$hier_frame->pack( -side => 'bottom', -fill => 'both', -expand => 'yes' );


#-------------------------------------------------------------------------------
# Setup the scroll bars -------------------------------------------
#-------------------------------------------------------------------------------
my $yscroll = $h_canvas->Scrollbar( -command => [ 'yview', $h_canvas ] );
my $xscroll = $h_canvas->Scrollbar( -command => [ 'xview', $h_canvas ], 
                                 -orient => 'horiz' );
$h_canvas->configure( -yscrollcommand=>[ 'set', $yscroll ]);
$h_canvas->configure( -xscrollcommand=>[ 'set', $xscroll ]);
$yscroll->pack( -side => 'right',  -fill => 'y' );
$xscroll->pack( -side => 'bottom', -fill => 'x' );

$h_canvas->pack( -expand => 'yes', -fill => 'both' );


$h_canvas->bind( 'node', '<Any-Enter>' => [\&items_enter, \%iinfo]);
$h_canvas->bind( 'node', '<Any-Leave>' => [\&items_leave, \%iinfo]);





#------------------------------------------------------------------------------------
#- Binding for accelerator key equivalents ----
#------------------------------------------------------------------------------------
$h_canvas->Tk::bind('<Any-Control-h>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  $MW->bell;
  $MW->bell;
  create_host($can, $e->x, $e->y);  
});


$h_canvas->Tk::bind('<Any-Control-d>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  $MW->bell;
  $MW->bell;
  create_dept($can, $e->x, $e->y);  
});




#------------------------------------------------------------------------------------
#- Binding for object movement and key-focus selection ----
#------------------------------------------------------------------------------------
$h_canvas->Tk::bind('<Button-1>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  items_start_drag($can, $e->x, $e->y, \%iinfo);
});

$h_canvas->Tk::bind('<B1-Motion>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  items_drag($can, $e->x, $e->y, \%iinfo);
});

$h_canvas->Tk::bind('<B1-ButtonRelease>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  items_drop($can, $e->x, $e->y, \%iinfo);
});



#------------------------------------------------------------------------------------
#- Binding for department monitoring changes  ----
#------------------------------------------------------------------------------------
$h_canvas->Tk::bind('<2>' => sub {
  my ($can) = @ARG;
  my $e = $can->XEvent;
  my ($x, $y, $W, $K, $A ) = ( $e->x, $e->y, $e->W, $e->K, $e->A );
  my $name = get_item_name($W);
  change_focus($W, $name); 
  draw_hierarchy($W); 
});



#------------------------------------------------------------------------------------
#- Binding for context based pop-up menus ----
#------------------------------------------------------------------------------------
$h_canvas->Tk::bind('<3>' => [\&do_hierarchy_popup, @ARG] );



#------------------------------------------------------------------------------------
# Create widgets for manipulating files
#------------------------------------------------------------------------------------
#my $fileSave    = $MW->FileSave();
  
my $fileSelect  = $MW->Component(FileSelect => 'fs',
				 -width => 25, 
				 -height => 8,
				 'accept' => sub 
				 { my $file = shift ; 
				   return 0 if (-s $file && !(stat(_))[12]);
				   return (-r $file) && (-T $file);  
				 },
				 Name    => 'fs', 
				 -filter => '*');

#############


#====================================================================================
#===== Startup GrIDS ================================================================
#====================================================================================
#- display the hierarchy on the main canvas -----
#-
#-   the user may modify the heirarchy and control variables at this point
#-
#-
#-   require the user to select a department to monitor before alowing monitoring to
#-     be active. Selecting the department brings up the alert/ruleset window and the
#-     alert log.
#-


&change_to_manager_mode;
print "*** 1 ***\n";
&init_monitor;
print "*** 2 ***\n";
&do_startup($ohs_host,$ohs_port,1);
print "*** 3 ***\n";


#------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------
# Main event loop -------
#------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------
my ($ruleset,$comment,$graph,$level);
my $time;

my $done = 'false';
while ($done eq 'false') {

# !! Just for DEBUGging -----  
  exit if $save_log ne $log;

  #- see if any X-events need processing, but do not block -----
  $time = time;
  DoOneEvent(1);    
  $time = time-$time;
  print ">>>>>>>>>>>>>>>> UI: DoOneEvent wait = $time\n" if $time > 3;

  undef $proto; #- make sure there is nothing in $proto
  
  #- see if any comm messages have arrived, but do not block -----
  $time = time;
  ($proto,$host,$port,$header,@body) = &Comm::mesg_recv('nonblocking');
  $time = time-$time;
  print ">>>>>>>>>>>>>>>> UI: mesg_recv wait = $time\n" if $time > 3;

  #- if a message has been recieved, process it
  if ( defined $proto && defined $ruleset_canvas ) {
    
    $time = localtime(time);
    
    #- see if we have a query_response -----
    if ($header eq 'qr') {
      #print "*** we have a query response in the main loop!!\n";
      $log->warn("*** received query response from $host($port)\n");
      show_report($body[0]);
    }
    #- see if we have an alert -----
    elsif ($header eq 'alt') {
      ($ruleset,$comment,$graph,$level) = @body;
      #print "*** received alert from $host($port) at $time  ( r:$ruleset, c:$comment, l:$level )\n";
      $log->warn("*** received alert from $host($port) at $time -- ruleset:$ruleset, comment:$comment, level:$level\n");
      
      #- if we have reached the maximum number of alerts to retain, dump
      #-     the oldest alert in the list before adding the new alert
      if (($lb0->size > $ui_prefs::MAX_ALERTS)) {
	$lb0->delete('end');
	$lb1->delete('end');
	$lb2->delete('end');
	$lb3->delete('end');
#	$lb4->delete('end');
	$lb5->delete('end');
      } 
      
      #- add the alert to the list widget ----
      $lb0->insert(0, $level);   $lb0->see(0);
      $lb1->insert(0, $comment); $lb1->see(0);
      $lb2->insert(0, $time);    $lb2->see(0);
      $lb3->insert(0, $host);    $lb3->see(0);
#      $lb4->insert(0, $port);    $lb4->see(0);
      $lb5->insert(0, $ruleset); $lb5->see(0);
      
      
      #- log the alert to the alert list -----
      log_alert($level, $comment, $time, $host, $port, $ruleset, $graph);
      
      
      #- log alerts to a file
      if ( $ui_prefs::FILE_LOGGING eq 'true' ) {
	open (FID, ">>alert.log");
	print FID ($level, $comment, $time, $host, $port, $ruleset, $graph);
	print FID "\n\n";
	close FID;
      };
      
      #- display the alert in the monitor graph frame -----
      $alert_comment = $comment; 
      $alert_time    = $time;
      draw_graph($graph,$g_canvas,1); #- draw the graph with scaling ON -----

      #- update the ruleset icon to indicate its current alert status ---
      if ($rulesets{$ruleset}{'name'} ne '') {
	$rulesets{ $ruleset }{'status'} = $level;
	show_rulesets( $ruleset_canvas, %rulesets );
      }
      else {
	print "*** ignoring alert from unknown ruleset\n";
      }
    }
    else {
      print "***** UI: bad header in mesg_rcv: proto=$proto, header=$header\n";
      $MW->bell;
    }
    
  } #- if a message recieved

} #- main event loop
&exitUI;  #- cleanup after UI, notify GrIDS of its departure and exit




#====================================================================================
#== Subroutine Definitions ==========================================================
#====================================================================================





#====================================================================================
# Cleanly shut down the UI
#   - let engine know to stop sending alerts to this UI
#   -
#====================================================================================
sub exitUI {
  print "EXITING GRIDS UI\n";
  send_shutup_to_engine($ui_host,$ui_port,$h_node{'_focus'}) if defined $h_node{'_focus'};
  exit 1;
}


sub change_to_manager_mode {
  print "***** Changing to MANAGER MODE\n";
  $monitor_frame->packForget;
  $MW->title($MW_title." - Management Screen");
  $hier_frame->pack( -side => 'bottom', -fill => 'both', -expand => 'yes' );
}

sub change_to_monitor_mode {
  print "***** Changing to MONITOR MODE\n";
  $hier_frame->packForget;
  $MW->title($MW_title." - Monitoring Screen");
  $monitor_frame->pack( -side => 'bottom', -fill => 'both', -expand => 'yes' ); 

  draw_hierarchy($hs_canvas);
  $hs_canvas->scale('all', 0, 0, $hs_scale, $hs_scale);

}



#- uses state of global variables to determine what to zoom -----
sub zoom {
  my ($dir) = @_;
  print "***** sub ZOOM called: zooming ",($dir > 0) ? "OUT" : "IN","!\n";

  my $factor;

  $dir = +1 unless defined $dir; #- make sure this is set to something

  $factor = ($dir < 0) ? 0.75 : 1.50;

  if (defined $h_canvas && $display_mode eq 'manager' && ($h_scale>$ui_prefs::MIN_SCALE && $factor<1 || $h_scale<$ui_prefs::MAX_SCALE && $factor>1) ) {
    $h_scale *= $factor;
    cleanup_hierarchy();
    draw_hierarchy($h_canvas);
  };
 
  if (defined $hs_canvas && $display_mode eq 'monitor' && ($hs_scale>$ui_prefs::MIN_SCALE*4 && $factor<1 || $hs_scale<$ui_prefs::MAX_SCALE*4 && $factor>1) ) {
    $hs_scale *= $factor;
    cleanup_hierarchy();
    draw_hierarchy($hs_canvas);
    $hs_canvas->scale('all', 0, 0, $hs_scale, $hs_scale);
  };

#  if (defined $g_canvas && $display_mode eq 'monitor' && $active_frame eq 'graph') {
#    $g_scale *= $factor if $g_scale > $ui_prefs::MIN_SCALE && $g_scale < $ui_prefs::MAX_SCALE;
#    show_graph($the_graph, $g_canvas);
#  };

}


#====================================================================================
#====================================================================================
sub find {
  print "***** sub FIND called\n";
}


#====================================================================================
#====================================================================================
sub rotate {
  print "***** sub ROTATE called\n";
}


#====================================================================================
#====================================================================================
sub hardcopy {
  print "***** sub HARDCOPY called\n";
  plot( $h_canvas, 'hierachy' ) if (defined $h_canvas && $display_mode eq 'manager');
  plot( $h_canvas, 'hierachy' ) if (defined $h_canvas && $display_mode eq 'monitor' && $active_frame eq 'hierarchy');
  plot( $g_canvas, 'graph' )    if (defined $g_canvas && $display_mode eq 'monitor' && $active_frame eq 'graph');
}


#====================================================================================
#====================================================================================
sub bug_report {
   `$ENV{'GRIDSPATH'}/common/bug_report_script`;

#  `cat $ENV{'GRIDSPATH'}/common/bug_rep_header bug_rep_temp | mail templets\@cs `;
#  `rm bug_rep_temp`;

}



#====================================================================================
#====================================================================================
sub add_button {
  my ( $frame, $img, $pos ) = @_;

  $pos = 'top' unless (defined $pos);

  my $btn = $frame->Button(-image => $img );
  $btn->pack(-side => $pos, -fill => 'both', -expand => 0 );
  
  return $btn;
}



#====================================================================================
#====================================================================================
sub add_check_button {
  my($frame, $img_up, $img_dn ) = @_;
  
  my $btn = $frame->Radiobutton(-image        => $img_up,
				-selectimage  => $img_dn,
				-indicatoron  => 0,
				-variable     => \$display_mode,
				-value        => $img_dn,
			       );
  $btn->pack(-side => 'top', -fill => 'both', -expand => 0 );
  $btn->configure(-selectcolor => $frame->cget(-background));

  return $btn;
}



#====================================================================================
#====================================================================================
sub init_monitor {

    #------------------------------------------------------------------------------------
    # Define the main window and its widgets
    #        - listbox, scrollbar
    #------------------------------------------------------------------------------------
    $monitor_list_frame->Label(-text=>'Alert History',-justify=>'left',-bg=>'lightgray')->pack(-side=>'top',-fill=>'x',-expand=>0);

    my $title_frame   = $monitor_list_frame->Frame(-bg=>'white',-bd=>2,-relief=>'ridge');
    my $data_frame    = $monitor_list_frame->Frame();
#    my $sbr           = $data_frame->Scrollbar;
    my $sbr           = $monitor_list_frame->Scrollbar;

    $sbr->pack        (-side => 'right', -fill => 'y',    -expand=>0);  
    $title_frame->pack(-side => 'top',   -fill => 'x',    -expand=>0);
    
    $lb5 = create_listbox($monitor_list_frame, $sbr, $title_frame, 12, 'ruleset' );


#    $data_frame->pack (-side => 'top',   -fill => 'both', -expand=>1);
    $lb0 = create_listbox($monitor_list_frame, $sbr, $title_frame,  3, 'lvl' );
    $lb1 = create_listbox($monitor_list_frame, $sbr, $title_frame, 32, 'comment' );
    $lb2 = create_listbox($monitor_list_frame, $sbr, $title_frame, 24, 'date' );
    $lb3 = create_listbox($monitor_list_frame, $sbr, $title_frame, 20, 'aggregator' );
#    $lb4 = create_listbox($monitor_list_frame, $sbr, $title_frame,  5, 'port' );
    
#    $sbr->configure(-command => [\&scrollcols, $sbr, [$lb0,$lb1,$lb2,$lb3,$lb5]]); #- !! NOTE: $lb4 removed from list

  
    #-------------------------------------------------------------------------------
    # Setup the window to display the rulesets  ------------------------------------
    #-------------------------------------------------------------------------------
    
    #- ruleset_canvas is defined globally so that main event loop can use it -----
    $monitor_rule_frame->Label(-text=>'Active Rulesets',-justify=>'left',-bg=>'lightgray',-width=>'32')->pack(-side=>'top',-fill=>'x',-expand=>0);

    $ruleset_canvas = $monitor_rule_frame->Canvas(-background   => 'white',
				  -scrollregion => [ '0', '0',  $rsw_xsize,  $rsw_ysize ],
				  -width        => $rsw_xsize,
				  -height       => $rsw_ysize,
				  -relief       => 'sunken',
				  -bd           => '2'
				 );
 

    $ruleset_canvas->pack(-side=>'bottom',-expand=>'yes',-fill=>'both');
    
    $ruleset_canvas->bind('ruleset', '<1>' => sub {
      my $rsn =  get_item_name($ruleset_canvas);
      show_alert( $rsn );
    });
    
    $ruleset_canvas->bind('ruleset', '<2>' => sub { 
      my $rsn =  get_item_name($ruleset_canvas);
      do_query_dialog( $rsn ); # popup query window
    });
    
    $ruleset_canvas->bind('ruleset', '<3>' => sub { do_ruleset_popup($ruleset_canvas); } );
    
    #-------------------------------------------------------------------------------
    # Setup the scroll bars for ruleset window ---------------------------------------
    #-------------------------------------------------------------------------------
    my $rsc_yscroll = $ruleset_canvas->Scrollbar( -command => [ 'yview', $ruleset_canvas ] );
    my $rsc_xscroll = $ruleset_canvas->Scrollbar( -command => [ 'xview', $ruleset_canvas ], -orient => 'horiz' );
    $ruleset_canvas->configure( -yscrollcommand=>[ 'set', $rsc_yscroll ]);
    $ruleset_canvas->configure( -xscrollcommand=>[ 'set', $rsc_xscroll ]);
    $rsc_yscroll->pack( -side => 'right',  -fill => 'y' );
    $rsc_xscroll->pack( -side => 'bottom', -fill => 'x' );


    $monitor_grph_frame->Label(-text=>'Graph Report',-justify=>'left',-bg=>'lightgray')->pack(-side=>'top',-fill=>'x',-expand=>0);

    my $comment_frame = $monitor_grph_frame->Frame(-height=>'0.5i',-width=>'6i',-bg=>'white',-bd=>2,-relief=>'sunken');
    $comment_frame->pack(-side=>'top',-fill=>'x', -expand=>0);
    $comment_frame->Label(-textvariable=>\$alert_comment,-justify=>'left', -bg=>'white')->pack(-side=>'left', -fill=>'x',-expand=>0);
    $comment_frame->Label(-textvariable=>\$alert_time,   -justify=>'right',-bg=>'white')->pack(-side=>'right',-fill=>'x',-expand=>0);
    
    $g_canvas = $monitor_grph_frame->Canvas(-background   => $paper1,
#			    -scrollregion => [ '0i', '0i', '10i', '16i' ],
			    -width        => '400',
#			    -height       => '500',
			    -relief       => 'sunken',
			    -bd           => '2'
			   );
    $g_canvas->pack(-side=>'top',-fill=>'both', -expand=>1);
    
    $monitor_hier_frame->Label(-text=>'Hierarchy Summary',-justify=>'left',-bg=>'lightgray')->pack(-side=>'top',-fill=>'x',-expand=>0);
    $hs_canvas = $monitor_hier_frame->Canvas(-background   => $paper1,
#					     -scrollregion => [ '0i', '0i', '10i', '16i' ],
					     -width        => $rsw_xsize,
#					     -height       => '200',
					     -relief       => 'sunken',
					     -bd           => '2'
			   );
    $hs_canvas->pack(-side=>'top',-fill=>'both', -expand=>0);
    
}; # init_monitor



sub monitor_startup {
  my ($deptname) = @_;
  my ($rslt);

  if ( $rslt= get_rulesets($deptname) ) {
    %rulesets = %$rslt;
    cleanup_rulesets($ruleset_canvas);               #- make ruleset icons aranged neatly in window
    show_rulesets( $ruleset_canvas, %rulesets );     #- display the ruleset icons
  };
  send_hello_to_engine($ui_host,$ui_port,$deptname);
}; # monitor startup





#==============================================================================
#
# Display a dialog that will give some help
#
#==============================================================================
sub showHelp {
  my $helpW = $MW->Toplevel();
  
  $helpW->configure(-title => "GrIDS Help",
		    -height => '5c', 
		    -width  => '9c',
		    -bg     => 'lightgray' );

  my $helpMsg = $helpW->Label(-font => $SNfont,
			      -wraplength => '5i',
			      -justify    => 'left' ,
			      -text       => "The GrIDS UI does need help. Can you?\n",
			     );
  
  $helpMsg->pack();
}; #- showHelp {}






#==============================================================================
#
# Display a text edit window for editing the ruleset
#
#==============================================================================
sub edit_ruleset {
  my ($dept,$ruleset) = @_;

  my $editW     = $MW->Toplevel(-title => "Edit Window",
				-height => '5c', 
				-width  => '8c',
				-bg     => 'white' );

  $editW->optionAdd('*TextUndo.Background' => 'white');
  my $ruleText = $editW->Scrolled('TextUndo', -wrap => 'none' );
 
#- !! change this to get ruleset text from OHS rather than disk !! ----- 
  if (defined $ruleset && $ruleset) {
    my ($ok,$rslt) = get_ruletext($dept,$ruleset);
    unless ($ok) {
      Dialog_OK("Error: getting ruleset text for $ruleset from $dept",$rslt);
      $log->warn("Error: getting ruleset text for $ruleset from $dept: $rslt");
    }
    else {
      $ruleText->insert('end', $rslt);
    };
  };
  
  my $dragNDrop = $ruleText->DragDrop(-event => '<Meta-B1-Motion>');
  $dragNDrop->configure(-startcommand => 
			sub
			{
			  return 1 unless (eval { $ruleText->tagNextrange(sel => '1.0','end')});
			  $dragNDrop->configure(-text => $ruleText->get('sel.first','sel.last')); 
			});
  
  $ruleText->DropSite(-motioncommand => 
		      sub 
		      { my ($x,$y) = @_;
			$ruleText->markSet(insert => "\@$x,$y");
		      },
		      -dropcommand => 
		      sub 
		      { my ($seln,$x,$y) = @_;
			$ruleText->markSet(insert => "\@$x,$y");
			$ruleText->insert(insert => $ruleText->SelectionGet(-selection => $seln));
		      }
		     );

  my $EW_mb = $editW->Menubar;
  $EW_mb->Menubutton(-text => '~File', -menuitems => 
		     [[Button => '~Open',
		       -command => sub { 
			 my $file = $fileSelect->Show(-popover => $MW);
			 $ruleText->Load($file) if (defined $file);
			 print "-- NO FILE --\n" if (!defined $file);
			 $ruleText->pack(-expand => 1, -fill => 'both' );
		       }
		       ]]);
  
  $EW_mb->Menubutton(-text => '~File', -menuitems => 
		     [['Button' => '~Save', 
		       -command =>  sub { $ruleText->Save;
					  #$fileSave->Show(); 
					},
		       -accelerator => '^S'],
		      ['Button' => 'Save as...', -command => [$ruleText, 'save_as' ]],
		      ['Button' => '~Submit', -command =>sub { submit_ruleset($dept,$ruleText->get('1.0','end')); }],
		      ['Button' => '~Quit', -command => [ 'destroy',$editW ], -accelerator => '^Q']
		      ]);
  $EW_mb->Menubutton(-text => '~Edit', -menuitems => 
		     [['Button' => '~Undo',  
		       -command => [ $ruleText, 'undo' ], 
		       -accelerator => '^Z'],
		      ['Button' => '~Cut',   
		       -command => [ $ruleText, 'cut',   '1.0', 'end' ], 
		       -accelerator => '^X'],
		      ['Button' => '~Copy',  
		       -command => [ $ruleText, 'copy',  '1.0', 'end' ], 
		       -accelerator => '^C'],
		      ['Button' => '~Paste', 
		       -command => [ $ruleText, 'paste', '1.0', 'end' ], 
		       -accelerator => '^V'],
		      ['Button' => '~Clear', -command => [ $ruleText, 'delete','1.0', 'end' ]],
		      ['Button' => '~Select All', -command => [ $ruleText, 'select','1.0', 'end' ]]
		      ]);
  
  
  $ruleText->pack(-expand => 1, -fill => 'both' );

}; #- edit_ruleset {}





#==============================================================================
# returns the the tag associated with the 'current' (ie under the cursor) 
#   widget on the specified canvas. 
#
# NOTE: this assumes the name is the first tag encountered
#==============================================================================
sub get_item_name {
  my $c = shift;
  my $item = $c->find('withtag','current');
  my @taglist = $c->gettags($item);
  my $name;
  foreach (@taglist) {
    next if ($_ eq 'current');
    $name = $_;
    last;
  }
  return $name;
}



#==============================================================================
# distributes ruleset icons neatly across the display
#
#
#==============================================================================
sub cleanup_rulesets {
  my $c = shift;

  my $x = 0;
  my $y = 0;
  my $num_rulesets = keys %rulesets;

#- make sure the window is at least large enough to show one ruleset --
  $rsw_xsize = max(2*($rs_xsize+$rs_sep)+$rs_sep+20,$c->width );
  $rsw_ysize = max(($num_rulesets/2)*($rs_ysize+$rs_sep)+$rs_sep+20,$c->height );
  
  my $x_limit = int($rsw_xsize / ($rs_xsize+$rs_sep) );

  foreach (sort( keys( %rulesets ))) {
    if ( $x >= $x_limit ) {
      $x = 0;
      $y++;
    }

    $rulesets{ $_ }{'xpos'} = $x*($rs_xsize+$rs_sep)+$rs_sep;
    $rulesets{ $_ }{'ypos'} = $y*($rs_ysize+$rs_sep)+$rs_sep;
   
    $x++;
    
  };

  $c->configure( -scrollregion=> [ '0', '0', $rsw_xsize, ($y+1)*($rs_ysize+$rs_sep)+$rs_sep ]);

}; #- cleanup_rulesets {}





#==============================================================================
# draw all ruleset icons on the canvas
#
#
#==============================================================================
sub show_rulesets {
  my ($c, %rsl) = @_; #- input parameters: 
                      #    $c is the canvas widget,
                      #    %rsl is the current list of rulesets/ruleset info

  my ($left,$top,$right,$bottom);   
  my $label;
  my $color;
  my $color_id;
  
  $c->delete('all');  #- clear the canvas of all previously drawn objects

  #- display an labeled icon for each of the active rulesets -----
  foreach (keys(%rsl)) {

    $label = $rsl{$_}{'name'};
    $left  = $rsl{$_}{'xpos'};
    $top   = $rsl{$_}{'ypos'};

    $color_id = $rsl{$_}{'status'};
    $color = $status_color{ $color_id };

    $c->create('bitmap',
	       $left, $top, 
	       -bitmap     => "\@$ENV{'GRIDSPATH'}/bin/rule.icon",
	       -background => $color,
	       -anchor     => 'nw',
	       -tags       => [$label, 'ruleset']
	      );

    $c->create('text', ($left+$rs_xsize/2, $top+$rs_ysize+15 ),
	       -anchor => 'center',
	       -text   => $label, 
	       -justify=> 'center', 
	       -font   => $SNfont, 
	       -tags   => [$label, 'ruleset']
	      );
  };

}; #- show_rulesets{}





#==============================================================================
#
#
#
#==============================================================================
sub do_ruleset_popup {
  my ($rsW) = @_;
  my $rsn =  get_item_name($ruleset_canvas);
  my $thisDept = $h_node{'_focus'};  #- !! may not be valid if monitoring has been turned off
                                     #-      and will be null

  my $rsPopM = $MW->Menu(-tearoff=>0,
			 -menuitems => 
			 [[Button => 'Display Alert Graph', 
			   -command => sub { show_alert( $rsn ); }, 
			   -accelerator => '^G' ],
			  [Button => 'Query Engine',        
			   -command => sub { do_query_dialog($rsn); }, 
			   -accelerator => '^R' ],
                          [Button => 'View Ruleset Text',   
			   -command => sub { edit_ruleset( $h_node{'_focus'}, $rsn ); }, 
			   -accelerator => '^T' ],
			  '------',
			  [Button => 'Remove Ruleset',     
			   -command => sub { remove_ruleset( $h_node{'_focus'}, $rsn ); }]
			  ]);
  $rsPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor' );

} #- do_ruleset_popup {}





#==============================================================================
# creates the popup menu for a report icon
#
#
#==============================================================================
sub do_report_popup {
  my ($rpW, %graph_hash) = @_;

  my $rsPopM = $MW->Menu(-tearoff=>0,
			 -menuitems => 
			 [ '------',
			  [Button => 'Display Graph', 
			   -command =>  sub { my $rpn = get_item_name($rpW);
					      my $idx = $graph_hash{$rpn};
					      	show_report_graph( $qreport{$idx}, $rpn);
					    }, 
			   -accelerator => '^G' ],
			  [Button => 'Display Info',        
			   -command => sub { &show_info( get_item_name($rpW)); }, 
			   -accelerator => '^I' ],
			  '------'
			  ]);
  $rsPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor');

} #- do_report_popup {}





#==============================================================================
# creates the popup menu for an icon on the hierarchy. This menu is context
#   sensitive to the type of object the menu is popped up on
#
#==============================================================================
sub do_hierarchy_popup {
  my ($widget) = @_;

  my $e = $widget->XEvent;
  my ($x, $y, $W, $K, $A ) = ( $e->x, $e->y, $e->W, $e->K, $e->A );
  my $name;

#- convert pixels to inches --
  $x /= $ppi;
  $y /= $ppi;

  #- check to see if click is on canvas ---
  return if ($W != $h_canvas );

  my $item = $h_canvas->find( 'withtag', 'current' );
  if ( defined $item ) {
    $name = get_item_name($W);
  };

  if ( !defined($item) || $name eq '_edge'  ) { # -- then on the canvas --
    my $dPopM = $MW->Menu(-tearoff=>0,
			  -menuitems => 
			  [ '------',
			   [Button => 'Add Dept...', 
			    -command => sub { create_dept($W,$x,$y); }, 
			    -accelerator => '^D' ],
			   [Button => 'Add Host...',        
			    -command => sub { create_host($W,$x,$y); }, 
			    -accelerator => '^H' ],
			   '------',
			   [Button => 'Find...',        
			    -command => [ \&find_hierarchy_item, $W ], 
			    -accelerator => '^F' ],
			   '------',
			   [Button => 'Reorganize Window',        
			    -command => [ \&reorganize ],
			    -accelerator => '^O' ],
			   '------',
			   [Button => 'User Account Managment',        
			    -command => sub { account_mgmt(''); },
			    -accelerator => '^U' ],
			   '------',
			   [Button => 'Quit',        
			    -command =>  [ 'destroy', $MW ], 
			    -accelerator => '^Q' ],
			   '------'
			   ]);
    $dPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor');
    return 1;
  };
    
  my $type = $h_node{$name}->{'type'};
  shift_hilight($W,$name);

  if ($type eq 'host') {
    my $dPopM = $MW->Menu(-tearoff=>0,
			  -menuitems => 
			  [ '------',
			   [Button => 'Inspect', 
			    -command => [ \&host_inspector, $name ], 
			    -accelerator => '^I' ],
			   [Button => 'Remove Host',        
			    -command => sub { delete_host($W, $name); }, 
			    -accelerator => '^K' ],
			   '------'
			   ]);
    $dPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor');
  }
  elsif ($type eq 'dept') {

    #- don't allow deletes if the dept has children -----
    my $state = 'normal';
    $state = 'disabled' if (defined @{$h_node{$name}->{'links'}});

    my $monitor_cmnd = 'Start Monitoring';
    $monitor_cmnd = 'Stop Monitoring' if (defined $h_node{'_focus'} && ($h_node{'_focus'} eq $name));

    my $dPopM = $MW->Menu(-tearoff=>0,
			  -menuitems => 
			  [ '------',
			   [Button => 'Inspect', 
			    -command => [ \&dept_inspector, $name ], 
			    -accelerator => '^I' ],
			   [Button => $monitor_cmnd,        
			    -command => sub { if ($monitor_cmnd eq 'Start Monitoring') {
			                        change_focus($W, $name); 
					      }
					      else {
						change_focus($W,undef); 
					      }; 
					      draw_hierarchy($W); 
					    },
			    -accelerator => '^M' ],
			   [Button => 'List Rulesets',        
			    -command => [ \&list_rulesets, $name ], 
			    -accelerator => '^L' ],
			   '------',
			   [Button => 'Add Dept...', 
			    -command => sub { create_dept($W, $x,$y, $name); }, 
			    -accelerator => '^D' ],
			   [Button => 'Add Host...',        
			    -command => sub { create_host($W, $x,$y, $name); }, 
			    -accelerator => '^H' ],
			   '------',
			   [Button => 'Remove Dept',        
			    -state => $state,
			    -command => sub { delete_dept($W, $name); }, 
			    -accelerator => '^K' ],
			   '------',
			   [Button => 'User Account Managment',        
			    -command => sub { account_mgmt($name); },
			    -accelerator => '^U' ],
			   '------'
			   ]);

    $dPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor');
  }
  else {
    $log->warn("*** ERROR: unknown node type ($name:$type) ***\n");
  }

} # do_hierarchy_popup





#==============================================================================
#
#
#
#==============================================================================
sub do_module_inspector_popup {
  my ( $dept, $host, $ds_type, $ds_ver ) = @_;

  my $miPopM = $MW->Menu(-tearoff=>0,
			 -menuitems => 
			 [[Button => 'Examine Control Variables', 
			   -command =>  sub {
			     module_inspector( $dept, $host, $ds_type, $ds_ver );
			   }],
			  [Button => 'Display Some Other Info',        
			   -command => sub { return; } ]
			  ]);
  $miPopM->Popup(-popanchor  => 'n', -overanchor => 'n', -popover => 'cursor');
};





#==============================================================================
# determines the most recent alert for the specified ruleset if any, and
#   and has its graph, comment, and ruleset name  displayed in its own window. 
#
#==============================================================================
sub show_alert {
  my ($rsn) = @_;

  #- retrieve the most recent alert from the alert log for the specified ruleset 
  my %alert = get_recent_alert( $rsn );

  #- inform user if no alerts were present -----
  unless (defined %alert) {
    Dialog_OK('No Alerts','No alerts were available for ruleset "'.$rsn.'".');
    return;
  };

  my $comment = $alert{'comment'};
  my $graph   = $alert{'graph'};

  show_alert_graph( $graph, $comment, $rsn );  
} #- show_alert{}





#==============================================================================
# retruns the most recent alert for the specified ruleset
#
#
#==============================================================================
sub get_recent_alert {
  my ($rsn) = @_;
  
  my $i;
  for ( $i=$#alert_log; $i>=0; $i-- ) {
    return %{$alert_log[$i]} if $alert_log[$i]{'ruleset'} eq $rsn;
  };
  return undef;   #- no alert found for specified ruleset
}




#==============================================================================
# returns all alerts in the alert-list for the specified ruleset
#
#
#==============================================================================
sub get_alerts {
  my ($rsn) = @_;
  my @alert_set;
  
  my $i;
  for ( $i=0; $i < @alert_log; $i++ ) {
    push( @alert_set, $alert_log[$i] ) if $alert_log[$i]{'ruleset'} eq $rsn;
  };
  return @alert_set;
}





#==============================================================================
# adds an alert to the alert-list 
#
#
#==============================================================================
sub log_alert {
  my ($level, $comment, $time, $host, $port, $ruleset, $graph) = @_;
  my $log_entry = {};
  
  $log_entry->{'level'}   = $level;
  $log_entry->{'comment'} = $comment;
  $log_entry->{'time'}    = $time;
  $log_entry->{'host'}    = $host;
  $log_entry->{'port'}    = $port;
  $log_entry->{'ruleset'} = $ruleset;
  $log_entry->{'graph'}   = $graph;
  
  shift(@alert_log) if (@alert_log > $ui_prefs::MAX_ALERTS);
  push(@alert_log, $log_entry);
};




#==============================================================================
#==============================================================================
sub show_report_graph {
  show_graph(@_,0);
}


#==============================================================================
#==============================================================================
sub show_alert_graph {
  show_graph(@_,1);
}


#==============================================================================
# opens a window and displays the graph, comment, and ruleset name specified.
#   A popup list giving access to the alert history for the
#   ruleset is also available.
#
#==============================================================================
sub show_graph {
  my ($graph,$comment,$rsn,$is_alert) = @_;  
  
  my $option_choice = 'no graphs available';
  my $option;
  my @option_list;
  my @alert_list;
  my $can;

  my ($xsize,$ysize);
  my $max_graph_x = 1000;
  my $max_graph_y = 600;

  #-------------------------------------------------------------------------------
  # Setup the window to display the graph  ---------------------------------------
  #-------------------------------------------------------------------------------
 
   my $GW = $MW->Toplevel(-title  => $rsn.' Ruleset Report Graph',
			 -relief => 'flat',
			 -bg     => 'gray' );
  
  if ( $is_alert ) {
    $GW->configure(-title=>$rsn.' Ruleset Alert Graph');
    my $popup_frame   = $GW->Frame(-height=>'0.5i',-width=>'6i',-bg=>'ltgray');
    $popup_frame->pack(-side=>'top',-expand=>'no',-fill=>'x');
    my $popmenu = $popup_frame->Optionmenu(-textvariable => \$option_choice, 
					   -width => 32,
					   -bg => 'white',			 
					   -relief => 'raised',
					   -command => sub { $option_choice = shift;
							     ($option) = $option_choice =~ /^Graph\s+\#(\d+).*$/;
							     $comment = $alert_list[$option-1]->{'comment'};
							     $graph = $alert_list[$option-1]->{'graph'};
							     ($xsize,$ysize) = draw_graph( $graph, $can ); #- returns size of graph in inches 
							     $xsize *= $ppi;  $xsize = int($xsize); #- convert inches to pixels
							     $ysize *= $ppi;  $ysize = int($ysize); #- convert inches to pixels
							     $GW->geometry( min($xsize,$max_graph_x).'x'.min($ysize,$max_graph_y).'+20+20' );
							   }
					  );

    #- generate the elements to built the alert history list popup -----
    my $i = 0;
    my $op;
    @alert_list = get_alerts($rsn);
    foreach ( @alert_list ) {
      $op = 'Graph #'.++$i.':'.$$_{'time'};
      push(@option_list, $op );
    };
    $option_choice = $option_list[0];
    
    $popmenu->configure(-options =>[@option_list]);
    
    $popmenu->pack(-side=>'top', -expand=>'no', -fill=>'none');
  };

  my $comment_frame = $GW->Frame(-height=>'0.5i',-width=>'6i',-bg=>'white',-bd=>2,-relief=>'sunken');
  $comment_frame->pack(-side=>'top', -expand=>'no',-fill=>'x');

  $can = $GW->Canvas(-background   => $paper1,
		     -scrollregion => [ '0i', '0i', '10i', '16i' ],
		     -width        => '600',
		     -height       => '720',
		     -relief       => 'sunken',
		     -bd           => '2'
		    );

  my $print_button = $comment_frame->Button(-text   =>'Print Graph',
					    -bd     =>'2',
					    -relief =>'raised',
					    -command=>sub { plot($can,'graph'); } #- !! this should really be some rational name
					   );
  $print_button->pack(-side=>'right',-expand=>'no',-fill=>'none');
  my $comment_label = $comment_frame->Label(-font=>$SBfont,-anchor=>'sw',-bg=>'white',-textvariable=>\$comment);
  $comment_label->pack(-side=>'left',-expand=>'no',-fill=>'x');

  $can->pack(-side=>'bottom',-expand=>'yes',-fill=>'both');

  #-------------------------------------------------------------------------------
  # Setup the scroll bars for graph window ---------------------------------------
  #-------------------------------------------------------------------------------
  my $yscroll = $can->Scrollbar( -command => [ 'yview', $can ] );
  my $xscroll = $can->Scrollbar( -command => [ 'xview', $can ], -orient => 'horiz' );
  $can->configure( -yscrollcommand=>[ 'set', $yscroll ]);
  $can->configure( -xscrollcommand=>[ 'set', $xscroll ]);
  $yscroll->pack( -side => 'right',  -fill => 'y');
  $xscroll->pack( -side => 'bottom', -fill => 'x');

  ($xsize,$ysize) = draw_graph( $graph, $can ); #- returns size of graph in inches 

  $xsize *= $ppi;  $xsize = int($xsize);  #- convert inches to pixels
  $ysize *= $ppi;  $ysize = int($ysize);  #- convert inches to pixels

  $GW->geometry( min($xsize,$max_graph_x).'x'.min($ysize,$max_graph_y).'+20+20' );

} #- show_graph {} 





#==============================================================================
# submit a query to the engine
#
#
#==============================================================================
sub do_query_dialog {
  my $rsn = shift; 
  
  my $done = 'false';
  my $query_text = "ruleset eq \"$rsn\";";

  my $graph_query = $MW->Toplevel;
  $graph_query->withdraw;
 
  $graph_query->resizable(0,0);         #- make the window not resizable -----
  $graph_query->title('Graph Query');

  my $button_frame = $graph_query->Frame(-relief=>'ridge');
  my $prompt_frame = $graph_query->Frame(-relief=>'flat');
  my $entry_frame  = $graph_query->Frame(-relief=>'flat');
  my $qr_label     = $prompt_frame->Label(-text=>'Enter Query: ');
  my $qr_text      = $entry_frame->Scrolled('Text',(-width=>40,-height=>10 ));

  $qr_text->insert('end',$query_text);
  
  my $query_btn = $button_frame->Button(-text   =>'Query',
					-bd     =>'2',
					-relief =>'raised',
					-command=>sub {
					  $query_text = $qr_text->get('1.0','end');
					  if (&validate_query($query_text) == 0 ) {
					    $graph_query->destroy;
					    query_engine($query_text);
					    #do_query($query_text);
					  }
					}
				       );

  my $cancel_btn = $button_frame->Button(-text=>'Cancel', 
					 -relief=>'raised', 
					 -command=>sub { $graph_query->destroy; }
					);
  $button_frame -> pack(-side=>'bottom',-expand=> 'yes', -fill=>   'x');
  $prompt_frame -> pack(-side=>'left',  -expand=> 'no',  -fill=>   'y');
  $qr_label     -> pack(-side=>'top',   -expand=> 'yes', -fill  => 'both');

  $entry_frame  -> pack(-side=>'left',  -expand=> 'yes',  -fill=>   'both');
  $qr_text      -> pack(-side=>'top',   -expand=> 'yes',  -fill=>   'both');

  $query_btn   -> pack(-side=>'left', -padx=>'1m', -pady=>'1m',   -expand=> 'yes');
  $cancel_btn  -> pack(-side=>'left', -padx=>'1m', -pady=>'1m',   -expand=> 'yes');

  $graph_query->Popup(-popanchor  => 'c', -overanchor => 'c', -popover => $MW );  
  $qr_text->focus;                    
}; #- do_query_dialog {}



#==============================================================================
# make sure the query is syntactically correct
#
#
#==============================================================================
sub validate_query {
  my $query = shift;

  my @errors = check_query($query);

  if ( defined @errors ) {
    #- create error dialog                  
    my $D = $MW->Dialog(-title=>'Graph Language Error',
			-text=> '',
			-default_button=>'Ok',
			-buttons => ['Ok']
		       );
    $D->configure( -text => "The query was not valid. Try Again.\n\n".join('\n',@errors) );
    $D->Show;
    return 1; #- errors occured
  }
  else {
    return 0; #- no errors 
  }
} #- validate_query{}



#==============================================================================
# open a window and display the  report in iconic format
#
#
#==============================================================================
sub show_report {
  my ($report) = @_;
  my %graph_hash = ();

 print "in show REPORT:$report\n\n";

  if ($report eq '') {   #- no report returned 
    $log->warn("NO REPORTS FOUND\n");
    print "NO REPORTS FOUND\n";
   return; 
  };

  #- build a list with members being each subgraph in the report -----
  my @graphs = &parse_dot::get_graphs($report);
 
  if ( @graphs < 1 ) {   #- no report returned 
    my $D = $MW->Dialog(-title=>'No Report Graphs',
			-text=> "The query returned no graphs. Please respecify your query.",
			-default_button=>'Ok',
			-buttons => ['Ok']
		       );
    return if ( $D->Show eq 'Ok');
    $log->warn("NO GRAPHS IN REPORT ::: $report\n");
    return; 
  };

  if ( $#graphs > 20 ) {
    my $D = $MW->Dialog(-title=>'Large Report',
			-text=> "The query returned over 20 graphs. Press Continue to display all the first 20 graphs, or Retry to refine your query.",
			-default_button=>'Retry',
			-buttons => ['Retry','Continue']
		       );
    return if ( $D->Show eq 'Retry');
  };

  

  my $reportW = $MW->Toplevel(-title  => 'Reports',
			      -relief => 'flat',
			      -bg     => 'gray' );
  $reportW->withdraw;

  my $report_canvas = $reportW->Canvas(-background   => 'white',
				       -scrollregion => [ '0i', '0i', '10i', '16i' ],
				       -width        => '600',
				       -height       => '720',
				       -relief       => 'sunken',
				       -bd           => '2'
				      );
  $report_canvas->pack(-side=>'bottom',-expand=>'yes',-fill=>'both');
  
  $report_canvas->bind('report', '<1>' => sub { 
                                                my $rpn = get_item_name($report_canvas);
						my $idx = $graph_hash{$rpn};
						show_report_graph( $qreport{$idx}, $rpn); # =~ s/\n/  /m );
					      } 
		      );
  
  $report_canvas->bind('report', '<3>' => sub { do_report_popup( $report_canvas, \%graph_hash) });
  
  my $x = 0;
  my $y = 0;
  my $i = 0;
  my $d = $rp_size+$rp_sep;

  foreach (@graphs) {
    my $thisGraph = $_;
    my ($name,$nodesref,$edgesref,@attrs) = &parse_dot::graph_parts($_);
    my ($nnodes, $nedges, $ruleset, $stime);
    foreach (@attrs) {
      my ($attrname, $val) = &parse_dot::split_attr($_);
      $nnodes = $val if $attrname eq 'nnodes';
      $nedges = $val if $attrname eq 'nedges';
      $ruleset= $val if $attrname eq 'ruleset';
      $stime  = $val if $attrname eq 'stime';
    }

    $qreport{$i} = $thisGraph;
    $i++;
    my $label = "$ruleset\#$i\n$name (n:$nnodes, e:$nedges)";
    $graph_hash{$label} = $i-1;

    #- create the icon for the report -----   
    $report_canvas->create('bitmap',
			   $x*$d+$rp_sep, $y*$d+$rp_sep, 
			   -bitmap     => "\@$ENV{'GRIDSPATH'}/bin/graph.icon",
			   -background => $paper2,
			   -anchor     => 'nw',
			   -tags       => [$label, 'report']  #- make the tag the label
			  );

    
    #- create the label widget for the report icon -----
    $report_canvas->create('text', ( $x*$d+$rp_sep+$rp_size/2, ($y+1)*$d+25 ),
			   -anchor => 'center',
			   -text   => $label, 
			   -justify=> 'center', 
			   -font   => $SNfont, 
			   -tags   => [$label, 'report']  #- make the tag the label
			  );

    #- calculate the screen position for the next icon on the canvas -----
    $x++;
    if ( $x*$d + $rp_sep > $rpw_xsize ) {
      $x = 0;
      $y++; #- move to next line 
    }

  }; #- end: foreach graph

#- resize window to intelligently contain the report icons -----
  $reportW->geometry($rpw_xsize+$d.'x'.min($rpw_ysize,($y+2)*$d+25));
  $reportW->geometry('+50+50');
  $reportW->deiconify;

}; #- show_report {}





#==============================================================================
# handle communications of query/response from the engine. 
#
# !! Should be somewhat move to the main event loop
#
#==============================================================================
sub query_engine {
  my ($q) = @_;
  
  $log->warn("SENDING QUERY TO $engine_host:$engine_port(query = $q)");
  
  $log->warn("Error in tcp_send\n") unless (&Comm::tcp_send( $engine_host, $engine_port, 'q', $q ));

}; #- query_engine {}





#==============================================================================
# draw a graph on the canvas
#
# graph is passed to draw_graph in dot format
#==============================================================================
sub draw_graph {
  my ($t,$c,$gr) = @_;   #- input parameters: 
                         #    $t is the graph spec. in dot, 
                         #    $c is the canvas widget

  my $scale  = $g_scale; #- set local copy of scaling factor from global value
  my $margin = 0.25;     #- margin around graph (in inches )
  my $cmd;               #- constructed pTk string command for eval
  my $text_in;           #- the text specifying the graph(in dot format)
  my $text_out;          #- the plaintext specification created by passing $text_in to dot
  my $o_xsize = $c->width/$ppi;
  my $o_ysize = $c->height/$ppi;
  my $scale_factor = 1;
  my %local_nodes;


#print "CANVAS SIZE = ($o_xsize,$o_ysize)\n";

  my ($graph_xsize,$graph_ysize);

  $c->delete('all');  #- clear the canvas of all previously drawn objects

  #- preprocess the input text  -----  
  $text_in = $t;

  #- insert dot commands to force the graph into the expected default mode -----
  my $fs = int(48/$scale);
  $text_in =~ s/{/{graph [rankdir=LR];node [shape=ellipse, color=white, fontname=Helvetica, fontsize=$fs];/;
  $text_in =~ s/^subgraph/digraph/;

  #->> Note: the following uses a shell trick -----
  $text_out = `dot -Tplain<<EOF\n$text_in\nEOF\n`;

#-----------------------------------------------------------------------------	
#- convert each dot plaintext statement into its pTk equivalent and create 
#-   the object on the canvas.
#-   		   
#-----------------------------------------------------------------------------	
  foreach (split('\n',$text_out)) {


    /^node/ && do {
      my ($node, $name, $x, $y, $xsize, $ysize, $label, $style, $shape, $color) = split;

      #- save position of the node so we can determine where to put the arrow-head -----
      #-  This makes the assumptions that all nodes are created before the edges,
      #-  and that no two nodes have the same name
      $local_nodes{$name}{'x'} = $x;
      $local_nodes{$name}{'y'} = $y;

    #- scale and shift the anchor position for the node -----
      $x = ($x/$scale)+$margin; 
      $y = ($y/$scale)+$margin;

    #- calculate left,top and right,bottom from center + length,width -----
      $xsize /= 2*$scale; $ysize /= 2*$scale;
      my $x1 = ($x - $xsize).'i';
      my $y1 = ($y - $ysize).'i';
      my $x2 = ($x + $xsize).'i';
      my $y2 = ($y + $ysize).'i';

    #- convert dot shapenames to Tk shapenames -----
      $shape = 'rectangle' if $shape eq 'box';
      $shape = 'oval' if $shape eq 'ellipse';
      

    #- set the fill color for the node -----
      $color = 'white' if $color eq 'black';  #- make sure we can see something


    #- create the node -----
      $c->create($shape, ($x1, $y1, $x2, $y2),
		 -outline => 'black',
		 -fill    => $color,
		 -tags    => ['node', $name]
		);
      my $screen_name = $name;
      $screen_name =~ s/\"//g;
      $screen_name =~ s/$ui_prefs::IMPLICIT_SUFFIX$//;
      $c->create('text', ($x.'i', $y.'i'), 
		 -anchor  => 'center',
		 -text    => $screen_name, 
		 -justify => 'center', 
		 -font    => $SNfont, 
		 -tags    => 'node'
		);
      next;
    };
			

  #- process 'edge' statements -----
    /^edge/ && do {
      my ($edge, $head, $tail, $n, $rest) = split(' ',$_,5);
      
    #- calculate the number of list elements used for curve control points,
    #-  and extract all remaining parameters from the dot plaintext command   
      $n = $n * 2 - 1;
      my @control_pts = split(' ',$rest);

    #- extract line attributes (just after actual control points) -----
      my ($style, $color) = @control_pts[$n+1..$n+2];

      #- get x,y for start of line and determine whether $head or $tail is closer -----
      my $d_head = ($local_nodes{$head}{'x'}-$control_pts[0])**2+($local_nodes{$head}{'y'}-$control_pts[1])**2;    
      my $d_tail = ($local_nodes{$tail}{'x'}-$control_pts[0])**2+($local_nodes{$tail}{'y'}-$control_pts[1])**2;      


    #- scale and adjust length of last segment of edge so that it reaches the dest. node -----
      @control_pts = map(($_/$scale)+$margin, @control_pts[0 .. $n] );
      if ($d_head > $d_tail) {
	@control_pts[0]    += (@control_pts[0  ] <=> @control_pts[2]) * (0.1/$scale);
	@control_pts[1]    += (@control_pts[1  ] <=> @control_pts[3]) * (0.1/$scale);
      }
      else {
	@control_pts[$n-1] += (@control_pts[$n-1] <=> @control_pts[$n-3]) * (0.1/$scale);
	@control_pts[$n]   += (@control_pts[$n  ] <=> @control_pts[$n-2]) * (0.1/$scale);
      }	

#      @control_pts[$n]    += (@control_pts[$n  ] <=> @control_pts[$n-2])*(0.1/$scale);
#      @control_pts[$n-1]  += (@control_pts[$n-1] <=> @control_pts[$n-3])*(0.1/$scale);

    #- convert styles to line widths -----
      my $width;
      if ($style eq 'solid') {
	$width = 1;
      } 
      elsif ($style eq 'bold') {
	$width = 8;   
      }
      else {
	($width) = $style =~ /^[wW](\d+)$/;
      };

      if ( !defined $width || $width <1 || $width > 128) {
	$width = 1;
	$log->warn("ERROR: Bad width specification \'$style\' for $head --> $tail\n");
      };

      #- convert color to hex format -----
      $color =~ s/^[xX]/\#/;

    #- create the edge -----
      $c->create('line', map("$_".'i', @control_pts[0 .. $n]),
		 -smooth  => 'on',
		 -arrow   => ($d_head > $d_tail) ? 'first' : 'last',
		 -tags    => 'edge',
		 -fill    => $color,
		 -width   => $width
		);

      next;
    };


  #- process 'graph' statements -----
    /^graph/ && do {
      my ($graph, $sf, $xsize, $ysize) = split;
      
    #- scale and shift the length and width of the graph ----
      $xsize = (($xsize/$scale)+$margin*2)+1.0 unless $xsize == 0; 
      $ysize = (($ysize/$scale)+$margin*2)+1.0 unless $ysize == 0; 
    
    #- set scroll region to include just the graph, and increase size of window to show the graph. ----  
    #-   ... that is unless scaling is enabled. Then the canvas should stay the same size... always!
      $c->configure(-scrollregion => [ '0i', '0i', $xsize.'i', $ysize.'i' ] );
      unless ( defined $gr && $gr ) {
	$c->configure(-width  => $xsize.'i', -height => $ysize.'i') ;

	#- update the variables to pass back to caller -----
	$graph_xsize = $xsize;     
	$graph_ysize = $ysize;
      }
      else {
	$scale_factor = min(1,min($o_xsize/$xsize, $o_ysize/$ysize)) if ( defined $gr && $gr );

	#- update the variables to pass back to caller (original canvas size) -----
	$graph_xsize = $o_xsize;     
	$graph_ysize = $o_ysize;
	print "Setting scale_factor to $scale_factor: ",$o_xsize/$xsize,", ", $o_ysize/$ysize," \n";
      };

      next;
    };

    last if /^stop/;
  };
		 
  #- resize graph if it will not fit in alert graph frame ---
  $c->scale('all', 0, 0, $scale_factor, $scale_factor) if ( defined $gr && $gr );

  return($graph_xsize,$graph_ysize);

} #draw_graph()





#==============================================================================
# display the hierarchy on the hierarchy canvas
#
#
#==============================================================================
sub draw_hierarchy {
  my $c = shift;      #- input parameters: 
                      #    $c is the canvas widget


  $c->delete('all');  #- clear the canvas of all previously drawn objects


  #- set scroll region to include just the graph, and increase size of window to show the graph -----  
#  $c->configure(-scrollregion => [ '0i', '0i', $h_node{'_width'}.'i', $h_node{'_height'}.'i' ],
#		-width  => $h_node{'_width'}.'i',
#		-height => $h_node{'_height'}.'i'
#	       );

  #- draw each host/dept and edges to its children -----
  foreach ( keys %h_node ) {
    next if /^_/;

    showNode( $c,  $_ );
    my $parent = $_;
    foreach ( @{$h_node{$parent}->{'links'}}) {
      showEdge($c,$parent,$_);
    }
  };
 
  #- show which dept is being monitored (if any) -----
  showFocus( $c );

  #- show which host/dept is selected -----
  # !! this may be silly because of context sensitive popups, but it does give a
  #      visual cue back to the item being manipulated
  showHilight( $c );
  
} #- draw_hierarchy()





#==============================================================================
# uses dot to calculate a neat distribution of elements on the hierarchy canvas
#
#
#==============================================================================
sub cleanup_hierarchy {
  #- no parameters passed -----

  my $t = &hierarchy2dot;  #- $t is the graph spec. in dot, 

  my $margin = 0.25;       #- margin around graph (in inches )
  my $text_in;             #- the text specifying the graph(in dot format)
  my $text_out;            #- the plaintext specification created by passing $text_in to dot
  my $xscale = $h_scale;
  my $yscale = $h_scale/$ui_prefs::H_XY_RATIO;


  #- preprocess the input text  -----  
  $text_in = $t;

  #- insert dot commands to force the graph into the expected default mode -----
  $text_in =~ s/{/{graph [rankdir=TB];node [shape=ellipse, fontname=Helvetica, fontsize=24, height=4, width=0.25];/; 
  $text_in =~ s/^subgraph/digraph/;

  #->> Note: the following uses a shell trick -----
  $text_out = `dot -Tplain<<EOF\n$text_in\nEOF\n`;

  $text_out =~ s/\"//g;

#-----------------------------------------------------------------------------	
#- use the dot node statements to determine the position of the hosts and
#-   departments on the canvas. also, get the overall size.
#-----------------------------------------------------------------------------	
		   
  foreach (split('\n',$text_out)) {
     
    #- process 'node' statements -----
    /^node/ && do {
      my ($node, $name, $x, $y, $xsize, $ysize, $label, $style, $shape, $color) = split;

      #- scale and shift the anchor position for the node -----
      $h_node{$name}{'x'} = ($x/$xscale)+$margin; 
      $h_node{$name}{'y'} = $h_node{'_height'} - ($y/$yscale); #+$margin;
      
      #- set the type of node -----
      $h_node{$name}{'type'} = 'host' if $shape eq 'box';
      $h_node{$name}{'type'} = 'dept' if $shape eq 'ellipse';

    };
    
    
    #- just ignore the edge statements -----
    /^edge/ && do {
      next;
    };
    
    
    #- process 'graph' statements -----
    /^graph/ && do {
      my ($graph, $sf, $xsize, $ysize) = split;
      
      #- scale and shift the length and width of the graph ----
      $h_node{'_width'}  = (($xsize/$xscale)+$margin*2); 
      $h_node{'_height'} = (($ysize/$yscale)+$margin*2); 
      next;
    };
    
    last if /^stop/;
  };
		   
#  draw_hierarchy($h_canvas);
   
} #- cleanup_hierarchy();






	
#==============================================================================
#==============================================================================
sub reorganize { 
  cleanup_hierarchy();
  draw_hierarchy($h_canvas) if $display_mode eq 'manager';
  draw_hierarchy($hs_canvas) if $display_mode eq 'monitor';
};




		 
#==============================================================================
# change which department is being monitored
#
#==============================================================================
sub change_focus {
  my ($can, $dept ) = @_;

  $MW->configure(-cursor, 'watch');

  #- cleanup from the prior department -----
  $alert_comment = '';
  $alert_time = '';
  $g_canvas->delete('all');
  foreach ( $lb0, $lb1, $lb2, $lb3, $lb5 ) { #-lb4 removed from list
    $_->delete(0, 'end');
  }
  #- send message to engine informing it to stop sending alerts to the UI ----
  send_shutup_to_engine($ui_host,$ui_port,$h_node{'_focus'}) if defined $h_node{'_focus'};

  if (defined $dept) {
    $engine_host = $hi->{$dept}->{'aggregator_host'};
    $engine_port = $hi->{$dept}->{'aggregator_port'};
    monitor_startup($dept);  #- startup monitoring on named department 
    $h_node{'_focus'} = $dept;
  }
  else {
    #stop_monitoring();
    undef $h_node{'_focus'};
  };

  $MW->configure(-cursor, 'arrow');
} #- change_focus




#==============================================================================
# display a host/dept icon on the canvas
#
#
#==============================================================================
sub showNode {
    my ($can, $name ) = @_;

    my $xsize = $ui_prefs::STD_HIER_OBJ_XSIZE;  #- in inches
    my $ysize = $ui_prefs::STD_HIER_OBJ_YSIZE;  #- in inches

    my $type =  $h_node{$name}->{'type'};
    my $x    =  $h_node{$name}->{'x'};
    my $y    =  $h_node{$name}->{'y'};

    if ( $ui_prefs::ICON_MODE ) {
      my $img; 
      if ( defined $h_node{$name}->{'type'} ) {
	$img  =  'deptgif' if $h_node{$name}->{'type'} eq 'dept';
	$img  =  'hostgif' if $h_node{$name}->{'type'} eq 'host';
      };
      
      $can->create('image', $x.'i', $y.'i', 
		   -image => $img,
		   -anchor => 'center',
		   -tags => [$name, 'node'] 
		  );
    }
    else {
      my $shape = 'oval';
      my $color = $ui_prefs::STD_HIER_OBJ_COLOR;

    #- calculate left,top and right,bottom from center + length,width -----
      $xsize /= 2*$h_scale; $ysize /= 2*$h_scale;
      my $x1 = ($x - $xsize).'i';
      my $y1 = ($y - $ysize).'i';
      my $x2 = ($x + $xsize).'i';
      my $y2 = ($y + $ysize).'i';

      if ( defined $h_node{$name}->{'type'} ) {
	$shape  =  $ui_prefs::STD_DEPT_SHAPE  if $h_node{$name}->{'type'} eq 'dept';
	$shape  =  $ui_prefs::STD_HOST_SHAPE  if $h_node{$name}->{'type'} eq 'host';
      };

      $can->create($shape, ($x1, $y1, $x2, $y2),
		   -outline => 'black',
		   -fill    => $color,
		   -tags    => [$name, 'node']
		  );

    };
    
    #- show labels unless the scale is high -----
    unless ( $xsize < 0.1 ) {
      #- adjust y to position the label underneath the icon -----
      $y += $ysize/2+(10/$ppi);
      
      my $screen_name = $name;
      $screen_name =~ s/\"//g;
      $screen_name =~ s/$ui_prefs::IMPLICIT_SUFFIX$//;
      
      $can->create('text', $x.'i', $y.'i', 
		   -anchor  => 'n',
		   -fill    => 'black',
		   -text    => $screen_name, 
		   -justify => 'center', 
		   -font    => $SNfont, 
		   -tags    => [$name,'text']
		  );
    };

} #- showNode {}





#==============================================================================
# give a visual cue to which department is being monitored
#
# !! several styles are available, and we must consider monitoring multiple E's
#==============================================================================
sub showFocus {
    my $can = shift;

    my $focus = $h_node{'_focus'};  #- get name of item with focus
    if (defined $focus) {
      my ($x0,$y0,$x1,$y1) = $can->bbox($h_node{'_focus'});
       
      if ($ui_prefs::FOCUS_STYLE eq 'frame' ) {
	$x0 -= 20;
	$y0 -= 20;
	$x1 += 20;
	$y1 += 20;
	
	$can->create('rectangle', ($x0, $y0, $x1, $y1), 
		     -outline => 'white',
		     -width   => 8,
		     -tags    => '_focus' 
		    );
      }
      elsif ($ui_prefs::FOCUS_STYLE eq 'eye' ) {
	$can->create( 'image', ($x0+$x1)/2, $y0+5,
		     -image => 'eye',
		     -anchor => 'n',
		     -tags => '_focus' 
		    );
      }
      elsif ($ui_prefs::FOCUS_STYLE eq 'color' ) {
	$can->itemconfigure($focus,-fill=>$ui_prefs::STD_FOCUS_OBJ_COLOR,-width=>2);
	$can->itemconfigure('text',-fill=>'black',-width=>0);
      }
      else {
	$x0 -= 5;  $y0 -= 5;  $x1 += 5;  $y1 += 5;
	
	$can->create('rectangle', ($x0, $y0, $x1, $y1), 
		     -outline => 'black',
		     -fill    => 'white',
		     -width   => 2,
		     -tags    => '_focus' 
		    );
	$can->lower('_focus');

      }
      
    }
  } #- showFocus {}





#==============================================================================
# hilight an hierarchy item
#
#
#==============================================================================
sub showHilight{
    my $can = shift;

    my $item = $h_node{'_hilight'};  #- get name of item with focus
    if (defined $item) {
      my ($x0,$y0,$x1,$y1) = $can->bbox($h_node{'_hilight'});
       
      $x0 -= 5;  $y0 -= 5;  $x1 += 5;  $y1 += 5;
      
      $can->create('rectangle', ($x0, $y0, $x1, $y1), 
		   -outline => 'black',
		   -fill    => 'white',
		   -width   => 2,
		   -tags    => '_hilight' 
		  );
      $can->lower('_hilight');
    }
} #- showHilight





#==============================================================================
# draw a connecting edge between two hierarchy icons
#
#
#==============================================================================
sub showEdge {
    my ( $can, $from, $to ) = @_;
    
    my $edge = $can->create( 'line', $h_node{$from}->{'x'}.'i', $h_node{$from}->{'y'}.'i', 
                                     $h_node{$to}->{'x'}.'i',   $h_node{$to}->{'y'}.'i',
			    -tags    => '_edge' 
			   );
    $can->lower($edge);
} #- showEdge {}









#==============================================================================
# create the scrollable columized listbox
#
#
#==============================================================================
sub create_listbox {
  my ( $df, $sbr, $tf, $wd, $lbl ) = @_;
  my $lb;
  

  $lb = $df->Listbox(
		     #-yscrollcommand => ['set', $sbr],
		     -relief  => 'ridge',
		     -bd      => 1,
		     -width   => $wd,
		     -height  => 10,
		     #-setgrid => 'yes'
		    );
  $lb->pack(-side => 'left', -fill=> 'both', -expand => 1 );
=keep
=cut
  $tf->Label(#-font   => $SNfont, 
	     -bg      => 'white', 
	     -width   => $wd, 
	     -anchor  => 'sw', 
	     -text    => ' '.$lbl )->pack(-side=>'left', -fill=> 'x', -expand => 1 );
  
  return $lb;
} # end create_listbox()





#==============================================================================
# syncronously scroll multiple columns
#
#
#==============================================================================
sub scrollcols {
  my ($sb, $wigs, @args) = @ARG;
  my $w;

  foreach $w ( @$wigs) {
    $w->yview(@args);
  }
}





#==============================================================================
#
# Setup and manage a department inspector
#  
# !! must be make not a canned dialog, but acutall get info from dept
#==============================================================================
sub dept_inspector {
  my $dept = shift;

  my %dinfo = ();

  $dinfo{'name'        }{'label'} = "";
  $dinfo{'up_since'    }{'label'} = "Up since:";
  $dinfo{'status'      }{'label'} = "Status:";
  $dinfo{'created'     }{'label'} = "Created:";
  $dinfo{'aggregator'  }{'label'} = "Aggregator:";
  $dinfo{'manager'     }{'label'} = "Manager:";
  $dinfo{'num_hosts'   }{'label'} = "Hosts:";
  $dinfo{'dir_hosts'   }{'label'} = "Direct Hosts:";
  $dinfo{'children'    }{'label'} = "Children:";
  $dinfo{'sniffer'     }{'label'} = "Sniffers:";

  $dinfo{'name'        }{'data'} = $dept;
  $dinfo{'up_since'    }{'data'} = get_uptime('dept',$dept);
  $dinfo{'status'      }{'data'} = get_status('dept',$dept);
  $dinfo{'created'     }{'data'} = get_create_info('dept',$dept);
  $dinfo{'aggregator'  }{'data'} = $hi->{$dept}->{'aggregator_host'}.':'.$hi->{$dept}->{'aggregator_port'};
  $dinfo{'manager'     }{'data'} = $hi->{$dept}->{'manager_host'}.':'.$hi->{$dept}->{'manager_port'};
  $dinfo{'num_hosts'   }{'data'} = '??';
  $dinfo{'dir_hosts'   }{'data'} = scalar(@{$hi->{$dept}->{'host_children'}});
  $dinfo{'children'    }{'data'} = scalar(@{$hi->{$dept}->{'dept_children'}});
  $dinfo{'sniffer'     }{'data'} = '??';


#- find the longest label -----
  my $width = 0;
  foreach (keys %dinfo) {
    my $l = length($dinfo{$_}{'label'});
    $width = $l if ($l > $width);
  };

  #- create the window -----
  my $diW = $MW->Toplevel(-height => '16c', 
			 -width  => '10c',
			 -title  => 'Department Inspector',
			 -relief => 'flat',
			 -bg     => 'gray' );
  

#- create a frame for the department title -----
  my $title_frame = $diW->Frame(-height =>'0.5i',
			       -width  =>'6i',
			       -bg     =>'lightgray',
			       -relief => 'flat' );
  $title_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');

#- set the title in the frame -----
  my $title_label = $title_frame->Label(-font   => $LBfont,
					-anchor => 'sw',
					-width  => 32,
					-bg     => 'lightgray',
					-text   => $dinfo{'name'}{'data'});
  $title_label->pack(-side=>'top',-expand=>'yes',-fill=>'x');



  my $ot_frame = $diW->Frame(-bg     =>'lightgray',
			    -relief => 'flat' );
  $ot_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  my $ot = $ot_frame->Label(-font   => $SNfont,
			    -anchor => 'sw',
			    -bg     => 'lightgray',
			    -text   => 'overview');
  $ot->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  


#- declare and pack a frame for the overview items -----
  my $overview_frame = $diW->Frame(-bg     => 'lightgray',
				  -padx   => '.5i',
				  -pady   => '.5i',
				  -relief => 'ridge',
				  -bd     => 4 );
  $overview_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  my $st_frame = $diW->Frame(-bg     =>'lightgray',
			    -height => '0.5i',
			    -relief => 'flat' );
  $st_frame->pack(-side=>'top',-expand=>'no',-fill=>'both');

  my $st = $st_frame->Label(-font   => $SNfont,
			    -anchor => 'sw',
			    -bg     => 'lightgray',
			    -text   => 'statistics');
  $st->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  

#- declare and pack a frame for the stats items -----
  my $stats_frame = $diW->Frame(-bg     => 'lightgray',
			       -padx   => '.5i',
			       -pady   => '.5i',
			       -relief => 'ridge',
			       -bd     => 4 );
  $stats_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');




  my $overview_lbl_frame = $overview_frame->Frame(-bg     => 'lightgray',
						  -padx   => '.5i',
						  -pady   => '.5i',
						  -width  => '3i',
						  -relief => 'flat' );
  $overview_lbl_frame->pack(-side=>'left',-expand=>'no',-fill=>'x');
  
  my $overview_ent_frame = $overview_frame->Frame(-bg     => 'lightgray',
						  -padx   => '.5i',
						  -pady   => '.5i',
						  -relief => 'flat' );
  $overview_ent_frame->pack(-side=>'left',-expand=>'yes',-fill=>'x');
  
  my $stats_lbl_frame = $stats_frame->Frame(-bg     => 'lightgray',
					    -padx   => '.5i',
					    -pady   => '.5i',
					    -width  => '3i',
					    -relief => 'flat' );
  $stats_lbl_frame->pack(-side=>'left',-expand=>'no',-fill=>'x');
  
  my $stats_ent_frame = $stats_frame->Frame(-bg     => 'lightgray',
					    -padx   => '.5i',
					    -pady   => '.5i',
					    -relief => 'flat' );
  $stats_ent_frame->pack(-side=>'left',-expand=>'yes',-fill=>'x');
  



  foreach ( qw(up_since status created aggregator manager) ) {
    my $lbl = $overview_lbl_frame->Label(-font    => $SBfont,
					 -width   => $width,
					 -justify => 'right',
					 -anchor  => 'e',
					 -text    => $dinfo{$_}{'label'}
					);
    $lbl->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
    my $inf = $overview_ent_frame->Label(-font    => $SNfont,
					 -justify => 'left',
					 -anchor  => 'w',
					 -text    => $dinfo{$_}{'data'}
					);
    $inf->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
  };
  
  foreach ( qw(num_hosts dir_hosts children sniffer) ) {
    my $lbl = $stats_lbl_frame->Label(-font    => $SBfont,
				      -width   => $width,
				      -justify =>'right',
				      -anchor  => 'e',
				      -text    => $dinfo{$_}{'label'}
				     );
    $lbl->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
    my $inf = $stats_ent_frame->Label(-font    => $SNfont,
				      -justify => 'left',
				      -anchor  => 'w',
				      -text    => $dinfo{$_}{'data'}
				     );
    $inf->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  };
  
}; #- dept_inspector {}





#==============================================================================
#
# Setup and manage a host inspector
#
# !! must be make not a canned dialog, but acutall get info from host
#==============================================================================
sub host_inspector {
  my ($host) = @_;

  my %hinfo = ();
  my $department = $hi->{$host}->{'parent'};
  
  $hinfo{'name'}{'label'}       = $host;
  $hinfo{'name'}{'data'}        = $host;

  $hinfo{'mod_cont'}{'label'}   = "Module Controller";
  $hinfo{'mod_cont'}{'data'}    = '(port '.$hi->{$host}->{'port'}.')';

#- get info about all aggregators running on this host -----
  my @aggs;
  foreach (get_agg_info($host)) {
    my ($dept,$port) = @{$_};
    $hinfo{$dept}{'label'} = "Aggregator ";
    $hinfo{$dept}{'data'}  = '(dept '.$dept.', pid '.$port.')';
    @aggs = (@aggs,$dept);
  };

#- get info about all datasources running on this host -----
  my @dts;

# !! actually returns datasource, change after fixing subroutine !!!!!!!
  %datasource = update_datasource_info($host,%datasource);

  foreach ( sort keys %datasource ) {
    $hinfo{$_}{'label'} = $_;
    $hinfo{$_}{'data'}  = $datasource{$_}->{'state'};
    @dts = (@dts,$_);
  };


#- find the longest label -----
  my $width = 0;
  foreach (keys %hinfo) {
    my $l = length($hinfo{$_}{'label'});
    $width = $l if ($l > $width);
  };

  #- create the window -----
  my $hiW = $MW->Toplevel(-height => '16c', 
			 -width  => '10c',
			 -title  => 'Host Inspector',
			 -relief => 'flat',
			 -bg     => 'gray' );
  

#- create a frame for the department title -----
  my $title_frame = $hiW->Frame(-height =>'0.5i',
			       -width  =>'6i',
			       -bg     =>'lightgray',
			       -relief => 'flat' );
  $title_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');

#- set the title in the frame -----
  my $title_label = $title_frame->Label(-font   => $LBfont,
					-anchor => 'sw',
					-width  => 32,
					-bg     => 'lightgray',
					-text   => $hinfo{'name'}{'data'});
  $title_label->pack(-side=>'top',-expand=>'yes',-fill=>'x');


  my $it_frame = $hiW->Frame(-bg     =>'lightgray',
			    -relief => 'flat' );
  $it_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');

  my $it = $it_frame->Label(-font   => $SNfont,
			    -anchor => 'sw',
			    -bg     => 'lightgray',
			    -text   => 'infrastructure');
  $it->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  


#- declare and pack a frame for the infra items -----
  my $infra_frame = $hiW->Frame(-bg     => 'lightgray',
				  -padx   => '.5i',
				  -pady   => '.5i',
				  -relief => 'ridge',
				  -bd     => 4 );
  $infra_frame->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  
  my $st_frame = $hiW->Frame(-bg     =>'lightgray',
			    -height => '0.5i',
			    -relief => 'flat' );
  $st_frame->pack(-side=>'top',-expand=>'no',-fill=>'x');

  my $st = $st_frame->Label(-font   => $SNfont,
			    -anchor => 'sw',
			    -bg     => 'lightgray',
			    -text   => 'data sources');
  $st->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  

#- declare and pack a frame for the ds items -----
  my $dts_frame = $hiW->Frame(-bg     => 'lightgray',
			      -padx   => '.5i',
			      -pady   => '.5i',
			      -relief => 'ridge',
			      -bd     => 4 );
  $dts_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  
  my $infra_lbl_frame = $infra_frame->Frame(-bg     => 'lightgray',
					    -padx   => '.5i',
					    -pady   => '.5i',
					    -width  => '3i',
					    -relief => 'flat' );
  $infra_lbl_frame->pack(-side=>'left',-expand=>'no',-fill=>'x');
  
  my $infra_ent_frame = $infra_frame->Frame(-bg     => 'lightgray',
					    -padx   => '.5i',
					    -pady   => '.5i',
					    -relief => 'flat' );
  $infra_ent_frame->pack(-side=>'left',-expand=>'yes',-fill=>'x');
  
  my $dts_left_frame = $dts_frame->Frame(-bg     => 'lightgray',
				       -padx   => '.5i',
				       -pady   => '.5i',
				       -width  => '3i',
				       -relief => 'flat' );
  $dts_left_frame->pack(-side=>'left',-expand=>'yes',-fill=>'both');
  
  my $dts_right_frame = $dts_frame->Frame(-bg     => 'lightgray',
					-padx   => '.5i',
					-pady   => '.5i',
					-relief => 'flat' );
  $dts_right_frame->pack(-side=>'left',-expand=>'yes',-fill=>'both');
  



  foreach ('mod_cont', @aggs) {
    my $lbl = $infra_lbl_frame->Label(-font    => $SBfont,
					 -width   => $width,
					 -justify => 'right',
					 -anchor  => 'e',
					 -text    => $hinfo{$_}{'label'}
					);
    $lbl->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
    my $inf = $infra_ent_frame->Label(-font    => $SNfont,
					 -justify => 'left',
					 -anchor  => 'w',
					 -text    => $hinfo{$_}{'data'}
					);
    $inf->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
  };

  my $frame = $dts_left_frame;    #- Assign buttons to left frame first 
  my $ds_name;
  foreach $ds_name ( @dts ) {
    my ($ds_type, $ds_ver) = split(':',$ds_name);
    my $btn = $frame->Checkbutton(-font    => $SBfont,
				  -width   => $width,
				  -anchor  => 'w',
				  -relief  => 'raised',
				  -text    => "$ds_type ($ds_ver)",
				  -variable=> \$hinfo{$ds_name}{'data'},
				  -command => sub {  my $state = 'KILL';
						    $state = 'START' if $hinfo{$ds_name}{'data'} == 1;
						    change_datasource_activation($department,
										 $host,
										 $ds_type,
										 $ds_ver,
										 $state
										)
						  }
				 );
    $btn->bind( '<3>' => sub { if ($hinfo{$ds_name}{'data'} == 1) {
                                 do_module_inspector_popup( $department, $host, $ds_type, $ds_ver );
			       } 
			       else {
				 $MW->bell;
			       };
			     });

    $btn->pack(-side=>'top');

    #- toggle placement between the left and right sides of the frame -----
    if ($frame eq $dts_left_frame){
      $frame = $dts_right_frame;
    }
    else {
      $frame = $dts_left_frame;
    };
    
  };
  
}; #- host_inspector {}





#==============================================================================
#
# Setup and manage a module inspector
#
# !! must be make not a canned dialog, but acutall get info from sm
#==============================================================================
sub module_inspector {
  my ($dept,$host,$mod_name,$mod_ver) = @_;
  my $port;
  my $module = "$mod_name:$mod_ver";
  my %minfo = ();
  my %cvars = ();

  $minfo{'title'}                         = "$mod_name";

  $minfo{'overview'}{'dept'}{'label'}     = 'department';
  $minfo{'overview'}{'dept'}{'data'}      = $dept;
  $minfo{'overview'}{'host'}{'label'}     = 'host';
  $minfo{'overview'}{'host'}{'data'}      = "$host:$port";
  
  $minfo{'overview'}{'uptime'}{'label'}   = "Up since:";
  $minfo{'overview'}{'uptime'}{'data'}    = get_uptime('mod',$host,$module);
  $minfo{'overview'}{'status'}{'label'}   = "Status:";
  $minfo{'overview'}{'status'}{'data'}    = get_status('mod',$host,$module);
  
  #- list of keys to display in top frame -----
  my @items = ('dept','host','uptime','status'); 
  
  #- get list of all control variables for the module -----
  $log->warn("getting all control variables and values for the $module ($host, $mod_name, $dept)\n");
  my ($ok, $rslt) = $hierarchy->get($host, $mod_name, $dept,
				    'ALL_STATEVARS{ALL_STATEVARS}', '_'
				   );
  unless ($ok) {
    Dialog_OK('Error: getting ALL_STATEVARS{ALL_STATEVARS} from $dept',$rslt);
    $log->warn("'Error: getting ALL_STATEVARS{ALL_STATEVARS} from $dept: ",join(', ',@$rslt) );
    return 0;
  };


  #-  place the control var label/data pairs into a hash -----
  %{$minfo{'c_vars'}} = @$rslt;
  
  #- save a copy of the current values of all control vars -----
  foreach (keys %{$minfo{'c_vars'}}) {
    $cvars{$_} = $minfo{'c_vars'}{$_};
  };
  
  #- find the longest label -----
  my $width = 0;
  my $l;
  foreach (keys %{$minfo{'overview'}}) {
    $l = length($minfo{'overview'}{$_}{'label'});
    $width = $l if ($l > $width);
  };
  
  foreach (keys %{$minfo{'c_vars'}}) {
    $l = length($_);
    $width = $l if ($l > $width);
  };
  
  #- create the window -----
  my $miW = $MW->Toplevel(-height => '16c', 
			  -width  => '10c',
			  -title  => 'Module Inspector',
			  -relief => 'flat',
			  -bg     => 'gray' );
  
  
  #- create a frame for the module title -----
  my $title_frame = $miW->Frame(-height =>'0.5i',
				-width  =>'6i',
				-bg     =>'lightgray',
				-relief => 'flat' );
  $title_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  #- set the title in the frame -----
  my $title_label = $title_frame->Label(-font   => $LBfont,
					-anchor => 'sw',
					-width  => 32,
					-bg     => 'lightgray',
					-text   => $minfo{'title'});
  $title_label->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  
  
  #- create a frame for the overview section label -----
  my $ovlbl_frame = $miW->Frame(-bg     =>'lightgray',
				-relief => 'flat' );
  $ovlbl_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  my $ov = $ovlbl_frame->Label(-font   => $SNfont,
			       -anchor => 'sw',
			       -bg     => 'lightgray',
			       -text   => 'overview');
  $ov->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  
  
  #- create a frame for the overview section info -----
  my $ovinfo_frame = $miW->Frame(-bg     => 'lightgray',
				 -padx   => '.5i',
				 -pady   => '.5i',
				 -relief => 'ridge',
				 -bd     => 4 );
  $ovinfo_frame->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  
  #- create a frame for the control variable label  -----
  my $cvlbl_frame = $miW->Frame(-bg    => 'lightgray',
				-height => '0.5i',
				-relief => 'flat' );
  $cvlbl_frame->pack(-side=>'top',-expand=>'no',-fill=>'x');
  
  my $cv= $cvlbl_frame->Label(-font   => $SNfont,
			      -anchor => 'sw',
			      -bg     => 'lightgray',
			      -text   => 'control variables');
  $cv->pack(-side=>'top',-expand=>'yes',-fill=>'x');
  
  
  #- create a frame for the control variable info -----
  my $cvinfo_frame = $miW->Frame(-bg     => 'lightgray',
				 -padx   => '.5i',
				 -pady   => '.5i',
				 -relief => 'ridge',
				 -bd     => 4 );
  $cvinfo_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  
  #- create a frame for the buttons -----
  my $btn_frame = $miW->Frame(-bg     => 'lightgray',
			      -padx   => '.5i',
			      -pady   => '.5i',
			      -relief => 'ridge',
			      -bd     => 4 );
  $btn_frame->pack(-side=>'top',-expand=>'yes',-fill=>'both');
  
  my $cancel_btn = $btn_frame->Button(-text=>'Cancel', 
				      -relief=>'raised', 
				      -command=>[ 'destroy', $miW ] );
  $cancel_btn->pack(-side=>'right',-expand=>'yes',-fill=>'none');
  
  my $ok_btn = $btn_frame->Button(-text    => 'Submit',
				  -bd      => '2',
				  -relief  => 'raised',
				  -command => sub { 
				    change_control_vars($dept,$host,$mod_name,$mod_ver,$minfo{'c_vars'},\%cvars);
				  }
				 );
  $ok_btn->pack(-side=>'right',-expand=>'yes',-fill=>'none');
  
  
  #- create label and entry frames for the overview frame -----  
  my $ovinfo_lbl_frame = $ovinfo_frame->Frame(-bg     => 'lightgray',
					      -padx   => '.5i',
					      -pady   => '.5i',
					      -width  => '3i',
					      -relief => 'flat' );
  $ovinfo_lbl_frame->pack(-side=>'left',-expand=>'no',-fill=>'x');
  
  my $ovinfo_ent_frame = $ovinfo_frame->Frame(-bg     => 'lightgray',
					      -padx   => '.5i',
					      -pady   => '.5i',
					      -relief => 'flat' );
  $ovinfo_ent_frame->pack(-side=>'left',-expand=>'yes',-fill=>'x');
  
  
  #- create label and entry frames for the control variable section -----
  my $cvinfo_lbl_frame = $cvinfo_frame->Frame(-bg     => 'lightgray',
					      -padx   => '.5i',
					      -pady   => '.5i',
					      -width  => '3i',
					      -relief => 'flat' );
  $cvinfo_lbl_frame->pack(-side=>'left',-expand=>'yes',-fill=>'both');
  
  my $cvinfo_ent_frame = $cvinfo_frame->Frame(-bg     => 'lightgray',
					      -padx   => '.5i',
					      -pady   => '.5i',
					      -relief => 'flat' );
  $cvinfo_ent_frame->pack(-side=>'left',-expand=>'yes',-fill=>'both');
  
  
  
  
  foreach (@items) {
    my $lbl = $ovinfo_lbl_frame->Label(-font    => $SBfont,
				       -width   => $width,
				       -justify => 'right',
				       -anchor  => 'e',
				       -text    => $minfo{'overview'}{$_}{'label'}
				      );
    $lbl->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
    my $inf = $ovinfo_ent_frame->Label(-font    => $SNfont,
				       -justify => 'left',
				       -anchor  => 'w',
				       -text    => $minfo{'overview'}{$_}{'data'}
				      );
    $inf->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
  };
  
  #- create the control variable info window -----
  foreach ( sort keys %{$minfo{'c_vars'}} ) {
    
    #- do not display non-modifyable control_vars -----
    next if defined $Hierarchy_interface::dept_set_not_allowed{$_};   
    next if $_ eq 'command';
    next if $_ eq 'module';
    next if $_ eq 'version';
    
    my $lbl = $cvinfo_lbl_frame->Label(-font    => $SBfont,
				       -width   => $width,
				       -justify => 'right',
				       -anchor  => 'w',
				       -text    => $_
				      );
    $lbl->pack(-side=>'top',-expand=>'yes',-fill=>'x');
    
    
    my $cv_entry = $cvinfo_ent_frame->Entry(-width=>50, 
					    -textvariable=>\$minfo{'c_vars'}{$_}, 
					    -font=>$MNfont );
    
    $cv_entry->pack(-side=>'top');
  };  
} #- module_inspector




#==============================================================================
#
# GrIDS startup dialog 
#
#==============================================================================
sub startup {
  
  my $done = 'false';
  my $startup_mode;
  
  my $startup_window = $MW->Toplevel;

  $startup_window->resizable(0,0);          #- make the window not resizable -----
  $startup_window->title('GrIDS Startup');
  $startup_window->Popup();                 #- make this the top window
  $startup_window->focus;                   #- make this the "hot" window
  $startup_window->grab;                    #- restrict all activity to this window

  my $radio_frame  = $startup_window->Frame(-relief=>'flat')->pack(-side=>'top',    -fill=>'x', -expand=>1);
  my $button_frame = $startup_window->Frame(-relief=>'flat')->pack(-side=>'bottom', -fill=>'none', -expand=>1);
  my $prompt_frame = $startup_window->Frame(-relief=>'flat')->pack(-side=>'left',   -fill=>'x', -expand=>1);
  my $entry_frame  = $startup_window->Frame(-relief=>'flat')->pack(-side=>'left',   -fill=>'x', -expand=>1);

  my $new_ohs_radio = $radio_frame->Radiobutton();
  $new_ohs_radio->configure(-text     => 'New OHS', 
			    -variable => \$startup_mode,
			    -value    => 0,
			    -justify  => 'left',
			    -bg       => 'lightgray',
			   );
  $new_ohs_radio->pack( -fill=>'none', -expand=>0, -anchor => 'w', -side=>'left');

  my $prev_ohs_radio = $radio_frame->Radiobutton();
  $prev_ohs_radio->configure(-text     => 'Existing OHS', 
			     -variable => \$startup_mode,
			     -value    => 1,
			     -justify  => 'left',
			     -bg       => 'lightgray',
			    );
  $prev_ohs_radio->pack( -fill=>'none', -expand=>0, -anchor => 'w', -side=>'left' );
  $prev_ohs_radio->invoke;
  
  my $host_label = $prompt_frame->Label(-text=>'OHS Host: ', 
					-font=>$MNfont);
  $host_label->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  my $host_entry = $entry_frame->Entry(-width=>20, 
					-textvariable=>\$ohs_host, 
					-font=>$MNfont );
  $host_entry->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  
  my $port_label = $prompt_frame->Label(-text=>'OHS Port: ', 
					-font=>$MNfont);
  $port_label->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  
  my $port_entry = $entry_frame->Entry(-width=>20, 
					-font=>$MNfont, 
					-textvariable=>\$ohs_port );
  $port_entry->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  
  my $dept_label = $prompt_frame->Label(-text=>'Monitor Dept: ', 
					-font=>$MNfont);
  $dept_label->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  
  my $dept_entry = $entry_frame->Entry(-width=>20, 
					-font=>$MNfont, 
					-textvariable=>\$h_node{'_focus'} );
  $dept_entry->pack(-side=>'top', 
		    -fill=>'x', 
		    -expand=>'true' );
  
  my $cancel_btn = $button_frame->Button(-text=>'Cancel', 
					 -relief=>'raised', 
					 -command=>[ 'destroy', $MW ] );
  $cancel_btn->pack(-side=>'right');
  
  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub { 
				       $startup_window->withdraw;
				       $done = do_startup($ohs_host,$ohs_port,$startup_mode);
				       }
				    );
  $ok_btn->pack(-side=>'right');

  $startup_window->bind('<Return>' => sub { $startup_window->withdraw;
					    $done = do_startup($ohs_host,$ohs_port,$startup_mode);
					  });

#  $startup_window->geometry('+100+100');
  $host_entry->focus;                       #- make this the "hot" widget

  while ($done eq 'false') {     #- Event loop for &startup_window
    DoOneEvent(1);               #    
  };                             #
  
  $startup_window->grabRelease;
  $startup_window->destroy();
}; #- startup_dialog {}





#==============================================================================
#
# Let OHS know of the UI's presence and get initial structure of the heirarchy 
#   
#==============================================================================
sub do_startup {
  my ($ohs_host, $ohs_port, $mode) = @_;

print( "*** in do_startup ***\n" );
  &get_existing_hierarchy if $mode == 1;
  &start_new_hierarchy    if $mode == 0;
 
  cleanup_hierarchy();
  draw_hierarchy($h_canvas);

#  monitor_startup($h_node{'_focus'}) if defined $h_node{'_focus'};

  return 'true';
}; #- do_startup {}


      


#==============================================================================
#==============================================================================
sub get_existing_hierarchy {
  $log->warn("getting hierarchy from $ohs_host:$ohs_port, $root_sm_host:$root_sm_port\n");
  my $ok;
  ($ok, $hierarchy) = new Hierarchy_interface($username,$password,'ROOT',$log,'comm','main::FILE');
  unless ($ok) {
    $log->warn("Unable to obtain hierarchy interface\nok=$ok\nhi=$hierarchy\n");
    die("Unable to obtain hierarchy interface. (see log for details.)\n");
  }

  $hi = $hierarchy->{'hierarchy'};

 #- DEBUG - show the hierarchy on the console ----- 
 &print_hierarchy;
};



#==============================================================================
#==============================================================================
sub start_new_hierarchy {
  create_root_dept($h_canvas);
};





#==============================================================================
#
# Get a username and password from the user 
#
#==============================================================================
sub access_control {
  
  my ($username, $password) = ('','');
  my $done = 'false';
  my $font  = '-*-Helvetica-Medium-R-Normal--*-180-*-*-*-*-*-*';
  my ($wx, $wy) = (320,100);          #- define desired size for the window


  my $access_control  = MainWindow->new;
  
  $access_control->configure(-height=>$wy,-width=>$wx);         
  $access_control->resizable(0,0);    #- make the window not resizable -----
  $access_control->title('GrIDS Access Control');
  #$access_control->Popup();          #- make this the top window
  $access_control->focus;             #- make this the "hot" window
  $access_control->grab;              #- restrict all activity to this window

  
  my $button_frame = $access_control->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $prompt_frame = $access_control->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame  = $access_control->Frame(-relief=>'flat')->pack(-side=>'left');

  my $uname_label = $prompt_frame->Label(-text=>'Username: ', -font=>$font);
  $uname_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $pswrd_label = $prompt_frame->Label(-text=>'Password: ', -font=>$font);
  $pswrd_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $uname_entry = $entry_frame->Entry(-width=>20, 
					-textvariable=>\$username, 
					-font=>$font );
  $uname_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $pswrd_entry = $entry_frame->Entry(-width=>20, 
					-show=>'*', 
					-font=>$font, 
					-textvariable=>\$password );
  $pswrd_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { exit 0; } );
  $cancel_btn->pack(-side=>'right');
  
  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub { $done = validate_user($username,$password,$access_control); 
						       $uname_entry->focus;
						     } 
				    );
  $ok_btn->pack(-side=>'right');

  $access_control->bind('<Return>' => sub { $ok_btn->invoke(); });

  $uname_entry->focus;           #- make the username entry the active widget

  #- !! unfortunately, since the window hasn't been drawn, it has now height an width. We need to find
  # a way to determine its geometry first. For now the actual size was determined and hardcoded.
  my ($xroot, $yroot) = center_widget_on_screen($access_control, $wx, $wy);
  $access_control->geometry($wx.'x'.$wy.'+'.$xroot.'+'.$yroot);

  while ($done eq 'false') {     #- Event loop for &access_control
    DoOneEvent(1);               #    
  };                             #
  
  $access_control->grabRelease;
  $access_control->destroy();

  return ($username,$password);
}; #- access_control





#==============================================================================
#
# Validate the username and password 
# *** THIS IS JUST A DUMMY ROUTINE FROM BEFORE ACCESS CONTROL ***
# *** SUGGEST WE CREATE A VALIDATE CALL TO GIVE IMMEDIATE FEEDBACK
# *** RATHER THAN WAIT UNTIL LATER TO VALIDATE THE PASSWORD
#
#==============================================================================
sub validate_user {
  my ($u, $p, $w ) = @_;
  return 'true';
  unless (1){
    $w->bell;
    sleep(2);
    return 'false';
  } 
  else {
    return 'true';
  } 
}; #- validate_user {}




#==============================================================================
#
# Change a users password
#
#==============================================================================
sub account_mgmt {
  my ($dept) = @_;

  my ($username, $password, $vf_password) = ('','','');
  my $done   = 'false';
  my $result = 'false';
  my $font  = '-*-Helvetica-Medium-R-Normal--*-180-*-*-*-*-*-*';
  my ($wx, $wy) = (320,200);          #- define desired size for the window


  my $account  = MainWindow->new;
  
  $account->configure(-height=>$wy,-width=>$wx);         
  $account->resizable(0,0);    #- make the window not resizable -----
  $account->title('User Account Management');
  $account->focus;             #- make this the "hot" window
  $account->grab;              #- restrict all activity to this window

  
  my $button_frame = $account->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $prompt_frame = $account->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame  = $account->Frame(-relief=>'flat')->pack(-side=>'left');

  my $dept_label;
  if ( $dept eq '') {
    $dept_label = $prompt_frame->Label(-text=>'Department: ', -font=>$font);
    $dept_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  }
  else {
       $account->title("User Account Management on $dept");
  }
  my $uname_label = $prompt_frame->Label(-text=>'Username: ', -font=>$font);
  $uname_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $pswrd_label = $prompt_frame->Label(-text=>'Password: ', -font=>$font);
  $pswrd_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $vf_pswrd_label = $prompt_frame->Label(-text=>'Verify Password: ', -font=>$font);
  $vf_pswrd_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
    
  my $dept_entry;
  if ( $dept eq '') {
    $dept_entry = $entry_frame->Entry(-width=>20, 
					  -textvariable=>\$dept, 
					  -font=>$font );
    $dept_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  }
  
  my $uname_entry = $entry_frame->Entry(-width=>20, 
					-textvariable=>\$username, 
					-font=>$font );
  $uname_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $pswrd_entry = $entry_frame->Entry(-width=>20, 
					-show=>'*', 
					-font=>$font, 
					-textvariable=>\$password );
  $pswrd_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $vf_pswrd_entry = $entry_frame->Entry(-width=>20, 
					-show=>'*', 
					-font=>$font, 
					-textvariable=>\$vf_password );
  $vf_pswrd_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { $done = 'true'; $result = 0; } );
  $cancel_btn->pack(-side=>'right', -fill=> 'x', -expand => 'yes', -padx => '2m');
  
  my $change_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub { $done = do_change_passwd($username,$password,$vf_password,$dept); 
						       $uname_entry->focus;
						       $result = $done;
						     }
				    );
  $change_btn->pack(-side=>'right', -fill=> 'x', -expand => 'yes', -padx => '2m');

  if ( $dept eq '' ) {
    $dept_entry->focus;
  }
  else {
    $uname_entry->focus;           #- make the username entry the active widget
  }

  #- !! unfortunately, since the window hasn't been drawn, it has now height an width. We need to find
  # a way to determine its geometry first. For now the actual size was determined and hardcoded.
  my ($xroot, $yroot) = center_widget_on_screen($account, $wx, $wy);
  $account->geometry($wx.'x'.$wy.'+'.$xroot.'+'.$yroot);

  while ($done eq 'false') {     #- Event loop for &account
    DoOneEvent(1);               #    
  };                             #
  
  $account->grabRelease;
  $account->destroy();

  return $result;
}; #- account_mgmt



#==============================================================================
#
#==============================================================================
sub do_change_passwd {
  my ($nm,$np,$vp,$dp) = @_;
  
  unless ( $np eq $vp) { #- two passwords don't match, inform user
    Dialog_OK('Bad Password','Two passwords do not match. Please re-enter');
    return 0;
  }
  
  #- send change request to hierarchy -----
  my ($ok,$err) = $hierarchy->change_user($nm, $np, $dp);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error changing user info ($nm,$np,$dp): $err\n");
    return 0;
  }

  return 1;

};




#==============================================================================
# attempt to figure out how to position a window centered on screen
#
#
#==============================================================================
sub center_widget_on_screen {
  my ($wd, $wx, $wy) = @_;

  $wx = $wd->width  unless defined $wx;
  $wy = $wd->height unless defined $wy;

  my $x = int(($wd->screenwidth - 320)/2);
  my $y = int(($wd->screenheight - 100)/2);

  return ($x, $y);
};




      


#==============================================================================
# makes use of fact that no node will ever start with '_'. Special objects,
#   like the focus are also dragable objects.
#==============================================================================
sub items_start_drag {

  my($can, $x, $y, $iinfo) = @ARG;
 
  $iinfo->{'firstX'} = -1;
  $iinfo->{'firstY'} = -1;    
  
  my $item = $can->find( 'withtag', 'current' );
  
  if ( defined($item) ) { 
    my $name  = get_item_name($can);
    
    $iinfo->{'firstX'} = $can->canvasx($x);
    $iinfo->{'firstY'} = $can->canvasy($y);    
  
    #- see if we have clicked on a valid movable object -----
    if ( substr($name,0,1) ne '_' or $name eq '_focus' ) {
      
      if  ($name ne '_focus' ) {
	#- get width and height of the icon -----
	my ($x0,$y0,$x1,$y1) = $can->bbox('current');
	
	$can->create('rectangle', ($x0, $y0, $x1, $y1), 
		     -fill => 'lightgray', 
		     -tags => '_holder' 
		    );
	#$can->raise('holder');
      };
      
      $can->raise($name);
      $iinfo->{'lastX'} = $can->canvasx($x);
      $iinfo->{'lastY'} = $can->canvasy($y);    
      return 1;
    }
  };
  
  
  # -- clicks on the canvas or some invalid oject -----
  $MW->bell;
  $iinfo->{'lastX'} = -1; #- mark it as not movable
  $iinfo->{'lastY'} = -1;
  
} # end items_start_drag





#==============================================================================
#==============================================================================
sub items_drag {

    my($can, $x, $y, $iinfo) = @ARG;
    
    return if ( abs($x-$iinfo->{'firstX'}) < $ui_prefs::MIN_MOVE or abs($y-$iinfo->{'firstY'}) < $ui_prefs::MIN_MOVE );
    return if ($iinfo->{'lastX'} == -1);

    $x = $can->canvasx($x);
    $y = $can->canvasy($y);
    $can->move('current', $x-$iinfo->{'lastX'}, $y-$iinfo->{'lastY'});
    $iinfo->{'lastX'} = $x;
    $iinfo->{'lastY'} = $y;
} # end items_drag





#==============================================================================
#==============================================================================
sub shift_hilight {
  my ($can,$name) = @_;
  
  #- if anything was hilighted, remove the hilight -----
  my $item = $can->find('withtag','_hilight');
  $can->delete($item) if defined $item;
  
  $h_node{'_hilight'} = $name;
  showHilight($can);  
}





#==============================================================================
# clicking on an object unhilights any hilighted objects, and higlights the
#   object clicked unless it is already hilighted.
#==============================================================================
sub items_drop {

    my($can, $x, $y, $iinfo) = @ARG;

    my $item = $can->find( 'withtag', '_holder' );
    $can->delete($item) if defined $item;
    
    return if ($iinfo->{'lastX'} == -1);

    my $name  = get_item_name($can);

    if ( $iinfo->{'lastX'} == $iinfo->{'firstX'} and $iinfo->{'lastY'} == $iinfo->{'firstY'} ) {
    
    #- if anything was hilighted, remove the hilight -----
      my $item = $can->find('withtag','_hilight');
      $can->delete($item) if defined $item;
      
    #- no move, just a click and release
      if ( $h_node{'_hilight'} ne $name ) {  #- only hilight if object not already hilighted
	$h_node{'_hilight'} = $name;
	showHilight($can);
      }
      else {
	undef $h_node{'_hilight'};
      };
      showFocus($can);

      return 1;
    };

    my $tag;
    
    my $name  = get_item_name($can);
    
    if ($name eq '_focus') {
    #- moving a focus -----
      my $fs = 'enclosed';
      $fs = 'overlapping' if $ui_prefs::FOCUS_STYLE ne 'frame';
      $tag = get_drop_target($can, $fs);
      
      #- if target is a different department, change focus to that dept -----
      if (defined $tag and $h_node{$tag}->{'type'} eq 'dept') {
	change_focus($can,$tag) if $h_node{$tag}->{'name'} ne $h_node{$tag}->{'_focus'};
      } 
      
      # if target is a host, just beep
      elsif (defined $tag and $h_node{$tag}->{'type'} eq 'host') {
	$MW->bell; 
	return;
      }

      #- othewise assume user wishes to stop monitoring on this dept -----
      else {
	change_focus($can,undef);	
      }
    }
    else {
    #- moving a department or host -----
      $tag = get_drop_target($can, 'overlapping');
      
      #- if target i
      if (defined $tag) {
	if ( $h_node{$tag}->{'type'} eq 'dept') {
	  do_move_host($tag,$name) if $h_node{$name}->{'type'} eq 'host';
	  do_move_dept($tag,$name) if $h_node{$name}->{'type'} eq 'dept';
	}  
	else {
	  $MW->bell;
	};
      }
      else {
	$h_node{$name}->{'x'} = $can->canvasx($x)/$ppi;
	$h_node{$name}->{'y'} = $can->canvasx($y)/$ppi;
      };
      
      my $item = $can->find( 'withtag', '_holder' );
      $can->delete($item) if defined $item;
    }
    
    draw_hierarchy($can);
} # end items_drop





#==============================================================================
#==============================================================================
sub get_drop_target {
  my ($can, $mode) = @_;
  
  my $name  = get_item_name($can);
  
  my @bounds = $can->bbox('current');             #- get bounding box for object being dropped
  warn("bounds for $can is (",join(", ",@bounds),")");
  $log->warn("bounds for $can is (",join(", ",@bounds),")");
  my @id = $can->find($mode, @bounds);            #- get list of items inside

  #- search through the items for the name of a valid drop target -----
  my $tag;
  foreach (@id) {
    my @taglist = $can->gettags($_);
    foreach (@taglist) {
      last if (substr($_,0,1) eq '_');  #- skip over items with "special" names
      last if ($_ eq $name);            #- skip over current item ($name) 
      last if ($_ eq 'current');        #- skip over current item
      #next if (acceptible non-name identifiers);
      $tag = $_;
      last;
    };
    last if defined $tag;
    next;
  }; 

  return $tag;
} #- get_drop_target





#==============================================================================
#==============================================================================
sub items_enter {

    my($can, $iinfo) = @ARG;

    $info_text = get_item_name($can);

} # end items_enter





#==============================================================================
#==============================================================================
sub items_leave {

    my($can, $iinfo) = @ARG;

    $info_text = '';

} # end items_leave





#==============================================================================
#==============================================================================
sub items_mark {

    my($can, $x, $y, $iinfo) = @ARG;

    $iinfo->{'areaX1'} = $can->canvasx($x);
    $iinfo->{'areaY1'} = $can->canvasy($y);
    $can->delete('area');

} # end items_mark





#==============================================================================
#==============================================================================
sub items_stroke {

    my($can, $x, $y, $iinfo) = @ARG;

    $x = $can->canvasx($x);
    $y = $can->canvasy($y);
    if (($iinfo->{'areaX1'} != $x) and ($iinfo->{'areaY1'} != $y)) {
	$can->delete('area');
	$can->addtag('area', 'withtag', $can->create('rectangle',
	    $iinfo->{'areaX1'}, $iinfo->{'areaY1'}, $x, $y, -outline => 'black'));
	$iinfo->{'areaX2'} = $x;
	$iinfo->{'areaY2'} = $y;
    }

} # end items_stroke





#==============================================================================
# display dialog to provide info needed to add a new host to the hierarchy
# 
#
#==============================================================================
sub create_host {
  my ($c, $cx, $cy, $parent) = @_;

  my ($host, $port);
  
  my $chW  = $MW->Toplevel();
  $chW->withdraw;
  
  #- determine position to place new host icon -----
  my ($dx,$dy) = ($cx, $cy);                        #- put it under the cursor
  ($dx,$dy) = map ($_+$ui_prefs::NEW_OBJ_DISP, ($cx,$cy)) if defined $parent;
                                                    #- unless we clicked on a host,
                                                    #-   the we shift it 

  #- position the dialog at a reasonable position -- must be in pixels  -----
  my ($xroot, $yroot) = map ( int(($_+1)*$ppi), ($cx, $cy) );
  $chW->geometry('+'.$xroot.'+'.$yroot);

  $chW->title('Add New Host');
  $chW->title('Add New Host to \''.$parent.'\'') if (defined $parent);
    
  my $button_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $prompt_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame  = $chW->Frame(-relief=>'flat')->pack(-side=>'left');

  unless (defined $parent){
    my $parent_label = $prompt_frame->Label(-text=>'Parent Dept: ', -font=>$MNfont);
    $parent_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
    my $parent_entry = $entry_frame->Entry(-width=>32, 
					   -textvariable=>\$parent, 
					   -font=>$MNfont );
    $parent_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
    $parent_entry->focus;
  };
  my $host_label = $prompt_frame->Label(-text=>'Host Name: ', -font=>$MNfont);
  $host_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $port_label = $prompt_frame->Label(-text=>'Host Port: ', -font=>$MNfont);
  $port_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $host_entry = $entry_frame->Entry(-width=>32, 
				       -textvariable=>\$host, 
				       -font=>$MNfont );
  $host_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $port_entry = $entry_frame->Entry(-width=>8, 
				       -textvariable=>\$port, 
				       -font=>$MNfont );
  $port_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { $chW->destroy(); } );
  $cancel_btn->pack(-side=>'right');
  
  $chW->bind('<Return>' => sub { 
				 &create_host_obj($parent, $host, $port, $dx, $dy);
				 showNode( $c, $host );
				 showEdge( $c, $parent, $host );
				 $chW->grabRelease;
				 $chW->destroy();
			       } #- NOTE: this is identical to binding for 'Ok Button' --
	    ); 
  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub {
				       &create_host_obj($parent, $host, $port, $dx, $dy);
				       showNode( $c, $host );
				       showEdge( $c, $parent, $host );
				       $chW->grabRelease;
				       $chW->destroy();
				     } #- NOTE: this is identical to binding for <Return> --
				    );
  $ok_btn->pack(-side=>'right');


  #- make the host name entry the active widget -----
  $host_entry->focus if (defined $parent);

  $chW->Popup();          #- make this the top window
  $chW->grab;             #- restrict all activity to this window

}


#==============================================================================
# add a new host to the hierarchy and display it on the canvas
# 
#
#==============================================================================
sub create_host_obj {
  my ($parent, $host, $port, $x, $y) = @_;
  
  #- let ohs know about new host -----
  my ($ok,$err) = $hierarchy->add_host($parent,$host,$port);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error adding host ($host): $err\n");
    return 0;
  }
  else {
    
    $h_node{$host}->{'label'} = $host; 
    
    #- set the host icon position on the canvas -----
    $h_node{$host}->{'x'}    = $x;
    $h_node{$host}->{'y'}    = $y;
    
    $h_node{$host}->{'type'} = 'host';
    push(@{$h_node{$parent}->{'links'}},$host);
    1;
  }
}			       




#==============================================================================
# remove the specified host from the hierarchy
# 
#
#==============================================================================
sub delete_host {
  my ($can,$name) = @_;

  delete_host_obj($name);
  draw_hierarchy($can);
}



#==============================================================================
#
#
#
#==============================================================================
sub delete_host_obj {
  my $name = shift;
 
  my $par = $hierarchy->{'hierarchy'}{$name}{'parent'}; 

  my ($ok,$err) = $hierarchy->remove_host($name);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error deleting host ($name): $err\n");
    return 0;
  }
  else {
    @{$h_node{$par}->{'links'}} = grep ( $_ ne $name, @{$h_node{$par}->{'links'}});
  print "host name: $name\n";
  print "before undef: "; foreach ( keys %h_node ) { print "[$_] "; } print "\n";
    delete $h_node{$name};
  print "after undef:  "; foreach ( keys %h_node ) { print "[$_] "; } print "\n";

    return 1;
  }
} #delete_host_obj





#==============================================================================
#
#
#
#==============================================================================
sub do_move_host {
  my ($tag,$name) = @_;

  # must update local data structure -----
  
  # - remove item from old parents list
  my $par = $hierarchy->{'hierarchy'}->{$name}->{'parent'}; 


  # - notify ohs of the change 
  my ($ok,$err) = $hierarchy->move_host($tag,$name);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error moving host ($name): $err\n");
    return 0;
  }
  else {
    #- updating local hierarchy info -----
    @{$h_node{$par}->{'links'}} = grep ( $_ ne $name, @{$h_node{$par}->{'links'}});
    
    # - add item to new parents list
    push(@{$h_node{$tag}->{'links'}},$name);
    1;
  }
} #- do_move_host





#==============================================================================
#
#
#
#==============================================================================
sub create_dept {
  my ($c, $cx, $cy, $parent) = @_;

  my ($dept, $sm_host, $agg_host);
  
  my $chW  = $MW->Toplevel();
  $chW->withdraw;
						
  #- determine position to place new dept icon -----
  my ($dx,$dy) = ($cx, $cy);                        #- put it under the cursor
  ($dx,$dy) = map ($_+$ui_prefs::NEW_OBJ_DISP, ($cx,$cy)) if defined $parent;
                                                    #- unless we clicked on a dept,
                                                    #-   the we shift it 


  #- position the dialog at a reasonable position -- in pixels ---
  my ($xroot, $yroot) = map ( int(($_+1)*$ppi), ($cx, $cy) );
  $chW->geometry('+'.$xroot.'+'.$yroot);


  $chW->title('Add New Dept');
  $chW->title('Add New Dept to \''.$parent.'\'') if (defined $parent);

  my $button_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $prompt_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame  = $chW->Frame(-relief=>'flat')->pack(-side=>'left');

  unless (defined $parent) {
    my $parent_label = $prompt_frame->Label(-text=>'Parent Dept: ', -font=>$MNfont);
    $parent_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
    my $parent_entry = $entry_frame->Entry(-width=>32, 
					   -textvariable=>\$parent, 
					   -font=>$MNfont );
    $parent_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
    $parent_entry->focus;
  };

  my $dept_label = $prompt_frame->Label(-text=>'Dept Name: ', -font=>$MNfont);
  $dept_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $dept_entry = $entry_frame->Entry(-width=>32, 
					-textvariable=>\$dept, 
					-font=>$MNfont );
  $dept_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  

  my $sm_host_label = $prompt_frame->Label(-text=>'Software Manager Host: ', -font=>$MNfont);
  $sm_host_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $sm_host_entry = $entry_frame->Entry(-width=>8, 
					-textvariable=>\$sm_host, 
					-font=>$MNfont );
  $sm_host_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  

  my $agg_host_label = $prompt_frame->Label(-text=>'Aggregator Host: ', -font=>$MNfont);
  $agg_host_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $agg_host_entry = $entry_frame->Entry(-width=>8, 
					-textvariable=>\$agg_host, 
					-font=>$MNfont );
  $agg_host_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { $chW->destroy(); } );
  $cancel_btn->pack(-side=>'right');
  

  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub {
				       if ( create_dept_obj ($parent, $dept, $sm_host, $agg_host, $dx, $dy) ) {
print ">>>> back from greate dept_obj\n";
					 showNode( $c, $dept );
					 showEdge( $c, $parent, $dept );
				       };
				       $chW->grabRelease;
				       $chW->destroy();
				     } #- NOTE: this is identical to binding for <Return> --
				    );

  $chW->bind('<Return>' => sub { $ok_btn->invoke(); });

  $ok_btn->pack(-side=>'right');

  $dept_entry->focus if (defined $parent);

  $chW->Popup();          #- make this the top window
  $chW->grab;             #- restrict all activity to this window

} # create_dept


#==============================================================================
#
#
#
#==============================================================================
sub create_dept_obj {
  my ($parent, $dept, $sm_host, $agg_host, $x, $y) = @_;

  #-
  if ( $sm_host eq '' or $agg_host eq '' ) {
    Dialog_OK('Illegal Parameters','Host names for software manager and aggregator must be specified');
    return 0;
  }
  
  #- append $IMPLICIT_SUFFIX to names that are not fully specified
  #- ????? when is too much automation too much ???


  #- let ohs know about new host -----
  my ($ok,$err) = $hierarchy->add_dept($parent,$dept,$sm_host,$agg_host);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error creating dept: $err\n");
    return 0;
  }
  else {
    $h_node{$dept}->{'label'} = $dept; 
    
    #- set the host icon position on the canvas -----
    if (defined $x && defined $y ) {
      $h_node{$dept}->{'x'}    = $x;
      $h_node{$dept}->{'y'}    = $y;
    } else {
      $h_node{$dept}->{'x'}    = $h_node{$parent}->{'x'}+1;
      $h_node{$dept}->{'y'}    = $h_node{$parent}->{'y'}+1;
    };
    $h_node{$dept}->{'type'} = 'dept';
    push(@{$h_node{$parent}->{'links'}}, $dept );
    
    return 1;
  }
} # create_dept_obj      




		       
#==============================================================================
#
#
#
#==============================================================================
sub delete_dept {
  my ($can,$name) = @_;
  
  #- make-sure the dept has no children ---
  if (defined @{$h_node{$name}->{'links'}}){
print "links are: [",@{$h_node{$name}->{'links'}},"]\n";

    $MW->bell;
  }
  else {
    delete_dept_obj($name);
  };
  draw_hierarchy($can);
} #- delete_dept





#==============================================================================
#
#
#
#==============================================================================
sub delete_dept_obj {
  my $name = shift;
  
  my $par = $hierarchy->{'hierarchy'}{$name}{'parent'}; 
  print "\n-----\ndeleting department: $name, $par\n";

  my ($ok,$err) = $hierarchy->remove_dept($name);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error deleting dept:($name) $err\n");
    return 0;
  }

print "links before: [",@{$h_node{$par}->{'links'}},"]\n";

  @{$h_node{$par}->{'links'}} = grep ( $_ ne $name, @{$h_node{$par}->{'links'}});

print "links after:  [",@{$h_node{$par}->{'links'}},"]\n";

  delete $h_node{$name};

print "-----\n";  

  1;

} #- delete_dept_obj





#==============================================================================
#
#
#
#==============================================================================
sub do_move_dept {
  my ($tag,$name) = @_;

  # must update local data structure -----
  
  # - remove item from old parents list
  my $par = $hierarchy->{'hierarchy'}{$name}{'parent'}; 
  if ($par eq 'null') {
    Dialog_OK('Illegal Actio','The root may not be moved to another dept');
    return 0;
  }
  
  # - notify ohs of the change 
  my ($ok,$err) = $hierarchy->move_dept($tag,$name);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error moving dept: $err\n");
    return 0;
  }
  else {
print "links before: [",@{$h_node{$par}->{'links'}},"]\n";

    @{$h_node{$par}->{'links'}} = grep ( $_ ne $name, @{$h_node{$par}->{'links'}});
  
print "links after:  [",@{$h_node{$par}->{'links'}},"]\n";    
    # - add item to new parents list
    push(@{$h_node{$tag}->{'links'}},$name);
print "links last:   [",@{$h_node{$par}->{'links'}},"]\n";    
    return 1;
  }  
} #- do_move_dept





#==============================================================================
#
#
#
#==============================================================================
sub create_root_dept {
  my $c = shift;

  my ($root,$sm_host,$sm_port,$agg_host,$agg_port);
  
  my $chW  = $MW->Toplevel();
  $chW->withdraw;
						

  $chW->title('Add Root Dept');

  my $button_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $prompt_frame = $chW->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame  = $chW->Frame(-relief=>'flat')->pack(-side=>'left');

  my $root_label = $prompt_frame->Label(-text=>'Root Dept Name: ', -font=>$MNfont);
  $root_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $root_entry = $entry_frame->Entry(-width=>32, 
				       -textvariable=>\$root, 
				       -font=>$MNfont );
  $root_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  
  my $sm_host_label = $prompt_frame->Label(-text=>'Software Manager Host: ', -font=>$MNfont);
  $sm_host_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $sm_host_entry = $entry_frame->Entry(-width=>8, 
					  -textvariable=>\$sm_host, 
					  -font=>$MNfont );
  $sm_host_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  
  my $sm_port_label = $prompt_frame->Label(-text=>'Software Manager Port: ', -font=>$MNfont);
  $sm_port_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $sm_port_entry = $entry_frame->Entry(-width=>8, 
					  -textvariable=>\$sm_port, 
					  -font=>$MNfont );
  $sm_port_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $agg_host_label = $prompt_frame->Label(-text=>'Aggregator Host: ', -font=>$MNfont);
  $agg_host_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $agg_host_entry = $entry_frame->Entry(-width=>8, 
					   -textvariable=>\$agg_host, 
					   -font=>$MNfont );
  $agg_host_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $agg_port_label = $prompt_frame->Label(-text=>'Aggregator Port: ', -font=>$MNfont);
  $agg_port_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  my $agg_port_entry = $entry_frame->Entry(-width=>8, 
					   -textvariable=>\$agg_port, 
					   -font=>$MNfont );
  $agg_port_entry->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { exit 1; } );
  $cancel_btn->pack(-side=>'right');
  
  
  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub {
				       if (all_defined($root,$sm_host,$sm_port,$agg_host,$agg_port)) {
					 if ( create_root_obj($root,$sm_host,$sm_port,$agg_host,$agg_port) ) {
					   showNode($c, $root);
					 };
					 $chW->grabRelease;
					 $chW->destroy();
				       }
				       else {
					 $MW->bell;
				       }
				     } 
				    );
  $ok_btn->pack(-side=>'right');
  
  $chW->bind('<Return>' => sub { $ok_btn->invoke() });
  
  $root_entry->focus;
  
# $chW->Popup();          #- make this the top window
  $chW->grab;             #- restrict all activity to this window

  $chW->geometry('+300+100');
  $chW->deiconify;
  
  MainLoop;

} #- create_root_dept


#==============================================================================
#
#
#
#==============================================================================
sub create_root_obj {
  my ($root, $sm_host, $sm_port, $agg_host, $agg_port) = @_;
  
  $hierarchy = new Hierarchy_interface($username,$password,'null',$log,'file','main::FILE');
  
  #- let ohs know about new host -----
  my ($ok,$err) = $hierarchy->new_root($root,$sm_host,$sm_port,$agg_host,$agg_port);
  unless ($ok) {
    Dialog_OK('OHS Error',$err);
    $log->warn("Error creating root dept: $err\n");
    return 0;
  }
  else {
    
    #- set the host icon position on the canvas -----
    $h_node{$root}->{'x'}    = 1;
    $h_node{$root}->{'y'}    = 1;
    $h_node{$root}->{'type'} = 'dept';
    $h_node{$root}->{'links'} = ();
    
    1;
  }
}			       





#==============================================================================
#- open dialog to get name of object to find
#- determine if it exists
#- shift hilight the object
#- scroll the display to position the object on the screen
#==============================================================================
sub find_hierarchy_item {
  my ($c, $cx, $cy) = @_;

  my $name;

  my $findW  = $MW->Toplevel();
  $findW->withdraw;
  
  #- position the dialog at a reasonable position -- must be in pixels  -----
  my ($xroot, $yroot) = map ( int(($_+1)*$ppi), ($cx, $cy) );
  $findW->geometry('+'.$xroot.'+'.$yroot);

  my $search_in = 0;

  $findW->title('Find Hierarcy Object');
    
  my $button_frame   = $findW->Frame(-relief=>'flat')->pack(-side=>'bottom');
  my $findin_frame   = $findW->Frame(-relief=>'flat')->pack(-side=>'left');
  my $checkbox_frame = $findW->Frame(-relief=>'flat')->pack(-side=>'top');
  my $method_frame   = $findW->Frame(-relief=>'flat')->pack(-side=>'left');
  my $entry_frame    = $findW->Frame(-relief=>'flat')->pack(-side=>'left');

  my $host_radio = $checkbox_frame->Radiobutton();
  $host_radio->configure(-text     => 'Host', 
			    -variable => \$search_in,
			    -value    => 1,
			    -justify  => 'left',
			    -bg       => 'lightgray',
			   );
  $host_radio->pack( -fill=>'none', -expand=>0, -anchor => 'w', -side=>'left');

  my $dept_radio = $checkbox_frame->Radiobutton();
  $dept_radio->configure(-text     => 'Dept', 
			     -variable => \$search_in,
			     -value    => 1,
			     -justify  => 'left',
			     -bg       => 'lightgray',
			    );
  $dept_radio->pack( -fill=>'none', -expand=>0, -anchor => 'w', -side=>'left' );
  
  my $any_radio = $checkbox_frame->Radiobutton();
  $any_radio->configure(-text     => 'Any', 
			     -variable => \$search_in,
			     -value    => 0,
			     -justify  => 'left',
			     -bg       => 'lightgray',
			    );
  $any_radio->pack( -fill=>'none', -expand=>0, -anchor => 'w', -side=>'left' );
  $any_radio->invoke;
  
 
  my $search_in_label = $findin_frame->Label(-text=>'Search: ', -font=>$MNfont);
  $search_in_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );

#  my $target_label = $_frame->Label(-text=>'Host Port: ', -font=>$MNfont);
#  $target_label->pack( -side=>'top', -fill=>'x', -expand=>'true' );
  
  my $target_entry = $entry_frame->Entry(-width=>32, 
				       -textvariable=>\$name, 
				       -font=>$MNfont );
 
  my $cancel_btn = $button_frame->Button( -text=>'Cancel', -relief=>'raised', -command=>sub { $findW->destroy(); } );
  $cancel_btn->pack(-side=>'right');
  
  $findW->bind('<Return>' => sub { shiftHilight($c,$name);
				 $findW->destroy();
			       } #- NOTE: this is identical to binding for 'Ok Button' --
	    ); 
  my $ok_btn = $button_frame->Button(-text    => 'Ok',
				     -bd      => '2',
				     -relief  => 'raised',
				     -command => sub { shiftHilight($c,$name);
						       $findW->destroy();
						     } #- NOTE: this is identical to binding for <Return> --
				    );
  $ok_btn->pack(-side=>'right');


  #- make the host name entry the active widget -----
  $target_entry->focus;

  $findW->Popup();          #- make this the top window
  $findW->grab;             #- restrict all activity to this window

} #- find_hierarhy





#==============================================================================
#
#
#
#==============================================================================
sub print_hierarchy {
  print "user:        $hierarchy->{'user'}\n";
  print "password:    $hierarchy->{'password'}\n";
  print "dept:        $hierarchy->{'dept'}\n";
  print "log:         $hierarchy->{'log'}\n";
  print "output_mode: $hierarchy->{'output_mode'}\n";
  print "handle:      $hierarchy->{'handle'}\n";
  print "view_serial: $hierarchy->{'view_serial'}\n"; 
  print "hierarchy:   $hierarchy->{'hierarchy'}\n";
  print "ohs_location:$hierarchy->{'ohs_location'}\n";
  print "my_location: $hierarchy->{'my_location'}\n",
  print "ohs_open:    $hierarchy->{'ohs_open'}\n";
  print "\n";

  print "-----\n";
  my ($n,$i);
  foreach $n (keys(%{$hierarchy->{'hierarchy'}})) {
    print "$n: $hierarchy->{'hierarchy'}->{$n}\n";
    print "*** is a host!!\n" if  defined $hierarchy->{'hierarchy'}->{$n}->{'Host'};
    print "*** is a dept!!\n" if  defined $hierarchy->{'hierarchy'}->{$n}->{'Department'};
    foreach $i (keys(%{$hierarchy->{'hierarchy'}->{$n}})) {
      print "    $i: ";
      if ($i eq 'dept_children' || $i eq 'host_children') {
	if ( defined $hierarchy->{'hierarchy'}->{$n}->{$i} ) {
	  print "\n";
	};
	foreach (@{$hierarchy->{'hierarchy'}->{$n}->{$i}}) {
	  print "        $n".'->'.$_.";\n";
	}
      }
      else {
	print "$hierarchy->{'hierarchy'}->{$n}->{$i}";
      };
      print "\n";
    };
  };
  print "-----\n";
}




#==============================================================================
# calculate the positions of all hierchy elements using dot (for neatness)
#   - only the node placements are important; all nodes connected with
#       straight lines in practice
#
#==============================================================================
sub hierarchy2dot {
  #- no args passed --

  my ($n,$i);
  my $hg;

######################################################
# Added by J.Rowe to make a prettier hierarchy picture
#   The hierarchy picture, with hosts/depts as nodes and parent/child
#   relationships as edges, is layed out using the "dot" graph package.
#   The ordering of nodes in the picture comes from their order in the
#   string input to dot (see &cleanup_hierarchy above).  Filling all
#   dept_children before host_children puts all the sub-departments to
#   the left of the parent. When this algorithm is applied recursively to
#   a multi-level hierarchy the nodes are skewed to one side.  To obtain
#   a more balanced view, the dot string is filled with depts on the outside
#   and all nodes in the middle.


#   Get the name of the root node to begin the hierarchy traversal
  my $dept = $hierarchy->{'hierarchy'}{'root_name'};

#   For the root node, obtain the list of all it's children
  my @curr_depts = ( $dept );

#   Fill the dot string while there are child departments left to deal with.
  while (scalar(@curr_depts)) {

#       Keep track of all depts in the next level.
      my @next_depts;              

#       Get lists of dept and host child nodes
      foreach $dept (@curr_depts) {
          my (@j_hosts,@j_depts);
	  foreach (@{$hierarchy->{'hierarchy'}{$dept}{'host_children'}}) {
	    push(@j_hosts,($_));
	  };
	  foreach (@{$hierarchy->{'hierarchy'}{$dept}{'dept_children'}}) {
	    push(@j_depts,($_));

#  Add to the list of ALL child departments for the next iteration
            push(@next_depts,($_));
	  };

# Now that the list of all depts and hosts are formed, fill the dot string
# with depts distributed over the sides and hosts in the center
          my $num_hosts = scalar(@j_hosts);
          my $num_depts = scalar(@j_depts);

          if ($num_depts || $num_hosts) {
#  Add on the first half of the child departments
              for ($i=0; $i<int($num_depts/2); $i++) {
	          $hg .= "   \"$dept\"->\"$j_depts[$i]\";\n";
	          $hg .= "   \"$j_depts[$i]\" [height=2.5,width=2.5];\n";
              }
#  Add on all the child hosts
              foreach(@j_hosts) {
	          $hg .= "   \"$dept\"->\"$_\";\n";
	          $hg .= "   \"$_\" [shape=box,height=2,width=2];\n";
              }
#  Add on the last half of the child departments
              for ($i=int($num_depts/2); $i<$num_depts; $i++) {
	          $hg .= "   \"$dept\"->\"$j_depts[$i]\";\n";
	          $hg .= "   \"$j_depts[$i]\" [height=2.5,width=2.5];\n";
              }
          }
      }
#  Update the department list with the ones in the next level.
      @curr_depts = @next_depts;
  }

# End of J.Rowe's additions
#################################################
  

  foreach $n (keys(%{$hierarchy->{'hierarchy'}})) {
    
  #- add a reference to this item to the ui hierarchy data_struct -----   

    if ( $n ne 'root_name') { #- exclude the root_name entry -----
      $h_node{$n}->{'label'} = $n; 
      $h_node{$n}->{'tag'}   = ''; 
      $h_node{$n}->{'type'}  = ''; 
      $h_node{$n}->{'x'}     = 0; 
      $h_node{$n}->{'y'}     = 0;
      $h_node{$n}->{'links'} = ();

      foreach $i (keys(%{$hierarchy->{'hierarchy'}->{$n}})) {
      
	if ($i eq 'dept_children' ) {
	  foreach (@{$hierarchy->{'hierarchy'}->{$n}->{$i}}) {

#  Removed by J.Rowe (see above)
#	    $hg .= "    \"$n\"->\"$_\";\n";
#	    $hg .= "    \"$_\" [height=2.5,width=2.5];\n";

	    #- indicate a link between the two -----
	    push(@{$h_node{$n}->{'links'}},($_));
	  };
	};
	
	if ($i eq 'host_children') {
	  foreach (@{$hierarchy->{'hierarchy'}->{$n}->{$i}}) {

#  Removed by J.Rowe (see above)
#	    $hg .= "    \"$n\"->\"$_\";\n";
#	    $hg .= "    \"$_\" [shape=box,height=2,width=2];\n";

	    #- indicate a link between the two -----
	    push(@{$h_node{$n}->{'links'}},($_));
	  };
	};
      };
    };
  };
  #- if some graph is present, give it a container... -----
  $hg = "digraph hg {\n".$hg."}\n" if defined $hg;
  
} # hierarchy2dot





#==============================================================================
# display a list of rulesets running on specified department.
#
# allows editing, read/save from file, disabling, and submitting 
#   changed or modified rulesets
#==============================================================================
sub list_rulesets {
  my $dept = shift;

  my $rs_name;
  my $rs_info;
  my $rslt;

  my $rsW = $MW->Toplevel(-title  => 'Rulesets for '.$dept,
			 -relief => 'flat',
			 -bg     => 'lightgray' );
  $rsW->geometry('+50+50');

  my $button_frame   = $rsW->Frame(-relief=>'flat');
  $button_frame->pack( -side => 'right', -fill => 'both', -expand => 'yes' );

  $rsW->Label( -font=>$SBfont, 
		      -bg=>'lightgray', 
		      -width=>32, 
		      -anchor=>'sw', 
		      -text=>'Rulesets' )->pack(-side=>'top', -fill=> 'x', -expand => 'yes' );

  my $info_box = $rsW->Message(-font=>$SNfont, 
			       -bg=>'white',
			       -relief=>'ridge',
			       -justify=>'left',
			       -anchor=>'w', 
			       -width=>300,
			       -textvariable=> \$rs_info);
  $info_box->pack(-side=>'bottom', -fill=> 'x', -expand => 'yes' );

  my $list_frame = $rsW->Frame( -relief => 'sunken', -borderwidth => '2' );
  $list_frame->pack( -side => 'left', -fill => 'both', -expand => 'yes' );

  $rs_listbox = $list_frame->Listbox(-relief => 'flat',
				-width => 32,
				-height => 10,
				-setgrid => 'yes'
			       );
  $rs_listbox->pack(-side => 'left', -fill=> 'both', -expand => 'yes' );

  $rs_listbox->bind('<Button-1>' => sub { my $i = $rs_listbox->curselection;
				  $rs_name = $rs_listbox->get($i);
				  $rs_info = format_info($dept,$rs_name);
				 }
	   );

  
  my $sbr = $list_frame->Scrollbar(-command => ['yview', $rs_listbox] );
  $rs_listbox->configure( -yscrollcommand => ['set', $sbr] );
  $sbr->pack(-side => 'right', -fill => 'y' );

  my $view_btn = $button_frame->Button(-text=>'View Ruleset', 
					 -relief=>'raised', 
					 -command=> sub { edit_ruleset($dept,$rs_name); } 
					);
  $view_btn->pack(-side=>'top', -fill=> 'x', -expand => 'yes', -padx => '2m' );

  my $new_btn = $button_frame->Button(-text=>'New Ruleset', 
					 -relief=>'raised', 
					 -command=> sub { edit_ruleset($dept,''); }
					);
  $new_btn->pack(-side=>'top', -fill=> 'x', -expand => 'yes', -padx => '2m' );

  my $remove_btn = $button_frame->Button(-text=>'Remove Ruleset', 
					 -relief=>'raised', 
					 -command=> sub { remove_ruleset($dept,$rs_name); } 
					);
  $remove_btn->pack(-side=>'top', -fill=> 'x', -expand => 'yes', -padx => '2m', -pady => '4m');

  my $cancel_btn = $button_frame->Button(-text=>'Cancel', 
					 -relief=>'raised', 
					 -command=>sub { $rsW->destroy; }
					);
  $cancel_btn->pack(-side=>'top', -fill=> 'x', -expand => 'yes', -padx => '2m' );


  #- now fill the listbox with the active rulesets -----
  if ( $rslt= get_rulesets($dept) ) {
    %rulesets = %$rslt;
    foreach (sort keys %rulesets) {
      $rs_listbox->insert('end', $_);
    };
    $rs_listbox->see(0); #- makesure the top of the list is visible
    $rs_listbox->activate(0);
    $rs_name = $rs_listbox->get( 'active' );
    my $active = $rs_listbox->index('active');
    $rs_info = format_info($dept,$rs_name);
  };
} #- list_rulesets





#==============================================================================
#
#
#
#==============================================================================
sub format_info {
  my ($dept,$name) = @_;
  my $info = 'info for '.$name."\n";
  $info .= '  owner:   '.$dept."\n";
  $info .= '  started: '.localtime(time)."\n";
  $info .= '  status:   active'."\n";
  
  return $info;
}; 





#==============================================================================
# returns the list of rulesets running on the specified department 
#
# == for now just returns a hash of random rulesets ==
#==============================================================================
sub get_rulesets {
  my ($dept) = @_;   

  my %rsh;
  my ($request) = ['rulesets{ALL_STATE_NAMES}','_'];
  
  my ($ok,$rslt) = $hierarchy->get($dept,'','',@$request);
  unless ($ok) {
    Dialog_OK('Error: getting rulesets from $dept',$rslt);
    $log->warn("!! error[",$ok,"]:[",$rslt,"] getting rulesets for $dept [",join(', ',@$request),"]\n");
    return 0;
  };

  my @rs = sort map {if(/^rulesets\{(.*)\}$/) {$1} else {()} } @$rslt;
  
  foreach (0..$#rs) {
    %rsh = add_ruleset( $rs[$_], %rsh );
  };
  
  return \%rsh;
  
};


#==============================================================================
# returns text for specified ruleset
#==============================================================================
sub get_ruletext {
  my ($dept,$rsn) = @_;   

  my $request = ['rulesets{'.$rsn.'}','1'];
  my ($ok,$rslt) = $hierarchy->get($dept,'','',@$request);

  if (!$ok) {
    $log->warn("ERROR: getting ruleset text for $rsn at $dept [".join(', ',@$request)."]\nRESULT = $rslt\n");
    return ($ok,$rslt);
  };

  my @rs = @{$rslt}; shift @rs;
  return ($ok,@rs);
}


#==============================================================================
#
#
#
#==============================================================================
sub add_ruleset {
  my ($name,%rsl) = @_;

  unless ( defined $rsl{$name} ) {
    $rsl{$name}{'name'}   = $name;
    $rsl{$name}{'status'} = 0;
    $rsl{$name}{'xpos'}   = 0;
    $rsl{$name}{'ypos'}   = 0;
  }
  return %rsl;
} #- add_ruleset {}





#==============================================================================
#
#
#
#==============================================================================
sub remove_ruleset {
  my ($dept,$name) = @_;
  
#- send request to remove the ruleset to the engine -----
  my $request = ['rulesets{'.$name.'}',''];
  my ($ok,$rslt) = $hierarchy->set($dept,'','',@$request);
  if (!$ok) {
    $log->warn("ERROR: removing ruleset $name for $dept\nRESULT = $rslt\n");
    Dialog_OK('Ruleset Error',"Removing ruleset $name had errors:\n@$rslt[0].");
    return 0;
  };
  
#-----
#- remove ruleset from listbox list of rulesets 
#- This requires finding the named item in the list
#- This method of direct explicit search seems dumb, but...
#-----
  if (defined  $rs_listbox) {
    my $num_rs = $rs_listbox->index('end');
    #  my $num_rs = $rs_listbox->size;
    my $index = 0;
    my $item_name;
    foreach $index (0..$num_rs-1) {
      $item_name = $rs_listbox->get($index);
      if ($item_name eq $name) {
	$rs_listbox->delete($index); #- remove the item
	last;
      };
    };
  };
  
#- remove ruleset from hash of ruleset on the dept -----
  delete $rulesets{$name} if ( defined $rulesets{$name});

#- remove ruleset from window of ruleset icons -----
  if (defined $ruleset_canvas) {
    my $item = $ruleset_canvas->find( 'withtag', '$name' );
    $ruleset_canvas->delete($item);
    cleanup_rulesets($ruleset_canvas);    
    show_rulesets( $ruleset_canvas, %rulesets );
  };

} #- remove_ruleset





sub submit_ruleset {
#==============================================================================
#
#
#
#==============================================================================
  my ( $dept, $ruletext ) = @_;

  # added by JAH 2/3/97 to ensure that comments and blank lines at the top of the ruleset are ignored in deriving the ruleset name.
  $_= $ruletext;
  s/^\s*(\#.*|)?\n//g;

  my ($rsn) = /^\s*ruleset\s+(\w+)\s*\;/;
print "SUBMIT - dept:$dept, name=$rsn\nRS:  substr($_,0,32)\n";
  
  unless (defined $dept and defined $ruletext and defined $rsn) {
    $log->warn("SUBMIT RULESET - error: undefined parameter\n");
    return;
  };
  
  my @rslt = check_ruleset($ruletext);
  if (@rslt) {
    Dialog_OK('Ruleset Error', "Ruleset errors were found:\n".join("\n",@rslt) );
    return;
  };
  
  
  $log->warn("submitting ruleset to $dept: name=$rsn\n");

  my $request = ['rulesets{'.$rsn.'}',$ruletext];
  my ($ok,$rslt) = $hierarchy->set($dept,'','',@$request);
  if (!$ok) {
    $log->warn("ERROR: setting rulesets for $dept\nRESULT = $rslt\n");
    Dialog_OK('Ruleset Error',"The ruleset submitted was rejected due to error in the ruleset.");
    return;
  };

#-----
#- if the ruleset name is not already present, add the ruleset name to internal list of rulesets and
#-   make sure all open windows displaying rulesets or ruleset names are updated to show the new name
#-----
  unless (defined $rulesets{$rsn}) {
    %rulesets = add_ruleset($rsn,%rulesets);
    update_ruleset_windows($rsn,%rulesets);
  };

} #- submit_ruleset





#==============================================================================
#
#
#
#==============================================================================
sub update_ruleset_windows {
  my ($rsn,%rulesets) = @_;

  #- see if we need to update the ruleset listbox -----
  if (defined  $rs_listbox && defined $rulesets{$rsn}) {
    $rs_listbox->insert('end', $rsn);
  };
  
  #- see if we need to update the ruleset icon/alert window -----
  if (defined $ruleset_canvas ) {
    cleanup_rulesets($ruleset_canvas);    
    show_rulesets( $ruleset_canvas, %rulesets );
  };
}



#==============================================================================
#
#
#
#==============================================================================
sub all_defined {
  foreach (@_) {
    return 'false' unless defined $_;
  };
  return 'true';
}




#==============================================================================
#- get uptime for a component from the grids system -----
#==============================================================================
sub get_uptime {
  my ($type, $name) = @_;
 
### dummy code, must get real uptime from sm #####
  return localtime(time);
};



#==============================================================================
#- get creatoin info  for a component from the grids system -----
#==============================================================================
sub get_create_info {
  my ($type, $name) = @_;
  my ($unam) = getpwuid($<);

### dummy code, must get real stuff from sm #####
  return $unam.' on '.localtime(time);
};



#==============================================================================
#- get status info for a component from the grids system -----
#==============================================================================
sub get_status {
  my ($type, $name) = @_;

### dummy code, must get real stuff from sm #####
  return "Up and ok";
};



#==============================================================================
#- get list of aggregators/ports running on the specified host -----
#==============================================================================
sub get_agg_info {
    my ($host) = @_;

    my $dept = $hi->{$host}->{'parent'};
    
#- get the list of aggregators on host -----
    my ($ok, $rslt) = $hierarchy->get($host,'module_controller',$dept,
				      'mc_command','INVENTORY',
				      'mod_module','engine',
				     );
    unless ($ok) {
      Dialog_OK('Error: getting list engines on $host',join(', ',@$rslt) );
      $log->warn("\nERROR: getting list of engines on $host, rslt=(",join(', ',@$rslt),")\n");
      return 0;
    };
    
    my $entry;
    my ($engine_name,$engine_dept,$pid,$version);
    my @return;
    while (@$rslt) {
      shift @$rslt;
      $entry = shift @$rslt;
      ($engine_name,$engine_dept,$pid,$version) = split(' ',$entry);
      @return = (@return,[$engine_dept,$pid]);
print "\n----------\n($engine_name,  $engine_dept,  $pid,  $version)\n---------\n";      
    };

print "Returning - ",join(" :: ", @return),"\n\n";
    return @return
    
    #    return ( ['CSIF', 2337],['SECLAB', 2342] );
};





#==============================================================================
# get list of all available datasources from the constraints file -----
#==============================================================================
sub get_available_datasources {
  my %ds;
  my ($name,$ver,$run,$init,$path);
  my $id;

  open (FID,"$ENV{'GRIDSPATH'}/module_controller/mod.control.constraints.tasks") || $log->warn( "unable to open constraints file\n");
  while (<FID>) {
    next if $_ =~ /^\s*\#/; # ignore lines starting with a '#', whitespace may precede it
    next if $_ =~ /^\s*\n/; # ignore lines having no text

    chop;
    ($name,$ver,$run, $init,$path) = split( /\s+/,$_ ); 
    $init = "''" unless defined $init;
    $path = "''" unless defined $path;

    unless ($run =~ /\/engine\// or $run =~ /\/sm\// ){ 
      $id = "$name:$ver";
      next if defined $ds{$id};       # don't redefine something, i.e. use the first occurance
      $ds{$id}->{'name'} = $name;
      $ds{$id}->{'ver'}  = $ver;
      $ds{$id}->{'init'} = $init;
      $ds{$id}->{'path'} = $path;
    };
  };
  close FID;

  return %ds;
};





#==============================================================================
#- get list of data sources running on the specified host and their statti 
#==============================================================================
sub update_datasource_info { 
    my ($host,%ds) = @_;
    my $dept = $hi->{$host}->{'parent'};
    
    my ($ok, $rslt) = $hierarchy->get($host,'module_controller',$dept,
				      'mc_command','INVENTORY',
				      'mod_module','*',
				      'mod_version','_',
				      'mod_department',$dept
				     );
    unless ($ok) {
      Dialog_OK('Error: getting datasources for $host on $dept',join(', ',@$rslt) );
      $log->warn("\nRESULT getting datasource info for $host on $dept, rslt=(",join(', ',@$rslt),")\n");
      return 0;
    };


#- set on/off values in datasource hash ---
  #- set all data sources to off -----
    my ($ds_name,$mod_dept,$pid,$version);
    
    foreach (keys %ds) {
      $ds{$_}->{'state'} = 0;
    };

  #- now turn on just the data sources that are active -----

    while (@$rslt) {
      shift @$rslt;
      my $entry = shift @$rslt;
      ($ds_name,$mod_dept,$pid,$version) = split(' ',$entry);
      $ds{"$ds_name:$version"}->{'state'} = 1 if defined  $ds{"$ds_name:$version"};
      
    };
    
    return %ds;
};





#==============================================================================
#- start a data source 
#==============================================================================
sub start_datasource {
# my ($dept,$ds_host,$ds_type,$ds_ver) = @_;
  change_datasource_activation(@_,'START');
};


#==============================================================================
#- stop a data source 
#==============================================================================
sub stop_datasource {
# my ($dept,$ds_host,$ds_type,$ds_ver) = @_;
  change_datasource_activation(@_,'STOP');
};


#==============================================================================
#- start or stop a data source 
#==============================================================================
sub change_datasource_activation {
  my ($dept,$ds_host,$ds_type,$ds_ver,$action) = @_;
  my $agg = "$hi->{$dept}->{'aggregator_host'}:$hi->{$dept}->{'aggregator_port'}";
 
  my ($ok,$rslt) = $hierarchy->set($ds_host,'module_controller',$dept,
				   'mc_command',$action,
				   'mod_module',$ds_type,
				   'mod_version',$ds_ver,
				   'mod_department',$dept,
				   'parent_aggregator',$agg
			  );
  unless ($ok) {
    Dialog_OK('Error: Changing datasource actication',join(', ',@$rslt) );
    $log->warn( "\nERROR: Change datasource activation -- $dept, $ds_host, $ds_type, $ds_ver, $action, $agg \nRESULT = $rslt");
    return 0;
  };  
}





#==============================================================================
#==============================================================================
sub send_shutup_to_engine {
#  my ($ui_host,$ui_port,$deptname) = @_;
  change_alert_recipients(@_,0);
}


#==============================================================================
#==============================================================================
sub send_hello_to_engine {
#  my ($ui_host,$ui_port,$deptname) = @_;
  change_alert_recipients(@_,1);
}


#==============================================================================
#==============================================================================
sub change_alert_recipients {
  my ($ui_host,$ui_port,$deptname,$mode) = @_;

  my $ui_loc = "$ui_host:$ui_port";
  my $engine_host = $hi->{$deptname}->{'aggregator_host'}; 

  my ($ok, $rslt) = $hierarchy->set($engine_host,'engine',$deptname,
				    "alert_recipients{$ui_loc}",$mode
				   );
   
  unless ($ok) {
    Dialog_OK('Error: Changing alert recipients',join(', ',@$rslt) );
    $log->warn("\nERROR - change alert recipients: at $ui_loc($mode) for engine on $engine_host\n**RESULT = $rslt\n");
    return 0;
  };  

}



#==============================================================================
#- Ask for the host/port the specified engine/dept is listening on
#==============================================================================
sub get_engine_ear {
  my $deptname = shift;
  my $ear;

# - this is the proposed, real control variable way -----  
# - No - it is better gotten out of our local hierarchy -- Stuart.
#   $ear = $hierarchy->get($deptname,'_','_','listen_tcp','_');

  my $engine_host = $hi->{$deptname}->{'aggregator_host'};
  my $engine_port = $hi->{$deptname}->{'aggregator_port'};
  $ear = "$engine_host:$engine_port";

  return $ear;
}




#==============================================================================
#==============================================================================
sub change_control_vars {
  my ($dept, $host, $mod_name, $mod_ver, $new_cvs, $old_cvs) = @_;

  my ($ok, $rslt);
  my @cmnd = (); 

  $new_cvs->{'command'} = 'set';  #- force this to a set command
  
  foreach (sort keys %{$new_cvs}) {
    @cmnd = ( @cmnd, ($_,$new_cvs->{$_}) ) if $new_cvs->{$_} ne $old_cvs->{$_};
    print "changing $_ to $new_cvs->{$_}\n" if $new_cvs->{$_} ne $old_cvs->{$_};
  };
  print "CMD=",join(', ',@cmnd),"\n";
  ($ok, $rslt) = $hierarchy->set( $host, $mod_name, $dept, @cmnd );
  unless ($ok) {
    Dialog_OK('Error: setting control variables',join(', ',@$rslt) );
    $log->warn("ERROR: setting control variables for module $mod_name($mod_ver) on $host($dept)\nCVARS = ".join(', ',@cmnd)."\nRESULT = $rslt\n");    
    return 0;
  };  


} #- change_control_vars





#==============================================================================
#==============================================================================
sub Dialog_OK {
  my ($title, $text) = @_;
  my $D = $MW->Dialog(-title=>$title,
		      -text=> $text,
		      -default_button=>'Ok',
		      -buttons => ['Ok']
		     );
  $D->Show; #- stays in here until Ok is pressed
  return;
};





#==============================================================================
#==============================================================================
sub max {
  my ($a,$b) = @_;
  return $a>$b ? $a : $b;
};

#==============================================================================
#==============================================================================
sub min {
  my ($a,$b) = @_;
  return $b>$a ? $a : $b;
};





#==============================================================================
#
#
#------------------------------------------------------------------------------
# !! printing graphs always prints to graph.ps; when plot is called the caller
#      must give the desire name.
#==============================================================================
sub plot
{
 my ($c,$gname) = @_;

 my $ps = $c->postscript('x'         => 0, 
			 'y'         => 0, 
			 -width      => $c->width, 
			 -height     => $c->height,
			 'rotate'    => $c->width > $c->height,
#			 -pageheight => '11i',
#			 -pagewidth  => '8.5',
			);
 open(PS,">$gname.ps") || die "Cannot open $gname.ps:$!";
 print PS $ps,"\n";
 close(PS);
 `lpr -h $gname.ps`;
}


sub unimplemented {
  $MW->bell;
  Dialog_OK('unimplemented function','This function has not yet been implemented');
}


=Keep
sometimes another program takes "global lock" on the X-display and causes our UI to block
what if the UI goes down for a time. When we bring it backup can we get alerts that occurred
during the downtime, or the previous state of the UI (logged alerts)
=cut

